/*
 * Copyright 2015 Dynatrace
 *
 * Licensed under the Dynatrace SaaS terms of service (the "License");
 * You may obtain a copy of the License at
 *
 *      https://ruxit.com/eula/saas/#terms-of-service
 */
package com.dynatrace.tools.android;

import java.io.File;
import java.lang.reflect.Method;

import javax.inject.Inject;

import org.gradle.api.Action;
import org.gradle.api.DefaultTask;
import org.gradle.api.GradleException;
import org.gradle.api.Plugin;
import org.gradle.api.Project;
import org.gradle.internal.reflect.Instantiator;
import org.jetbrains.annotations.NotNull;

import com.android.build.gradle.AppExtension;
import com.android.build.gradle.api.ApkVariantOutput;
import com.android.build.gradle.api.ApplicationVariant;
import com.android.build.gradle.api.BaseVariantOutput;
import com.android.build.gradle.tasks.PackageApplication;

/**
 * Dynatrace Android Plugin.
 */
public class DynatraceAndroidPlugin implements Plugin<Project> {
	private final Instantiator instantiator;

	@Inject
	public DynatraceAndroidPlugin(Instantiator instantiator) {
		this.instantiator = instantiator;
	}

	@Override
	public void apply(@NotNull final Project project) {
		DynatraceConfig config = instantiator.newInstance(DynatraceConfig.class, project);
		project.getExtensions().add("dynatrace", config);

		createAutoInstrumentTasks(project);
	}

	private void createAutoInstrumentTasks(final Project project) {
		final DefaultTask prepareApkit = project.getTasks().create("prepareApkit", DefaultTask.class);
		prepareApkit.doLast(task -> task.getLogger()
				.info("prepare {}", project.getExtensions().getByType(DynatraceConfig.class).apkitDir()));

		AppExtension appExtension = project.getExtensions().getByType(AppExtension.class);
		appExtension.getApplicationVariants().all(new Action<ApplicationVariant>() {
			@Override public void execute(@NotNull final ApplicationVariant applicationVariant) {
				applicationVariant.getPreBuild().dependsOn(prepareApkit);

				final String capitalizedFlavorName =
						applicationVariant.getName().substring(0, 1).toUpperCase() + applicationVariant.getName().substring(1);

				Iterable<BaseVariantOutput> collection;
				try {
					Method methodOutputs = ApplicationVariant.class.getMethod("getOutputs");
					methodOutputs.setAccessible(true);
					collection = (Iterable) methodOutputs.invoke(applicationVariant);
				} catch (Exception e) {
					throw new GradleException("Failed to read: variant outputs", e);
				}

				for (BaseVariantOutput baseVariantOutput : collection) {
					ApkVariantOutput apkVariantOutput = (ApkVariantOutput) baseVariantOutput;
					final AndroidPluginVersion androidPluginVersion = determineAndroidPluginVersion(apkVariantOutput);
					final PackageApplication taskPackage;
					try {
						Method methodPackage = ApkVariantOutput.class.getDeclaredMethod("getPackageApplication");
						taskPackage = (PackageApplication) methodPackage.invoke(apkVariantOutput);
					} catch (Exception e) {
						throw new GradleException("Failed to get PackageApplication task", e);
					}
					final DynatraceConfig config = project.getExtensions().getByType(DynatraceConfig.class);
					AutoInstrumentTask instrumentTask = project.getTasks()
							.create("autoInstrument" + capitalizedFlavorName, AutoInstrumentTask.class,
									new Action<AutoInstrumentTask>() {
										@Override public void execute(@NotNull AutoInstrumentTask autoInstrumentTask) {
											autoInstrumentTask.setSigningConfig(taskPackage.getSigningConfig());
											autoInstrumentTask.setAndroidPluginVersion(androidPluginVersion);
											autoInstrumentTask.setPackageTask(taskPackage);

											autoInstrumentTask.conventionMapping("outputFile", () ->
													new File(autoInstrumentTask.getTemporaryDir(),
															(androidPluginVersion
																	.isNewerOrEqualTo(AndroidPluginVersion.VERSION_3_0) ?
																	apkVariantOutput.getOutputFile() :
																	taskPackage.getOutputFile()).getName())
											);
											autoInstrumentTask.conventionMapping("apkFile", () ->
													androidPluginVersion.isNewerOrEqualTo(AndroidPluginVersion.VERSION_3_0) ?
															apkVariantOutput.getOutputFile() :
															taskPackage.getOutputFile()
											);

											autoInstrumentTask.conventionMapping("apkitDir",
													config::apkitDir);
											autoInstrumentTask.conventionMapping("applicationId",
													config.findProductFlavor(applicationVariant.getFlavorName())::getApplicationId);
											autoInstrumentTask.conventionMapping("environmentId",
													config.findProductFlavor(applicationVariant.getFlavorName())::getEnvironmentId);
											autoInstrumentTask.conventionMapping("cluster",
													config.findProductFlavor(applicationVariant.getFlavorName())::getCluster);
											autoInstrumentTask.conventionMapping("startupPath",
													config.findProductFlavor(applicationVariant.getFlavorName())::getStartupPath);
											autoInstrumentTask.conventionMapping("agentProperties",
													config.findProductFlavor(
															applicationVariant.getFlavorName())::getAgentProperties);
										}
									});

					if (androidPluginVersion == AndroidPluginVersion.VERSION_1_5) {
						try {
							Method methodZipAlign = ApkVariantOutput.class.getDeclaredMethod("getZipAlign");
							methodZipAlign.setAccessible(true);
							Object zipAlign = methodZipAlign.invoke(apkVariantOutput);
							Method methodInputFile = zipAlign.getClass().getDeclaredMethod("setInputFile", File.class);
							methodInputFile.setAccessible(true);
							methodInputFile.invoke(zipAlign, instrumentTask.getOutputFile());
						} catch (Exception e) {
							throw new GradleException("Cannot set input file for task ZipAlign", e);
						}
					}

					taskPackage.finalizedBy(instrumentTask);
				}
			}
		});
	}

	private AndroidPluginVersion determineAndroidPluginVersion(ApkVariantOutput apkVariantOutput) {
		try {
			Class.forName("com.android.build.gradle.internal.tasks.BaseTask");
		} catch (ClassNotFoundException e) {
			return AndroidPluginVersion.VERSION_3_1;
		}

		try {
			Class.forName("com.android.build.gradle.tasks.ZipAlign");
		} catch (ClassNotFoundException e) {
			return AndroidPluginVersion.VERSION_3_0;
		}

		try {
			Method methodZipAlign = ApkVariantOutput.class.getDeclaredMethod("getZipAlign");
			methodZipAlign.setAccessible(true);
			Object zipAlign = methodZipAlign.invoke(apkVariantOutput);
			if (zipAlign != null) {
				return AndroidPluginVersion.VERSION_1_5;
			}
		} catch (Exception e) {
			throw new GradleException("Cannot access task ZipAlign", e);
		}
		try {
			Class.forName("com.android.builder.packaging.NativeLibrariesPackagingMode");
			return AndroidPluginVersion.VERSION_2_2;
		} catch (ClassNotFoundException e) {
			return AndroidPluginVersion.VERSION_2_3;
		}
	}
}
