/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.cronutils.utils;

/**
 * <p>Operations on {@link java.lang.String} that are
 * {@code null} safe.</p>
 *
 * <ul>
 *  <li><b>IsEmpty</b>
 *      - checks if a String contains text</li>
 *  <li><b>IndexOfAny</b>
 *      - index-of any of a set of Strings</li>
 * </ul>
 *
 * <p>The {@code StringUtils} class defines certain words related to
 * String handling.</p>
 *
 * <ul>
 *  <li>null - {@code null}</li>
 *  <li>empty - a zero-length string ({@code ""})</li>
 *  <li>space - the space character ({@code ' '}, char 32)</li>
 *  <li>whitespace - the characters defined by {@link Character#isWhitespace(char)}</li>
 *  <li>trim - the characters &lt;= 32 as in {@link String#trim()}</li>
 * </ul>
 *
 * <p>{@code StringUtils} handles {@code null} input Strings quietly.
 * That is to say that a {@code null} input will return {@code null}.
 * Where a {@code boolean} or {@code int} is being returned
 * details vary by method.</p>
 *
 * <p>A side effect of the {@code null} handling is that a
 * {@code NullPointerException} should be considered a bug in
 * {@code StringUtils}.</p>
 *
 * <p>Methods in this class give sample code to explain their operation.
 * The symbol {@code *} is used to indicate any input including {@code null}.</p>
 *
 * <p>#ThreadSafe#</p>
 * @see java.lang.String
 * @since 1.0
 * @version $Id: StringUtils.java 1648067 2014-12-27 16:45:42Z britter $
 */
//@Immutable
public class StringUtils {
  private StringUtils() {}

  // Empty checks
  //-----------------------------------------------------------------------
  /**
   * <p>Checks if a CharSequence is empty ("") or null.</p>
   *
   * <pre>
   * StringUtils.isEmpty(null)      = true
   * StringUtils.isEmpty("")        = true
   * StringUtils.isEmpty(" ")       = false
   * StringUtils.isEmpty("bob")     = false
   * StringUtils.isEmpty("  bob  ") = false
   * </pre>
   *
   * <p>NOTE: This method changed in Lang version 2.0.
   * It no longer trims the CharSequence.
   * That functionality is available in isBlank().</p>
   *
   * @param cs  the CharSequence to check, may be null
   * @return {@code true} if the CharSequence is empty or null
   * @since 3.0 Changed signature from isEmpty(String) to isEmpty(CharSequence)
   */
  public static boolean isEmpty(final CharSequence cs) {
      return cs == null || cs.length() == 0;
  }

  // ContainsAny
  //-----------------------------------------------------------------------
  /**
   * <p>Checks if the CharSequence contains any character in the given
   * set of characters.</p>
   *
   * <p>A {@code null} CharSequence will return {@code false}.
   * A {@code null} or zero length search array will return {@code false}.</p>
   *
   * <pre>
   * StringUtils.containsAny(null, *)                = false
   * StringUtils.containsAny("", *)                  = false
   * StringUtils.containsAny(*, null)                = false
   * StringUtils.containsAny(*, [])                  = false
   * StringUtils.containsAny("zzabyycdxx",['z','a']) = true
   * StringUtils.containsAny("zzabyycdxx",['b','y']) = true
   * StringUtils.containsAny("aba", ['z'])           = false
   * </pre>
   *
   * @param cs  the CharSequence to check, may be null
   * @param searchChars  the chars to search for, may be null
   * @return the {@code true} if any of the chars are found,
   * {@code false} if no match or null input
   * @since 2.4
   * @since 3.0 Changed signature from containsAny(String, char[]) to containsAny(CharSequence, char...)
   */
  public static boolean containsAny(final CharSequence cs, final char... searchChars) {
      if (isEmpty(cs) || searchChars == null || searchChars.length == 0) {
          return false;
      }
      final int csLength = cs.length();
      final int searchLength = searchChars.length;
      final int csLast = csLength - 1;
      final int searchLast = searchLength - 1;
      for (int i = 0; i < csLength; i++) {
          final char ch = cs.charAt(i);
          for (int j = 0; j < searchLength; j++) {
              if (searchChars[j] == ch) {
                  if (Character.isHighSurrogate(ch)) {
                      if (j == searchLast) {
                          // missing low surrogate, fine, like String.indexOf(String)
                          return true;
                      }
                      if (i < csLast && searchChars[j + 1] == cs.charAt(i + 1)) {
                          return true;
                      }
                  } else {
                      // ch is in the Basic Multilingual Plane
                      return true;
                  }
              }
          }
      }
      return false;
  }
}
