/*
 * Author: Jude Pereira
 * Copyright (c) 2014
 */

package com.clevertap.android.sdk;

import android.content.Context;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * Provides various methods to send arbitrary data to CleverTap.
 */
public final class DataHandler {

    private final Context context;

    DataHandler(Context context) {
        this.context = context;
    }

    /**
     * For internal use, don't call the public API internally
     *
     */
    static void pushDeviceToken(final Context context, final String token, final boolean register, final PushType type) {
        if (token == null || type == null ) return;

        JSONObject event = new JSONObject();
        JSONObject data = new JSONObject();
        try {
            String action = register ? "register" : "unregister";
            data.put("action", action);
            data.put("id", token);
            data.put("type", type.toString());
            event.put("data", data);
            Logger.v("DataHandler: pushing device token with action " + action + " and type " + type.toString());
            QueueManager.queueEvent(context, event, Constants.DATA_EVENT);
        } catch (JSONException e) {
            // we won't get here
        }
    }

    /**
     * Sends the GCM registration ID to CleverTap.
     *
     * @param gcmId    The GCM registration ID
     * @param register Boolean indicating whether to register
     *                 or not for receiving push messages from CleverTap.
     *                 Set this to true to receive push messages from CleverTap,
     *                 and false to not receive any messages from CleverTap.
     */
    public void pushGcmRegistrationId(String gcmId, boolean register) {
        pushDeviceToken(context, gcmId, register, PushType.GCM);
    }

    /**
     * Sends the FCM registration ID to CleverTap.
     *
     * @param fcmId    The FCM registration ID
     * @param register Boolean indicating whether to register
     *                 or not for receiving push messages from CleverTap.
     *                 Set this to true to receive push messages from CleverTap,
     *                 and false to not receive any messages from CleverTap.
     */
    public void pushFcmRegistrationId(String fcmId, boolean register) {
        pushDeviceToken(context, fcmId, register, PushType.FCM);
    }
}
