/*
 * Author: Jude Pereira
 * Copyright (c) 2014
 */

package com.clevertap.android.sdk;

import android.content.Context;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.Iterator;

/**
 * Provides methods to manipulate the event queue.
 */
final class QueueManager {
    private static final Boolean lock = true;
    private static Runnable commsRunnable = null;

    // our sqlite db wrapper; lazy loaded so always call loadDBAdapter to use
    private static DBAdapter dbAdapter;

    private static DBAdapter loadDBAdapter(Context context) {
        if (dbAdapter == null) {
            dbAdapter = new DBAdapter(context);
            dbAdapter.cleanupStaleEvents(DBAdapter.Table.EVENTS);
            dbAdapter.cleanupStaleEvents(DBAdapter.Table.PROFILE_EVENTS);
        }
        return dbAdapter;
    }

    private static void lazyCreateSession(Context context) {
        if (SessionManager.getCurrentSession() == 0) {
            SessionManager.createSession(context);
        }
    }

    /**
     * the one and only entry point into the QueueManager
     * Please only use this method to process an event
     */
    static void queueEvent(final Context context, final JSONObject event, final int eventType) {
        lazyCreateSession(context);

        CleverTapAPI.postAsyncSafely("queueEvent", new Runnable() {
            @Override
            public void run() {
                addToQueue(context, event, eventType);
            }
        });
    }

    // db objects wrapper obj
    static final class QueueCursor {
        private JSONArray data; // the db objects
        private String lastId; // the id of the last object returned from the db, used to remove sent objects
        private DBAdapter.Table tableName;

        JSONArray getData() {
            return data;
        }

        Boolean isEmpty() {
            return (lastId == null || data == null || data.length() <= 0);
        }

        private void resetForTableName(DBAdapter.Table tName) {
            tableName = tName;
            data = null;
            lastId = null;
        }

        @Override
        public String toString() {
            return (this.isEmpty()) ? "tableName: " + tableName + " | numItems: 0" :
                    "tableName: " + tableName + " | lastId: " + lastId + " | numItems: " + data.length() + " | items: " + data.toString();
        }
    }

    // external entry point to accessing the queue
    static QueueCursor getQueuedEvents(final Context context, final int batchSize, final QueueCursor previousCursor) {
        return getQueuedDBEvents(context, batchSize, previousCursor);
    }

    private static QueueCursor getQueuedDBEvents(final Context context, final int batchSize, final QueueCursor previousCursor) {

        synchronized (lock) {
            DBAdapter adapter = loadDBAdapter(context);
            DBAdapter.Table tableName = (previousCursor != null) ? previousCursor.tableName : DBAdapter.Table.EVENTS;

            // if previousCursor that means the batch represented by the previous cursor was processed so remove those from the db
            if (previousCursor != null) {
                adapter.cleanupEventsFromLastId(previousCursor.lastId, previousCursor.tableName);
            }

            // grab the new batch
            QueueCursor newCursor = new QueueCursor();
            newCursor.tableName = tableName;
            JSONObject queuedDBEvents = adapter.fetchEvents(tableName, batchSize);
            newCursor = updateCursorForDBObject(queuedDBEvents, newCursor);

            // if we have no events then try and fetch profile events
            if (newCursor.isEmpty() && tableName.equals(DBAdapter.Table.EVENTS)) {
                tableName = DBAdapter.Table.PROFILE_EVENTS;
                newCursor.resetForTableName(tableName);
                queuedDBEvents = adapter.fetchEvents(tableName, batchSize);
                newCursor = updateCursorForDBObject(queuedDBEvents, newCursor);
            }

            return newCursor.isEmpty() ? null : newCursor;
        }
    }

    // helper extracts the cursor data from the db object
    private static QueueCursor updateCursorForDBObject(JSONObject dbObject, QueueCursor cursor) {

        if (dbObject == null) return cursor;

        Iterator<String> keys = dbObject.keys();
        if (keys.hasNext()) {
            String key = keys.next();
            cursor.lastId = key;
            try {
                cursor.data = dbObject.getJSONArray(key);
            } catch (JSONException e) {
                cursor.lastId = null;
                cursor.data = null;
            }
        }

        return cursor;
    }

    /**
     * Only call inside our async queue.
     */
    static void clearQueues(final Context context) {
        synchronized (lock) {
            DBAdapter adapter = loadDBAdapter(context);
            DBAdapter.Table tableName = DBAdapter.Table.EVENTS;
            adapter.removeEvents(tableName);
            tableName = DBAdapter.Table.PROFILE_EVENTS;
            adapter.removeEvents(tableName);
            CommsManager.clearUserContext(context);
        }
    }

    /**
     * Adds a new event to the queue, to be sent later.
     *
     * @param context   The Android context
     * @param event     The event to be queued
     * @param eventType The type of event to be queued
     */

    // only call inside our async queue
    static void addToQueue(final Context context, final JSONObject event, final int eventType) {
        if (CommsManager.isMuted(context)) {
            return;
        }

        processEvent(context, event, eventType);
    }

    private static void processEvent(final Context context, final JSONObject event, final int eventType) {
        synchronized (lock) {
            try {

                int activityCount = CleverTapAPI.activityCount;
                activityCount = activityCount == 0 ? 1 : activityCount;
                String type;
                if (eventType == Constants.PAGE_EVENT) {
                    type = "page";
                } else if (eventType == Constants.PING_EVENT) {
                    type = "ping";
                    attachMeta(event, context);
                } else if (eventType == Constants.PROFILE_EVENT) {
                    type = "profile";
                } else if (eventType == Constants.DATA_EVENT) {
                    type = "data";
                } else {
                    type = "event";
                }

                // Complete the received event with the other params

                String currentActivityName = CleverTapAPI.getCurrentActivityName();
                if (currentActivityName != null) {
                    event.put("n", currentActivityName);
                }

                int session = SessionManager.getCurrentSession();
                event.put("s", session);
                event.put("pg", activityCount);
                event.put("type", type);
                event.put("ep", System.currentTimeMillis() / 1000);
                event.put("f", SessionManager.isFirstSession());
                event.put("lsl", SessionManager.getLastSessionLength());
                attachLastSessionActivityTrailAndPackageNameIfRequired(context, event);

                // Report any pending validation error
                ValidationResult vr = CleverTapAPI.popValidationResult();
                if (vr != null) {
                    event.put(Constants.ERROR_KEY, getErrorObject(vr));
                }

                LocalDataStore.setDataSyncFlag(context, event);

                queueEventToDB(context, event, eventType);

                updateLocalStore(context, event, eventType);

                scheduleQueueFlush(context);

            } catch (Throwable e) {
                Logger.v("Failed to queue event: " + event.toString(), e);
            }
        }
    }

    // only call inside our async queue
    private static void queueEventToDB(final Context context, final JSONObject event, final int type) {
        synchronized (lock) {
            DBAdapter adapter = loadDBAdapter(context);
            DBAdapter.Table table = (type == Constants.PROFILE_EVENT) ? DBAdapter.Table.PROFILE_EVENTS : DBAdapter.Table.EVENTS;

            int returnCode = adapter.storeObject(event, table);

            if (returnCode > 0) {
                Logger.d("Queued event: " + event.toString());
                Logger.v("Queued event to DB table " + table + ": " + event.toString());
            }
        }
    }

    // only call inside our async queue
    private static void updateLocalStore(final Context context, final JSONObject event, final int type) {
        if (type == Constants.RAISED_EVENT) {
            LocalDataStore.persistEvent(context, event, type);
        }
    }

    private static void scheduleQueueFlush(final Context context) {
        if (commsRunnable == null)
            commsRunnable = new Runnable() {
                @Override
                public void run() {
                    CommsManager.flushQueueAsync(context);
                }
            };
        // Cancel any outstanding send runnables, and issue a new delayed one
        CleverTapAPI.getHandlerUsingMainLooper().removeCallbacks(commsRunnable);
        CleverTapAPI.getHandlerUsingMainLooper().postDelayed(commsRunnable, Constants.PUSH_DELAY_MS);

        Logger.v("Scheduling delayed queue flush on main event loop");
    }

    private static JSONObject getErrorObject(ValidationResult vr) {
        JSONObject error = new JSONObject();
        try {
            error.put("c", vr.getErrorCode());
            error.put("d", vr.getErrorDesc());
        } catch (JSONException e) {
            // Won't reach here
        }
        return error;
    }

    /**
     * Attaches meta info about the current state of the device to an event.
     * Typically, this meta is added only to the ping event.
     */
    private static void attachMeta(final JSONObject o, final Context context) {
        // Memory consumption
        try {
            o.put("mc", Utils.getMemoryConsumption());
        } catch (Throwable t) {
            // Ignore
        }

        // Attach the network type
        try {
            o.put("nt", Utils.getCurrentNetworkType(context));
        } catch (Throwable t) {
            // Ignore
        }
    }

    private static void attachLastSessionActivityTrailAndPackageNameIfRequired(final Context context, final JSONObject event) {
        try {
            final String type = event.getString("type");
            // Send it only for app launched events
            if ("event".equals(type) && Constants.APP_LAUNCHED_EVENT.equals(event.getString("evtName"))) {
                final JSONArray lastSessionActivityTrail = SessionManager.getLastSessionActivityTrail();
                // The above will never be null
                event.put("lsat", lastSessionActivityTrail);
                event.put("pai", context.getPackageName());
            }
        } catch (Throwable t) {
            // Ignore
        }
    }
}
