/*
 * Author: Jude Pereira
 * Copyright (c) 2014
 */

package com.clevertap.android.sdk;

import android.content.Context;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.text.ParseException;
import java.util.*;

/**
 * Handles all push events which describe a user's profile.
 */
public final class ProfileHandler {
    private final Context context;

    private static void queueEvent(final Context context, final JSONObject event, final int eventType) {
        QueueManager.queueEvent(context, event, eventType);
    }

    private static void postAsyncSafely(final String name, final Runnable runnable) {
        CleverTapAPI.postAsyncSafely(name, runnable);
    }

    /**
     * Please do not create an object of this explicitly.
     *
     * @param context The Android context
     * @see CleverTapAPI
     */
    ProfileHandler(Context context) {
        this.context = context;
    }

    /**
     * Set a collection of unique values as a multi-value user profile property, any existing value will be overwritten.
     * Max 100 values, on reaching 100 cap, oldest value(s) will be removed.
     * Values must be Strings and are limited to 40 bytes.
     *
     * @param key    String
     * @param values {@link ArrayList} with String values
     */
    public void setMultiValuesForKey(final String key, final ArrayList<String> values) {
        postAsyncSafely("setMultiValuesForKey", new Runnable() {
            @Override
            public void run() {
                _handleMultiValues(values, key, Constants.COMMAND_SET);
            }
        });
    }

    /**
     * Add a unique value to a multi-value user profile property
     * If the property does not exist it will be created
     * <p/>
     * Max 100 values, on reaching 100 cap, oldest value(s) will be removed.
     * Values must be Strings and are limited to 40 bytes.
     * <p/>
     * If the key currently contains a scalar value, the key will be promoted to a multi-value property
     * with the current value cast to a string and the new value(s) added
     *
     * @param key   String
     * @param value String
     */
    public void addMultiValueForKey(String key, String value) {

        //noinspection ConstantConditions
        if (value == null || value.isEmpty()) {
            _generateEmptyMultiValueError(key);
            return;
        }

        addMultiValuesForKey(key, new ArrayList<String>(Collections.singletonList(value)));
    }

    /**
     * Add a collection of unique values to a multi-value user profile property
     * If the property does not exist it will be created
     * <p/>
     * Max 100 values, on reaching 100 cap, oldest value(s) will be removed.
     * Values must be Strings and are limited to 40 bytes.
     * <p/>
     * If the key currently contains a scalar value, the key will be promoted to a multi-value property
     * with the current value cast to a string and the new value(s) added
     *
     * @param key    String
     * @param values {@link ArrayList} with String values
     */
    public void addMultiValuesForKey(final String key, final ArrayList<String> values) {
        postAsyncSafely("addMultiValuesForKey", new Runnable() {
            @Override
            public void run() {
                final String command = (LocalDataStore.getProfileValueForKey(key) != null) ? Constants.COMMAND_ADD : Constants.COMMAND_SET;
                _handleMultiValues(values, key, command);
            }
        });
    }


    /**
     * Remove a unique value from a multi-value user profile property
     * <p/>
     * If the key currently contains a scalar value, prior to performing the remove operation
     * the key will be promoted to a multi-value property with the current value cast to a string.
     * If the multi-value property is empty after the remove operation, the key will be removed.
     *
     * @param key   String
     * @param value String
     */
    public void removeMultiValueForKey(String key, String value) {

        //noinspection ConstantConditions
        if (value == null || value.isEmpty()) {
            _generateEmptyMultiValueError(key);
            return;
        }

        removeMultiValuesForKey(key, new ArrayList<String>(Collections.singletonList(value)));
    }

    /**
     * Remove a collection of unique values from a multi-value user profile property
     * <p/>
     * If the key currently contains a scalar value, prior to performing the remove operation
     * the key will be promoted to a multi-value property with the current value cast to a string.
     * <p/>
     * If the multi-value property is empty after the remove operation, the key will be removed.
     *
     * @param key    String
     * @param values {@link ArrayList} with String values
     */
    public void removeMultiValuesForKey(final String key, final ArrayList<String> values) {
        postAsyncSafely("removeMultiValuesForKey", new Runnable() {
            @Override
            public void run() {
                _handleMultiValues(values, key, Constants.COMMAND_REMOVE);
            }
        });
    }

    /**
     * Remove the user profile property value specified by key from the user profile
     *
     * @param key String
     */
    public void removeValueForKey(final String key) {
        postAsyncSafely("removeValueForKey", new Runnable() {
            @Override
            public void run() {
                _removeValueForKey(key);
            }
        });
    }

    private void _removeValueForKey(String key) {
        try {
            key = (key == null) ? "" : key; // so we will generate a validation error later on

            // validate the key
            ValidationResult vr;

            vr = Validator.cleanObjectKey(key);
            key = vr.getObject().toString();

            if (key.isEmpty()) {
                ValidationResult error = new ValidationResult();
                error.setErrorCode(512);
                error.setErrorDesc("Key is empty, profile removeValueForKey aborted.");
                pushValidationResult(error);
                Logger.d("Key is empty, profile removeValueForKey aborted");
                // Abort
                return;
            }
            // Check for an error
            if (vr.getErrorCode() != 0) {
                pushValidationResult(vr);
            }

            // remove from the local profile
            LocalDataStore.removeProfileField(context, key);

            // send the delete command
            JSONObject command = new JSONObject().put(Constants.COMMAND_DELETE, true);
            JSONObject update = new JSONObject().put(key, command);
            pushBasicProfile(update);

            Logger.v("removing value for key " + key + " from user profile");

        } catch (Throwable t) {
            Logger.v("Failed to remove profile value for key " + key, t);
        }
    }

    /**
     * Push a profile update.
     *
     * @param profile A {@link Map}, with keys as strings, and values as {@link String},
     *                {@link Integer}, {@link Long}, {@link Boolean}, {@link Float}, {@link Double},
     *                {@link java.util.Date}, or {@link Character}
     */
    public void push(final Map<String, Object> profile) {
        if (profile == null || profile.isEmpty())
            return;

        postAsyncSafely("profilePush", new Runnable() {
            @Override
            public void run() {
                _push(profile);
            }
        });
    }

    private void _push(Map<String, Object> profile) {
        if (profile == null || profile.isEmpty())
            return;

        try {
            ValidationResult vr;
            JSONObject customProfile = new JSONObject();
            JSONObject fieldsToUpdateLocally = new JSONObject();
            for (String key : profile.keySet()) {
                Object value = profile.get(key);

                vr = Validator.cleanObjectKey(key);
                key = vr.getObject().toString();
                // Check for an error
                if (vr.getErrorCode() != 0) {
                    pushValidationResult(vr);
                }

                if (key == null || key.isEmpty()) {
                    ValidationResult keyError = new ValidationResult();
                    keyError.setErrorCode(512);
                    final String keyErr = "Profile push key is empty";
                    keyError.setErrorDesc(keyErr);
                    pushValidationResult(keyError);
                    Logger.d(keyErr);
                    // Skip this property
                    continue;
                }

                try {
                    vr = Validator.cleanObjectValue(value, Validator.ValidationContext.Profile);
                } catch (Throwable e) {
                    // The object was neither a String, Boolean, or any number primitives
                    ValidationResult error = new ValidationResult();
                    error.setErrorCode(512);
                    final String err = "Object value wasn't a primitive (" + value + ") for profile field " + key;
                    error.setErrorDesc(err);
                    pushValidationResult(error);
                    Logger.d(err);
                    // Skip this property
                    continue;
                }
                value = vr.getObject();
                // Check for an error
                if (vr.getErrorCode() != 0) {
                    pushValidationResult(vr);
                }

                // test Phone:  if no device country code, test if phone starts with +, log but always send
                if (key.equalsIgnoreCase("Phone")) {
                    try {
                        value = value.toString();
                        String countryCode = DeviceInfo.getCountryCode();
                        if (countryCode == null || countryCode.isEmpty()) {
                            String _value = (String) value;
                            if (!_value.startsWith("+")) {
                                ValidationResult error = new ValidationResult();
                                error.setErrorCode(512);
                                final String err = "Device country code not available and profile phone: " + value + " does not appear to start with country code";
                                error.setErrorDesc(err);
                                pushValidationResult(error);
                                Logger.d(err);
                            }
                        }
                        Logger.v("Profile phone is: "+ value + " device country code is: " + ((countryCode != null) ? countryCode : "null"));
                    } catch (Exception e) {
                        pushValidationResult(new ValidationResult(512, "Invalid phone number"));
                        Logger.d("Invalid phone number: " + e.getLocalizedMessage());
                        continue;
                    }
                }

                // add to the local profile update object
                fieldsToUpdateLocally.put(key, value);
                customProfile.put(key, value);
            }

            Logger.v("Constructed custom profile: " + customProfile.toString());

            // update local profile values
            if (fieldsToUpdateLocally.length() > 0) {
                LocalDataStore.setProfileFields(context, fieldsToUpdateLocally);
            }

            pushBasicProfile(customProfile);

        } catch (Throwable t) {
            // Will not happen
            Logger.v("Failed to push profile", t);
        }
    }

    private String getGraphUserPropertySafely(JSONObject graphUser, String key, String def) {
        try {
            String prop = (String) graphUser.get(key);
            if (prop != null)
                return prop;
            else
                return def;
        } catch (Throwable t) {
            return def;
        }
    }

    /**
     * Pushes everything available in the JSON object returned by the Facebook GraphRequest
     *
     * @param graphUser The object returned from Facebook
     */
    public void pushFacebookUser(final JSONObject graphUser) {
        postAsyncSafely("pushFacebookUser", new Runnable() {
            @Override
            public void run() {
                _pushFacebookUser(graphUser);
            }
        });
    }

    private void _pushFacebookUser(JSONObject graphUser) {
        try {
            if (graphUser == null)
                return;
            // Note: No validations are required here, as everything is controlled
            String name = getGraphUserPropertySafely(graphUser, "name", "");
            try {
                // Certain users have nasty looking names - unicode chars, validate for any
                // not allowed chars
                ValidationResult vr = Validator.cleanObjectValue(name,Validator.ValidationContext.Profile);
                name = vr.getObject().toString();

                if (vr.getErrorCode() != 0) {
                    pushValidationResult(vr);
                }
            } catch (IllegalArgumentException e) {
                // Weird name, wasn't a string, or any number
                // This would never happen with FB
                name = "";
            }

            String gender = getGraphUserPropertySafely(graphUser, "gender", null);
            // Convert to WR format
            if (gender != null) {
                if (gender.toLowerCase().startsWith("m"))
                    gender = "M";
                else if (gender.toLowerCase().startsWith("f"))
                    gender = "F";
                else
                    gender = "";
            } else {
                gender = null;
            }
            String email = getGraphUserPropertySafely(graphUser, "email", "");

            String birthday = getGraphUserPropertySafely(graphUser, "birthday", null);
            if (birthday != null) {
                // Some users don't have the year of birth mentioned
                // FB returns only MM/dd in those cases
                if (birthday.matches("^../..")) {
                    // This means that the year is not available(~30% of the times)
                    // Ignore
                    birthday = "";
                } else {
                    try {
                        Date date = Constants.FB_DOB_DATE_FORMAT.parse(birthday);
                        birthday = "$D_" + (int) (date.getTime() / 1000);
                    } catch (ParseException e) {
                        // Differs from the specs
                        birthday = "";
                    }
                }
            }

            String work;
            try {
                JSONArray workArray = graphUser.getJSONArray("work");
                work = (workArray.length() > 0) ? "Y" : "N";
            } catch (Throwable t) {
                work = null;
            }

            String education;
            try {
                JSONArray eduArray = graphUser.getJSONArray("education");
                // FB returns the education levels in a descending order - highest = last entry
                String fbEdu = eduArray.getJSONObject(eduArray.length() - 1).getString("type");
                if (fbEdu.toLowerCase().contains("high school"))
                    education = "School";
                else if (fbEdu.toLowerCase().contains("college"))
                    education = "College";
                else if (fbEdu.toLowerCase().contains("graduate school"))
                    education = "Graduate";
                else
                    education = "";
            } catch (Throwable t) {
                // No education info available
                education = null;
            }

            String id = getGraphUserPropertySafely(graphUser, "id", "");

            String married = getGraphUserPropertySafely(graphUser, "relationship_status", null);
            if (married != null) {
                if (married.equalsIgnoreCase("married")) {
                    married = "Y";
                } else {
                    married = "N";
                }
            }

            final JSONObject profile = new JSONObject();
            if (id != null && id.length() > 3) profile.put("FBID", id);
            if (name != null && name.length() > 3) profile.put("Name", name);
            if (email != null && email.length() > 3) profile.put("Email", email);
            if (gender != null && !gender.trim().equals("")) profile.put("Gender", gender);
            if (education != null && !education.trim().equals("")) profile.put("Education", education);
            if (work != null && !work.trim().equals("")) profile.put("Employed", work);
            if (birthday != null && birthday.length() > 3) profile.put("DOB", birthday);
            if (married != null && !married.trim().equals("")) profile.put("Married", married);

            pushBasicProfile(profile);
        } catch (Throwable t) {
            Logger.v("Failed to parse graph user object successfully", t);
        }
    }

    /**
     * Pushes basic user information, if available.
     * <p/>
     * Basic information is pushed daily. There is no need to call this method
     * explicitly.
     */
    void pushBasicProfile(JSONObject baseProfile) {
        try {

            CleverTapAPI api = CleverTapAPI.getInstance(context);
            String guid = api.getCleverTapID();

            JSONObject profileEvent = new JSONObject();

            if (baseProfile != null && baseProfile.length() > 0) {
                Iterator i = baseProfile.keys();
                while (i.hasNext()) {
                    String next = i.next().toString();

                    // need to handle command-based JSONObject props here now
                    Object value = null;
                    try {
                        value = baseProfile.getJSONObject(next);
                    } catch (Throwable t) {
                        try {
                            value = baseProfile.get(next);
                        } catch (JSONException e) {
                            //no-op
                        }
                    }

                    if (value != null) {
                        profileEvent.put(next, value);

                        // cache the valid identifier: guid pairs
                        if (Constants.PROFILE_IDENTIFIER_KEYS.contains(next)) {
                            try {
                                api.cacheGUIDForIdentifier(guid, next, value.toString());
                            } catch (Throwable t) {
                                // no-op
                            }
                        }
                    }
                }
            }

            try {
                String carrier = DeviceInfo.getCarrier();
                if (carrier != null && !carrier.equals("")) {
                    profileEvent.put("Carrier", carrier);
                }

                String cc = DeviceInfo.getCountryCode();
                if (cc != null && !cc.equals("")) {
                    profileEvent.put("cc", cc);
                }

                profileEvent.put("tz", TimeZone.getDefault().getID());

                JSONObject event = new JSONObject();
                event.put("profile", profileEvent);
                queueEvent(context, event, Constants.PROFILE_EVENT);
            } catch (JSONException e) {
                // We won't get here
                Logger.v("FATAL: Creating basic profile update event failed!");
            }
        } catch (Throwable t) {
            Logger.v("Basic profile sync", t);
        }
    }

    /**
     * Pushes an enum type of property for a user.
     * <p/>
     * Deprecated. Use {@link #push(Map)}
     *
     * @param key   The enum key
     * @param value The value for this profile
     */
    @Deprecated
    public void pushEnum(String key, String value) {
        HashMap<String, Object> m = new HashMap<String, Object>();
        m.put(key, value);
        push(m);
    }


    /**
     * Pushes everything useful within the Google Plus
     * {@link com.google.android.gms.plus.model.people.Person} object.
     *
     * @param person The {@link com.google.android.gms.plus.model.people.Person} object
     * @see com.google.android.gms.plus.model.people.Person
     */
    public void pushGooglePlusPerson(final com.google.android.gms.plus.model.people.Person person) {
        postAsyncSafely("pushGooglePlusPerson", new Runnable() {
            @Override
            public void run() {
                _pushGooglePlusPerson(person);
            }
        });
    }

    private void _pushGooglePlusPerson(com.google.android.gms.plus.model.people.Person person) {
        if (person == null) {
            return;
        }
        try {
            // Note: No validations are required here, as everything is controlled
            String name = "";
            if (person.hasDisplayName()) {
                try {
                    // Certain users have nasty looking names - unicode chars, validate for any
                    // not allowed chars
                    name = person.getDisplayName();
                    ValidationResult vr = Validator.cleanObjectValue(name, Validator.ValidationContext.Profile);
                    name = vr.getObject().toString();

                    if (vr.getErrorCode() != 0) {
                        pushValidationResult(vr);
                    }
                } catch (Throwable t) {
                    // Weird name, wasn't a string, or any number
                    // This would never happen with G+
                    name = "";
                }
            }

            String gender = null;
            if (person.hasGender()) {
                if (person.getGender() == com.google.android.gms.plus.model.people.Person.Gender.MALE) {
                    gender = "M";
                } else if (person.getGender() == com.google.android.gms.plus.model.people.Person.Gender.FEMALE) {
                    gender = "F";
                }
            }

            String birthday = null;

            if (person.hasBirthday()) {
                // We have the string as YYYY-MM-DD
                try {
                    Date date = Constants.GP_DOB_DATE_FORMAT.parse(person.getBirthday());
                    birthday = "$D_" + (int) (date.getTime() / 1000);
                } catch (Throwable t) {
                    // Differs from the specs
                    birthday = null;
                }
            }

            String work = null;
            if (person.hasOrganizations()) {
                List<com.google.android.gms.plus.model.people.Person.Organizations> organizations = person.getOrganizations();
                for (com.google.android.gms.plus.model.people.Person.Organizations o : organizations) {
                    if (o.getType() == com.google.android.gms.plus.model.people.Person.Organizations.Type.WORK) {
                        work = "Y";
                        break;
                    }
                }
            }

            String id = "";
            if (person.hasId()) {
                id = person.getId();
            }

            String married = null;
            if (person.hasRelationshipStatus()) {
                if (person.getRelationshipStatus() == com.google.android.gms.plus.model.people.Person.RelationshipStatus.MARRIED) {
                    married = "Y";
                } else {
                    married = "N";
                }
            }

            // Construct json object from the data
            final JSONObject profile = new JSONObject();
            if (id != null && id.trim().length() > 0) profile.put("GPID", id);
            if (name != null && name.trim().length() > 0) profile.put("Name", name);
            if (gender != null && gender.trim().length() > 0) profile.put("Gender", gender);
            if (work != null && work.trim().length() > 0) profile.put("Employed", work);
            if (birthday != null && birthday.trim().length() > 4) profile.put("DOB", birthday);
            if (married != null && married.trim().length() > 0) profile.put("Married", married);

            pushBasicProfile(profile);
        } catch (Throwable t) {
            // We won't get here
            Logger.v("FATAL: Creating G+ profile update event failed!");
        }
    }

    /**
     * Return the user profile property value for the specified key
     *
     * @param name String
     * @return {@link JSONArray}, String or null
     */
    public Object getProperty(String name) {
        if (!LocalDataStore.isPersonalisationEnabled(context)) return null;
        return LocalDataStore.getProfileProperty(context, name);
    }

    // use for internal profile getter doesn't do the personalization check
    private Object _getProfilePropertyIgnorePersonalizationFlag(String key) {
        return LocalDataStore.getProfileValueForKey(key);
    }

    // private multi-value handlers and helpers

    private void _handleMultiValues(ArrayList<String> values, String key, String command) {
        if (key == null) return;

        if (values == null || values.isEmpty()) {
            _generateEmptyMultiValueError(key);
            return;
        }

        ValidationResult vr;

        // validate the key
        vr = Validator.cleanMultiValuePropertyKey(key);

        // Check for an error
        if (vr.getErrorCode() != 0) {
            pushValidationResult(vr);
        }

        // reset the key
        Object _key = vr.getObject();
        String cleanKey = (_key != null) ? vr.getObject().toString() : null;

        // if key is empty generate an error and return
        if (cleanKey == null || cleanKey.isEmpty()) {
            _generateInvalidMultiValueKeyError(key);
            return;
        }

        key = cleanKey;

        try {
            JSONArray currentValues = _constructExistingMultiValue(key, command);
            JSONArray newValues = _cleanMultiValues(values, key);
            _validateAndPushMultiValue(currentValues, newValues, values, key, command);

        } catch (Throwable t) {
            Logger.v("Error handling multi value operation for key " + key, t);
        }
    }

    private JSONArray _constructExistingMultiValue(String key, String command) {

        Boolean remove = command.equals(Constants.COMMAND_REMOVE);
        Boolean add = command.equals(Constants.COMMAND_ADD);

        // only relevant for add's and remove's; a set overrides the existing value, so return a new array
        if (!remove && !add) return new JSONArray();

        Object existing = _getProfilePropertyIgnorePersonalizationFlag(key);

        // if there is no existing value
        if (existing == null) {
            // if its a remove then return null to abort operation
            // no point in running remove against a nonexistent value
            if (remove) return null;

            // otherwise return an empty array
            return new JSONArray();
        }

        // value exists

        // the value should only ever be a JSONArray or scalar (String really)

        // if its already a JSONArray return that
        if (existing instanceof JSONArray) return (JSONArray) existing;

        // handle a scalar value as the existing value
        /*
            if its an add, our rule is to promote the scalar value to multi value and include the cleaned stringified
            scalar value as the first element of the resulting array

            NOTE: the existing scalar value is currently limited to 120 bytes; when adding it to a multi value
            it is subject to the current 40 byte limit

            if its a remove, our rule is to delete the key from the local copy
            if the cleaned stringified existing value is equal to any of the cleaned values passed to the remove method

            if its an add, return an empty array as the default,
            in the event the existing scalar value fails stringifying/cleaning

            returning null will signal that a remove operation should be aborted,
            as there is no valid promoted multi value to remove against
         */

        JSONArray _default = (add) ? new JSONArray() : null;

        String stringified = _stringifyAndCleanScalarProfilePropValue(existing);

        return (stringified != null) ? new JSONArray().put(stringified) : _default;
    }

    private String _stringifyScalarProfilePropValue(Object value) {
        String val = null;

        try {
            val = value.toString();
        } catch (Exception e) {
            // no-op
        }

        return val;
    }

    private String _stringifyAndCleanScalarProfilePropValue(Object value) {
        String val = _stringifyScalarProfilePropValue(value);

        if (val != null) {
            ValidationResult vr = Validator.cleanMultiValuePropertyValue(val);

            // Check for an error
            if (vr.getErrorCode() != 0) {
                pushValidationResult(vr);
            }

            Object _value = vr.getObject();
            val = (_value != null) ? vr.getObject().toString() : null;
        }

        return val;
    }

    private JSONArray _cleanMultiValues(ArrayList<String> values, String key) {

        try {
            if (values == null || key == null) return null;

            JSONArray cleanedValues = new JSONArray();
            ValidationResult vr;

            // loop through and clean the new values
            for (String value : values) {
                value = (value == null) ? "" : value;  // so we will generate a validation error later on

                // validate value
                vr = Validator.cleanMultiValuePropertyValue(value);

                // Check for an error
                if (vr.getErrorCode() != 0) {
                    pushValidationResult(vr);
                }

                // reset the value
                Object _value = vr.getObject();
                value = (_value != null) ? vr.getObject().toString() : null;

                // if value is empty generate an error and return
                if (value == null || value.isEmpty()) {
                    _generateEmptyMultiValueError(key);
                    // Abort
                    return null;
                }
                // add to the newValues to be merged
                cleanedValues.put(value);
            }

            return cleanedValues;

        } catch (Throwable t) {
            Logger.v("Error cleaning multi values for key " + key, t);
            _generateEmptyMultiValueError(key);
            return null;
        }
    }

    private void _validateAndPushMultiValue(JSONArray currentValues, JSONArray newValues, ArrayList<String> originalValues, String key, String command) {

        try {

            // if any of these are null, indicates some problem along the way so abort operation
            if (currentValues == null || newValues == null || originalValues == null || key == null || command == null)
                return;

            String mergeOperation = command.equals(Constants.COMMAND_REMOVE) ? Validator.REMOVE_VALUES_OPERATION : Validator.ADD_VALUES_OPERATION;

            // merge currentValues and newValues
            ValidationResult vr = Validator.mergeMultiValuePropertyForKey(currentValues, newValues, mergeOperation, key);

            // Check for an error
            if (vr.getErrorCode() != 0) {
                pushValidationResult(vr);
            }

            // set the merged local values array
            JSONArray localValues = (JSONArray) vr.getObject();

            // update local profile
            // remove an empty array
            if (localValues == null || localValues.length() <= 0) {
                LocalDataStore.removeProfileField(context, key);
            } else {
                // not empty so save to local profile
                LocalDataStore.setProfileField(context, key, localValues);
            }

            // push to server
            JSONObject commandObj = new JSONObject();
            commandObj.put(command, new JSONArray(originalValues));

            JSONObject fields = new JSONObject();
            fields.put(key, commandObj);

            pushBasicProfile(fields);

            Logger.v("Constructed multi-value profile push: " + fields.toString());

        } catch (Throwable t) {
            Logger.v("Error pushing multiValue for key " + key, t);
        }
    }

    private void _generateEmptyMultiValueError(String key) {
        ValidationResult error = new ValidationResult();
        String msg = "Invalid multi value for key " + key + ", profile multi value operation aborted.";
        error.setErrorCode(512);
        error.setErrorDesc(msg);
        pushValidationResult(error);
        Logger.d(msg);
    }

    private void _generateInvalidMultiValueKeyError(String key) {
        ValidationResult error = new ValidationResult();
        error.setErrorCode(523);
        error.setErrorDesc("Invalid multi-value property key " + key);
        pushValidationResult(error);
        Logger.d("Invalid multi-value property key " + key + " profile multi value operation aborted");
    }

    private void pushValidationResult(ValidationResult vr) {
        CleverTapAPI.pushValidationResult(vr);
    }
}
