/*
 * Author: Jude Pereira
 * Copyright (c) 2014
 */

package com.clevertap.android.sdk;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.SharedPreferences;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.net.Uri;
import android.util.Base64;
import com.clevertap.android.sdk.exceptions.CleverTapMetaDataNotFoundException;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import javax.net.ssl.HttpsURLConnection;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URL;
import java.util.Iterator;
import java.util.Map;

/**
 * Provides various methods to communicate with the CleverTap servers.
 */
final class CommsManager {
    static final String NAMESPACE_ARP = "ARP";
    static final String NAMESPACE_IJ = "IJ";

    private static final String HEADER_DOMAIN_NAME = "X-WZRK-RD";
    private static final String HEADER_MUTE = "X-WZRK-MUTE";

    private static final String KEY_DOMAIN_NAME = "comms_dmn";
    private static final String KEY_MUTED = "comms_mtd";

    private static final String KEY_LAST_TS = "comms_last_ts";
    private static final String KEY_FIRST_TS = "comms_first_ts";

    static final String KEY_I = "comms_i";
    static final String KEY_J = "comms_j";

    private static int mResponseFailureCount = 0;

    private static int currentRequestTimestamp = 0;

    private static final String PRIMARY_DOMAIN = "wzrkt.com";

    static void flushQueueAsync(final Context context) {
        CleverTapAPI.postAsyncSafely("CommsManager#flushQueueAsync", new Runnable() {
            @Override
            public void run() {
                flushQueueSync(context);
            }
        });
    }

    static void flushQueueSync(final Context context) {
        if (!isOnline(context)) {
            Logger.logFine("Network connectivity unavailable. Will retry later");
            return;
        }

        if (needsHandshakeForDomain(context)) {
            mResponseFailureCount = 0;
            setDomain(context, null);
            performHandshakeForDomain(context, new Runnable() {
                @Override
                public void run() {
                    flushDBQueue(context);
                }
            });
        } else {
            flushDBQueue(context);
        }
    }

    private static void flushDBQueue(final Context context) {
        Logger.logFine("Somebody has invoked me to send the queue to CleverTap servers");

        QueueManager.QueueCursor cursor;
        QueueManager.QueueCursor previousCursor = null;
        boolean loadMore = true;
        while (loadMore) {
            cursor = QueueManager.getQueuedEvents(context, 50, previousCursor);

            if (cursor == null || cursor.isEmpty()) {
                Logger.logFine("No events in the queue, bailing");
                break;
            }

            previousCursor = cursor;
            JSONArray queue = cursor.getData();

            if (queue == null || queue.length() <= 0) {
                Logger.logFine("No events in the queue, bailing");
                break;
            }

            loadMore = sendQueue(context, queue);
        }
    }

    private static HttpsURLConnection buildHttpsURLConnection(final Context context, final String endpoint)
            throws IOException, CleverTapMetaDataNotFoundException {

        URL url = new URL(endpoint);
        HttpsURLConnection conn = (HttpsURLConnection) url.openConnection();
        conn.setConnectTimeout(10000);
        conn.setReadTimeout(10000);
        conn.setRequestProperty("Content-Type", "application/json; charset=utf-8");
        conn.setRequestProperty("X-CleverTap-Account-ID", ManifestMetaData.getMetaData(context, Constants.LABEL_ACCOUNT_ID));
        conn.setRequestProperty("X-CleverTap-Token", ManifestMetaData.getMetaData(context, Constants.LABEL_TOKEN));
        conn.setInstanceFollowRedirects(false);
        return conn;
    }

    private static void performHandshakeForDomain(final Context context, final Runnable handshakeSuccessCallback) {
        if (isMuted(context)) {
            return;
        }


        final String endpoint = getEndpoint(context, true);
        Logger.logExtraFine("Performing handshake with " + endpoint);

        HttpsURLConnection conn = null;
        try {
            conn = buildHttpsURLConnection(context, endpoint);
            final int responseCode = conn.getResponseCode();
            if (responseCode != 200) {
                Logger.error("Invalid HTTP status code received for handshake - " + responseCode);
                return;
            }

            Logger.logExtraFine("Received success from handshake :)");

            if (processIncomingHeaders(context, conn)) {
                Logger.logExtraFine("We are not muted");
                // We have a new domain, run the callback
                handshakeSuccessCallback.run();
            }
        } catch (Throwable t) {
            Logger.error("Failed to perform handshake!", t);
        } finally {
            if (conn != null) {
                try {
                    conn.getInputStream().close();
                    conn.disconnect();
                } catch (Throwable t) {
                    // Ignore
                }
            }
        }
    }

    /**
     * @return true if the network request succeeded. Anything non 200 results in a false.
     */
    private static boolean sendQueue(final Context context, final JSONArray queue) {

        if (queue == null || queue.length() <= 0) return false;

        HttpsURLConnection conn = null;
        try {
            final String endpoint = getEndpoint(context, false);

            // This is just a safety check, which would only arise
            // if upstream didn't adhere to the protocol (sent nothing during the initial handshake)
            if (endpoint == null) {
                return false;
            }

            conn = buildHttpsURLConnection(context, endpoint);
            Logger.logFine("Using endpoint " + endpoint);

            final String body;

            synchronized (CommsManager.class) {

                final String req = insertHeader(context, queue);
                Logger.logFine("Send queue contains " + queue.length() + " items: " + req);
                conn.setDoOutput(true);
                conn.getOutputStream().write(req.getBytes("UTF-8"));

                final int responseCode = conn.getResponseCode();

                // Always check for a 200 OK
                if (responseCode != 200) {
                    throw new IOException("Response code is not 200. It is " + responseCode);
                }

                // Check for a change in domain
                final String newDomain = conn.getHeaderField(HEADER_DOMAIN_NAME);
                if (newDomain != null && newDomain.trim().length() > 0) {
                    if (hasDomainChanged(context, newDomain)) {
                        // The domain has changed. Return a status of -1 so that the caller retries
                        setDomain(context, newDomain);
                        Logger.logExtraFine("The domain has changed to " + newDomain + ". The request will be retried shortly.");
                        return false;
                    }
                }

                if (processIncomingHeaders(context, conn)) {
                    BufferedReader br = new BufferedReader(new InputStreamReader(conn.getInputStream(), "utf-8"));

                    StringBuilder sb = new StringBuilder();
                    String line;
                    while ((line = br.readLine()) != null) {
                        sb.append(line);
                    }
                    body = sb.toString();
                    processResponse(context, body);
                }

                setLastRequestTimestamp(context, currentRequestTimestamp);
                setFirstRequestTimestampIfNeeded(context, currentRequestTimestamp);

                Logger.logFine("Completed successfully");
            }
            mResponseFailureCount = 0;
            return true;
        } catch (Throwable e) {
            Logger.logFine("An exception occurred while trying to send the queue", e);
            mResponseFailureCount++;
            return false;
        } finally {
            if (conn != null) {
                try {
                    conn.getInputStream().close();
                    conn.disconnect();
                } catch (Throwable t) {
                    // Ignore
                }
            }
        }
    }

    /**
     * Processes the incoming response headers for a change in domain and/or mute.
     *
     * @return True to continue sending requests, false otherwise.
     */
    private static boolean processIncomingHeaders(final Context context,
                                                  final HttpsURLConnection conn) {

        final String muteCommand = conn.getHeaderField(HEADER_MUTE);
        if (muteCommand != null && muteCommand.trim().length() > 0) {
            if (muteCommand.equals("true")) {
                setMuted(context, true);
                return false;
            } else {
                setMuted(context, false);
            }
        }

        final String domainName = conn.getHeaderField(HEADER_DOMAIN_NAME);
        if (domainName == null || domainName.trim().length() == 0) {
            return true;
        }

        setMuted(context, false);
        setDomain(context, domainName);
        return true;
    }

    private static String insertHeader(Context context, JSONArray arr) {
        try {
            // Insert our header at the first position
            final JSONObject header = new JSONObject();
            final CleverTapAPI api = CleverTapAPI.getInstance(context);

            String deviceId = api.getCleverTapID();
            if (deviceId != null && !deviceId.equals("")) {
                header.put("g", deviceId);
            } else {
                Logger.error("CRITICAL: Couldn't finalise on a device ID!");
            }

            header.put("type", "meta");

            JSONObject appFields = api.getAppLaunchedFields();
            header.put("af", appFields);

            long i = getI(context);
            if (i > 0) {
                header.put("_i", i);
            }

            long j = getJ(context);
            if (j > 0) {
                header.put("_j", j);
            }

            String accountId = null, token = null;
            try {
                accountId = ManifestMetaData.getMetaData(context, Constants.LABEL_ACCOUNT_ID);
                token = ManifestMetaData.getMetaData(context, Constants.LABEL_TOKEN);
            } catch (CleverTapMetaDataNotFoundException e) {
                // No account ID/token, be gone now!
                Logger.logFine("Account ID/token not found, will not add to queue header");
            }

            if (accountId != null) {
                header.put("id", accountId);
            }

            if (token != null) {
                header.put("tk", token);
            }

            header.put("l_ts", getLastRequestTimestamp(context));
            header.put("f_ts", getFirstRequestTimestamp(context));

            // Attach ARP
            try {
                final JSONObject arp = getARP(context);
                if (arp != null && arp.length() > 0) {
                    header.put("arp", arp);
                }
            } catch (Throwable t) {
                Logger.error("Failed to attach ARP", t);
            }

            JSONObject ref = new JSONObject();
            try {

                String utmSource = SessionHandler.getSource();
                if (utmSource != null) {
                    ref.put("us", utmSource);
                }

                String utmMedium = SessionHandler.getMedium();
                if (utmMedium != null) {
                    ref.put("um", utmMedium);
                }

                String utmCampaign = SessionHandler.getCampaign();
                if (utmCampaign != null) {
                    ref.put("uc", utmCampaign);
                }

                if (ref.length() > 0) {
                    header.put("ref", ref);
                }

            } catch (Throwable t) {
                Logger.error("Failed to attach ref", t);
            }

            JSONObject wzrkParams = SessionHandler.getWzrkParams();
            if (wzrkParams != null && wzrkParams.length() > 0) {
                header.put("wzrk_ref", wzrkParams);
            }

            InAppFCManager.attachToHeader(context, header);

            // Resort to string concat for backward compatibility
            return "[" + header.toString() + ", " + arr.toString().substring(1);
        } catch (Throwable t) {
            Logger.error("CommsManager: Failed to attach header", t);
            return arr.toString();
        }
    }

    private static boolean isOnline(Context context) {
        try {
            ConnectivityManager cm =
                    (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
            NetworkInfo netInfo = cm.getActiveNetworkInfo();
            return netInfo != null && netInfo.isConnected();
        } catch (Throwable ignore) {
            // Can't decide whether or not the net is available, maybe it just is
            // Therefore, let's wait for a timeout or something to create an IOException later
            return true;
        }
    }

    private static void processResponse(final Context context, final String responseStr) {
        if (responseStr == null) return;

        try {
            Logger.logFine("Trying to process response: " + responseStr);
            JSONObject response = new JSONObject(responseStr);
            try {
                if (Constants.ENABLE_INAPP) {
                    InAppManager.processResponseAsync(response, context);
                }
            } catch (Throwable t) {
                Logger.error("Failed to process in-app notifications from the response!", t);
            }

            // Always look for a GUID in the response, and if present, then perform a force update
            try {
                if (response.has("g")) {
                    final String deviceID = response.getString("g");
                    DeviceInfo.forceUpdateDeviceId(deviceID);
                    Logger.logFine("Got a new device ID: " + deviceID);
                }
            } catch (Throwable t) {
                Logger.error("Failed to update device ID!", t);
            }

            try {
                LocalDataStore.syncWithUpstream(context, response);
            } catch (Throwable t) {
                Logger.error("Failed to sync local cache with upstream", t);
            }

            // Handle "arp" (additional request parameters)
            try {
                if (response.has("arp")) {
                    final JSONObject arp = (JSONObject) response.get("arp");
                    if (arp.length() > 0) {
                        handleARPUpdate(context, arp);
                    }
                }
            } catch (Throwable t) {
                Logger.logFine("Failed to process ARP", t);
            }

            // Handle i
            try {
                if (response.has("_i")) {
                    final long i = response.getLong("_i");
                    setI(context, i);
                }
            } catch (Throwable t) {
                // Ignore
            }

            // Handle j
            try {
                if (response.has("_j")) {
                    final long j = response.getLong("_j");
                    setJ(context, j);
                }
            } catch (Throwable t) {
                // Ignore
            }

            // Handle "console" - print them as info to the console
            try {
                if (response.has("console")) {
                    final JSONArray console = (JSONArray) response.get("console");
                    if (console.length() > 0) {
                        for (int i = 0; i < console.length(); i++) {
                            Logger.log(console.get(i).toString());
                        }
                    }
                }
            } catch (Throwable t) {
                // Ignore
            }

            // Handle server set debug level
            try {
                if (response.has("dbg_lvl")) {
                    final int debugLevel = response.getInt("dbg_lvl");
                    if (debugLevel >= 0) {
                        CleverTapAPI.setDebugLevel(debugLevel);
                        Logger.logFine("Set debug level to " + debugLevel + " for this session (set by upstream)");
                    }
                }
            } catch (Throwable t) {
                // Ignore
            }

            // Handle stale_inapp
            try {
                InAppFCManager.processResponse(context, response);
            } catch (Throwable t) {
                // Ignore
            }
        } catch (Throwable t) {
            mResponseFailureCount++;
            Logger.error("Failed to send events to CleverTap", t);
        }
    }

    private static boolean needsHandshakeForDomain(final Context context) {
        final String domain = getDomainFromPrefsOrMetadata(context);
        return domain == null || mResponseFailureCount > 5;
    }

    /**
     * @return true if the mute command was sent anytime between now and now - 24 hours.
     */
    static boolean isMuted(final Context context) {
        final int now = (int) (System.currentTimeMillis() / 1000);
        final int muteTS = StorageHelper.getInt(context, KEY_MUTED, 0);

        return now - muteTS < 24 * 60 * 60;
    }

    private static void setMuted(final Context context, boolean mute) {
        if (mute) {
            final int now = (int) (System.currentTimeMillis() / 1000);
            StorageHelper.putInt(context, KEY_MUTED, now);
            setDomain(context, null);

            // Clear all the queues
            CleverTapAPI.postAsyncSafely("CommsManager#setMuted", new Runnable() {
                @Override
                public void run() {
                    QueueManager.clearQueues(context);
                }
            });
        } else {
            StorageHelper.putInt(context, KEY_MUTED, 0);
        }
    }

    private static boolean hasDomainChanged(final Context context, final String newDomain) {
        final String oldDomain = StorageHelper.getString(context, KEY_DOMAIN_NAME, null);
        return !newDomain.equals(oldDomain);
    }

    private static String getDomainFromPrefsOrMetadata(final Context context) {
        try {
            final String region = ManifestMetaData.getMetaData(context, Constants.LABEL_REGION);
            if (region != null && region.trim().length() > 0) {
                // Always set this to 0 so that the handshake is not performed during a HTTP failure
                mResponseFailureCount = 0;
                return region.trim().toLowerCase() + "." + PRIMARY_DOMAIN;
            }
        } catch (CleverTapMetaDataNotFoundException e) {
            // Ignore
        }

        return StorageHelper.getString(context, KEY_DOMAIN_NAME, null);
    }

    private static String getDomain(final Context context, boolean defaultToHandshakeURL) {
        String domain = getDomainFromPrefsOrMetadata(context);

        final boolean emptyDomain = domain == null || domain.trim().length() == 0;
        if (emptyDomain && !defaultToHandshakeURL) {
            return null;
        }

        if (emptyDomain) {
            domain = PRIMARY_DOMAIN + "/hello";
        } else {
            domain += "/a1";
        }

        return domain;
    }

    private static void setDomain(final Context context, String domainName) {
        Logger.logExtraFine("Setting domain to " + domainName);
        StorageHelper.putString(context, KEY_DOMAIN_NAME, domainName);
    }

    private static String getEndpoint(final Context context, final boolean defaultToHandshakeURL) {
        String domain = getDomain(context, defaultToHandshakeURL);

        if (domain == null && !defaultToHandshakeURL) {
            return null;
        }

        final boolean needsHandshake = needsHandshakeForDomain(context);

        String endpoint = "https://" + domain + "?os=Android&t=" + BuildInfo.SDK_SVN_REVISION;
        try {
            endpoint += "&z=" + ManifestMetaData.getMetaData(context, Constants.LABEL_ACCOUNT_ID);
        } catch (CleverTapMetaDataNotFoundException ignore) {
            // Ignore
        }

        // Don't attach ts if its handshake
        if (needsHandshake) {
            return endpoint;
        }

        currentRequestTimestamp = (int) (System.currentTimeMillis() / 1000);
        endpoint += "&ts=" + currentRequestTimestamp;

        return endpoint;
    }

    // timestamp helpers

    private static int getFirstRequestTimestamp(Context context) {
        return StorageHelper.getInt(context, KEY_FIRST_TS, 0);
    }

    private static int getLastRequestTimestamp(Context context) {
        return StorageHelper.getInt(context, KEY_LAST_TS, 0);
    }

    private static void setLastRequestTimestamp(Context context, int ts) {
        StorageHelper.putInt(context, KEY_LAST_TS, ts);
    }

    private static void setFirstRequestTimestampIfNeeded(Context context, int ts) {
        if (getFirstRequestTimestamp(context) > 0) return;
        StorageHelper.putInt(context, KEY_FIRST_TS, ts);
    }

    private static void clearLastRequestTimestamp(Context context) {
        StorageHelper.putInt(context, KEY_LAST_TS, 0);
    }

    private static void clearFirstRequestTimestampIfNeeded(Context context) {
        StorageHelper.putInt(context, KEY_FIRST_TS, 0);
    }

    /**
     * The ARP is additional request parameters, which must be sent once
     * received after any HTTP call. This is sort of a proxy for cookies.
     *
     * @return A JSON object containing the ARP key/values. Can be null.
     */
    private static JSONObject getARP(final Context context) {
        try {
            final SharedPreferences prefs = StorageHelper.getPreferences(context, NAMESPACE_ARP);
            final Map<String, ?> all = prefs.getAll();
            final Iterator<? extends Map.Entry<String, ?>> iter = all.entrySet().iterator();

            while (iter.hasNext()) {
                final Map.Entry<String, ?> kv = iter.next();
                final Object o = kv.getValue();
                if (o instanceof Number && ((Number) o).intValue() == -1) {
                    iter.remove();
                }
            }

            return new JSONObject(all);
        } catch (Throwable t) {
            Logger.logFine("Failed to construct ARP object", t);
            return null;
        }
    }

    // I/J handling

    private static long getI(Context context) {
        final SharedPreferences prefs = StorageHelper.getPreferences(context, NAMESPACE_IJ);
        return prefs.getLong(KEY_I, 0);
    }

    private static long getJ(Context context) {
        final SharedPreferences prefs = StorageHelper.getPreferences(context, NAMESPACE_IJ);
        return prefs.getLong(KEY_J, 0);
    }

    @SuppressLint("CommitPrefEdits")
    private static void setJ(Context context, long j) {
        final SharedPreferences prefs = StorageHelper.getPreferences(context, NAMESPACE_IJ);
        final SharedPreferences.Editor editor = prefs.edit();
        editor.putLong(KEY_J, j);
        StorageHelper.persist(editor);
    }

    @SuppressLint("CommitPrefEdits")
    private static void setI(Context context, long i) {
        final SharedPreferences prefs = StorageHelper.getPreferences(context, NAMESPACE_IJ);
        final SharedPreferences.Editor editor = prefs.edit();
        editor.putLong(KEY_I, i);
        StorageHelper.persist(editor);
    }

    @SuppressLint("CommitPrefEdits")
    private static void clearIJ(Context context) {
        final SharedPreferences prefs = StorageHelper.getPreferences(context, NAMESPACE_IJ);
        final SharedPreferences.Editor editor = prefs.edit();
        editor.clear();
        StorageHelper.persist(editor);
    }

    static void clearUserContext(final Context context) {
        clearIJ(context);
        _clearARP(context);
        clearFirstRequestTimestampIfNeeded(context);
        clearLastRequestTimestamp(context);
    }

    @SuppressLint("CommitPrefEdits")
    private static void _clearARP(Context context) {
        final SharedPreferences prefs = StorageHelper.getPreferences(context, NAMESPACE_ARP);
        final SharedPreferences.Editor editor = prefs.edit();
        editor.clear();
        StorageHelper.persist(editor);
    }

    @SuppressLint("CommitPrefEdits")
    private static void handleARPUpdate(final Context context, final JSONObject arp) {
        if (arp == null || arp.length() == 0) return;

        final SharedPreferences prefs = StorageHelper.getPreferences(context, NAMESPACE_ARP);
        final SharedPreferences.Editor editor = prefs.edit();

        final Iterator<String> keys = arp.keys();
        while (keys.hasNext()) {
            final String key = keys.next();
            try {
                final Object o = arp.get(key);
                if (o instanceof Number) {
                    final int update = ((Number) o).intValue();
                    editor.putInt(key, update);
                } else if (o instanceof String) {
                    if (((String) o).length() < 100) {
                        editor.putString(key, (String) o);
                    } else {
                        Logger.logFine("ARP update for key " + key + " rejected (string value too long)");
                    }
                } else if (o instanceof Boolean) {
                    editor.putBoolean(key, (Boolean) o);
                } else {
                    Logger.logFine("ARP update for key " + key + " rejected (invalid data type)");
                }
            } catch (JSONException e) {
                // Ignore
            }
        }
        StorageHelper.persist(editor);
    }
}
