/*
 * Author: Jude Pereira
 * Copyright (c) 2014
 */

package com.clevertap.android.sdk;

import android.content.Context;
import android.os.Bundle;
import android.telephony.TelephonyManager;
import android.util.DisplayMetrics;
import android.view.WindowManager;
import com.clevertap.android.sdk.exceptions.InvalidEventNameException;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

/**
 * Handles all push events.
 */
public final class EventHandler {
    private final Context context;

    /**
     * Please do not create an object of this explicitly.
     *
     * @param context The Android context
     * @see CleverTapAPI
     */
    EventHandler(Context context) {
        this.context = context;
    }

    private static void queueEvent(final Context context, final JSONObject event, final int eventType) {
        QueueManager.queueEvent(context, event, eventType);
    }

    private JSONObject getErrorObject(ValidationResult vr) {
        JSONObject error = new JSONObject();
        try {
            error.put("c", vr.getErrorCode());
            error.put("d", vr.getErrorDesc());
        } catch (JSONException e) {
            // Won't reach here
        }
        return error;
    }

    private void pushValidationResult(ValidationResult vr) {
        CleverTapAPI.pushValidationResult(vr);
    }

    /**
     * Pushes a basic event.
     *
     * @param eventName The name of the event
     */
    public void push(String eventName) {
        if (eventName == null || eventName.trim().equals(""))
            return;

        push(eventName, null);
    }

    /**
     * Push an event with a set of attribute pairs.
     *
     * @param eventName    The name of the event
     * @param eventActions A {@link HashMap}, with keys as strings, and values as {@link String},
     *                     {@link Integer}, {@link Long}, {@link Boolean}, {@link Float}, {@link Double},
     *                     {@link java.util.Date}, or {@link Character}
     */
    public void push(String eventName, Map<String, Object> eventActions) {

        if (eventName == null || eventName.equals(""))
            return;
        // Check for a restricted event name
        if (Validator.isRestrictedEventName(eventName)) {
            return;
        }

        if (eventActions == null) {
            eventActions = new HashMap<String, Object>();
        }

        JSONObject event = new JSONObject();
        try {
            // Validate
            ValidationResult vr = Validator.cleanEventName(eventName);

            // Check for an error
            if (vr.getErrorCode() != 0)
                event.put(Constants.ERROR_KEY, getErrorObject(vr));

            eventName = vr.getObject().toString();
            JSONObject actions = new JSONObject();
            for (String key : eventActions.keySet()) {
                Object value = eventActions.get(key);
                vr = Validator.cleanObjectKey(key);
                key = vr.getObject().toString();
                // Check for an error
                if (vr.getErrorCode() != 0)
                    event.put(Constants.ERROR_KEY, getErrorObject(vr));
                try {
                    vr = Validator.cleanObjectValue(value);
                } catch (IllegalArgumentException e) {
                    // The object was neither a String, Boolean, or any number primitives
                    ValidationResult error = new ValidationResult();
                    error.setErrorCode(512);
                    final String err = "For event \"" + eventName + "\": Property value for property " + key + " wasn't a primitive (" + value + ")";
                    error.setErrorDesc(err);
                    Logger.error(err);
                    pushValidationResult(error);
                    // Skip this record
                    continue;
                }
                value = vr.getObject();
                // Check for an error
                if (vr.getErrorCode() != 0)
                    event.put(Constants.ERROR_KEY, getErrorObject(vr));
                actions.put(key, value);
            }
            event.put("evtName", eventName);
            event.put("evtData", actions);
            queueEvent(context, event, Constants.RAISED_EVENT);
        } catch (Throwable t) {
            // We won't get here
        }
    }

    /**
     * Push an event which describes a purchase made.
     *
     * @param eventName     Has to be specified as "Charged". Anything other than this
     *                      will result in an {@link InvalidEventNameException} being thrown.
     * @param chargeDetails A {@link HashMap}, with keys as strings, and values as {@link String},
     *                      {@link Integer}, {@link Long}, {@link Boolean}, {@link Float}, {@link Double},
     *                      {@link java.util.Date}, or {@link Character}
     * @param items         An {@link ArrayList} which contains up to 15 {@link HashMap} objects,
     *                      where each HashMap object describes a particular item purchased
     * @throws InvalidEventNameException Thrown if the event name is not "Charged"
     */
    public void push(String eventName, HashMap<String, Object> chargeDetails,
                     ArrayList<HashMap<String, Object>> items)
            throws InvalidEventNameException {

        if (eventName == null || eventName.equals("")
                || chargeDetails == null || items == null)
            return;

        // Check for a restricted event name
        if (Validator.isRestrictedEventName(eventName)) {
            return;
        }

        if (items.size() > 15) {
            ValidationResult error = new ValidationResult();
            error.setErrorCode(522);
            error.setErrorDesc("Charged event contained more than 15 items.");
            Logger.error("Charged event contained more than 15 items.");
            pushValidationResult(error);
        }

        JSONObject evtData = new JSONObject();
        JSONObject chargedEvent = new JSONObject();
        ValidationResult vr;
        try {
            // This method is for only charged events
            if (!eventName.equals(CleverTapAPI.CHARGED_EVENT))
                throw new InvalidEventNameException("Not a charged event");

            for (String key : chargeDetails.keySet()) {
                Object value = chargeDetails.get(key);
                vr = Validator.cleanObjectKey(key);
                key = vr.getObject().toString();
                // Check for an error
                if (vr.getErrorCode() != 0)
                    chargedEvent.put(Constants.ERROR_KEY, getErrorObject(vr));

                try {
                    vr = Validator.cleanObjectValue(value);
                } catch (IllegalArgumentException e) {
                    // The object was neither a String, Boolean, or any number primitives
                    ValidationResult error = new ValidationResult();
                    error.setErrorCode(511);
                    final String err = "For event \"" + eventName + "\": Property value for property " + key + " wasn't a primitive (" + value + ")";
                    error.setErrorDesc(err);
                    pushValidationResult(error);
                    Logger.error(err);
                    // Skip this property
                    continue;
                }
                value = vr.getObject();
                // Check for an error
                if (vr.getErrorCode() != 0)
                    chargedEvent.put(Constants.ERROR_KEY, getErrorObject(vr));

                evtData.put(key, value);
            }

            JSONArray jsonItemsArray = new JSONArray();
            for (HashMap<String, Object> map : items) {
                JSONObject itemDetails = new JSONObject();
                for (String key : map.keySet()) {
                    Object value = map.get(key);
                    vr = Validator.cleanObjectKey(key);
                    key = vr.getObject().toString();
                    // Check for an error
                    if (vr.getErrorCode() != 0)
                        chargedEvent.put(Constants.ERROR_KEY, getErrorObject(vr));

                    try {
                        vr = Validator.cleanObjectValue(value);
                    } catch (IllegalArgumentException e) {
                        // The object was neither a String, Boolean, or any number primitives
                        ValidationResult error = new ValidationResult();
                        error.setErrorCode(511);
                        final String err = "An item's object value for key " + key + " wasn't a primitive (" + value + ")";
                        error.setErrorDesc(err);
                        Logger.error(err);

                        pushValidationResult(error);
                        // Skip this property
                        continue;
                    }
                    value = vr.getObject();
                    // Check for an error
                    if (vr.getErrorCode() != 0)
                        chargedEvent.put(Constants.ERROR_KEY, getErrorObject(vr));
                    itemDetails.put(key, value);
                }
                jsonItemsArray.put(itemDetails);
            }
            evtData.put("Items", jsonItemsArray);

            chargedEvent.put("evtName", CleverTapAPI.CHARGED_EVENT);
            chargedEvent.put("evtData", evtData);
            queueEvent(context, chargedEvent, Constants.RAISED_EVENT);
        } catch (Throwable t) {
            // We won't get here
        }
    }

    /**
     * Pushes the notification details to CleverTap.
     *
     * @param extras The {@link Bundle} object that contains the
     *               notification details
     */
    public void pushNotificationEvent(final Bundle extras) {
        if (extras != null && !extras.isEmpty()
                && extras.get(CleverTapAPI.NOTIFICATION_TAG) != null) {
            if (extras.containsKey(Constants.INAPP_PREVIEW_PUSH_PAYLOAD_KEY)) {
                CleverTapAPI.pendingInappRunnable = new Runnable() {
                    @Override
                    public void run() {
                        try {
                            Logger.logFine("Received in-app via push payload: " + extras.getString(Constants.INAPP_PREVIEW_PUSH_PAYLOAD_KEY));
                            InAppFCManager.destroySession();
                            JSONObject r = new JSONObject();
                            JSONArray inappNotifs = new JSONArray();
                            r.put(Constants.INAPP_JSON_RESPONSE_KEY, inappNotifs);
                            inappNotifs.put(new JSONObject(extras.getString(Constants.INAPP_PREVIEW_PUSH_PAYLOAD_KEY)));
                            InAppManager.processResponseAsync(r, context);
                        } catch (Throwable t) {
                            Logger.error("Failed to display inapp notification from push notification payload", t);
                        }
                    }
                };
                return;
            }
            JSONObject event = new JSONObject();
            JSONObject notif = new JSONObject();
            try {
                for (String x : extras.keySet()) {
                    if (!x.startsWith(Constants.WZRK_PREFIX))
                        continue;
                    Object value = extras.get(x);
                    notif.put(x, value);
                }

                event.put("evtName", Constants.NOTIFICATION_CLICKED_EVENT_NAME);
                event.put("evtData", notif);
                queueEvent(context, event, Constants.RAISED_EVENT);

                try {
                    SessionHandler.setWzrkParams(getWzrkFields(extras));
                } catch (Throwable t) {
                    // no-op
                }
            } catch (Throwable t) {
                // We won't get here
            }
        }
    }

    private static JSONObject getWzrkFields(Bundle root) throws JSONException {
        final JSONObject fields = new JSONObject();
        for (String s : root.keySet()) {
            final Object o = root.get(s);
            if (o instanceof Bundle) {
                final JSONObject wzrkFields = getWzrkFields((Bundle) o);
                final Iterator<String> keys = wzrkFields.keys();
                while (keys.hasNext()) {
                    final String k = keys.next();
                    fields.put(k, wzrkFields.get(k));
                }
            } else if (s.startsWith(Constants.WZRK_PREFIX)) {
                fields.put(s, root.get(s));
            }
        }

        return fields;
    }

    /**
     * Raises the Notification Clicked event, if {@param clicked} is true,
     * otherwise the Notification Viewed event, if {@param clicked} is false.
     *
     * @param clicked    Whether or not this notification was clicked
     * @param data       The data to be attached as the event data
     * @param customData Additional data such as form input to to be added to the event data
     */
    void pushInAppNotificationStateEvent(boolean clicked, Bundle data, Bundle customData) {
        JSONObject event = new JSONObject();
        try {
            JSONObject notif = getWzrkFields(data);

            if (customData != null) {
                for (String x : customData.keySet()) {

                    Object value = customData.get(x);
                    if (value != null) notif.put(x, value);
                }
            }

            if (clicked) {
                try {
                    SessionHandler.setWzrkParams(notif);
                } catch (Throwable t) {
                    // no-op
                }
                event.put("evtName", Constants.NOTIFICATION_CLICKED_EVENT_NAME);
            } else {
                event.put("evtName", Constants.NOTIFICATION_VIEWED_EVENT_NAME);
            }

            event.put("evtData", notif);
            queueEvent(context, event, Constants.RAISED_EVENT);
        } catch (Throwable ignored) {
            // We won't get here
        }
    }


    /**
     * Sends the basic device details(whatever is available).
     */
    void pushDeviceDetails() {
        pushDeviceDetailsWithExtras(null);
    }

    void pushDeviceDetailsWithExtras(JSONObject extras) {
        // Determine the screen dimensions
        WindowManager wm = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
        DisplayMetrics dm = new DisplayMetrics();
        wm.getDefaultDisplay().getMetrics(dm);
        // Calculate the width in inches
        double rWidth = dm.widthPixels / dm.xdpi;
        // Calculate the height in inches
        double rHeight = dm.heightPixels / dm.ydpi;
        JSONObject jsonObject = new JSONObject();
        try {
            jsonObject.put("OS", "Android");
            jsonObject.put("wdt", toTwoPlaces(rWidth));
            jsonObject.put("hgt", toTwoPlaces(rHeight));

            String cc = getCountryCode();
            if (cc != null && !cc.equals(""))
                jsonObject.put("cc", cc);


            // Add the extras
            if (extras != null && extras.length() > 0) {
                Iterator keys = extras.keys();
                while (keys.hasNext()) {
                    try {
                        String key = (String) keys.next();
                        jsonObject.put(key, extras.getString(key));
                    } catch (ClassCastException ignore) {
                        // Really won't get here
                    }
                }
            }
            queueEvent(context, jsonObject, Constants.PAGE_EVENT);
        } catch (Throwable t) {
            // We won't get here
        }
    }

    private double toTwoPlaces(double n) {
        double result = n * 100;
        result = Math.round(result);
        result = result / 100;
        return result;
    }

    private String getCountryCode() {
        try {
            TelephonyManager tm = (TelephonyManager) context.getSystemService(Context.TELEPHONY_SERVICE);
            return tm.getSimCountryIso();
        } catch (Throwable ignore) {
            return "";
        }
    }

    public EventDetail getDetails(String event) {
        return LocalDataStore.getEventDetail(context, event);
    }

    public Map<String, EventDetail> getHistory() {
        return LocalDataStore.getEventHistory(context);
    }

    public int getFirstTime(String event) {
        EventDetail eventDetail = LocalDataStore.getEventDetail(context, event);
        if (eventDetail != null) return eventDetail.getFirstTime();

        return -1;
    }

    public int getLastTime(String event) {
        EventDetail eventDetail = LocalDataStore.getEventDetail(context, event);
        if (eventDetail != null) return eventDetail.getLastTime();

        return -1;
    }

    public int getCount(String event) {
        EventDetail eventDetail = LocalDataStore.getEventDetail(context, event);
        if (eventDetail != null) return eventDetail.getCount();

        return -1;
    }

    /**
     * Internally records an "Error Occurred" event, which can be viewed in the dashboard.
     *
     * @param errorMessage The error message
     * @param errorCode    The error code
     */
    public void pushError(final String errorMessage, final int errorCode) {
        final HashMap<String, Object> props = new HashMap<String, Object>();
        props.put("Error Message", errorMessage);
        props.put("Error Code", errorCode);

        try {
            final String activityName = CleverTapAPI.getCurrentActivityName();
            if (activityName != null) {
                props.put("Location", activityName);
            } else {
                props.put("Location", "Unknown");
            }
        } catch (Throwable t) {
            // Ignore
            props.put("Location", "Unknown");
        }

        push("Error Occurred", props);
    }
}
