/*
 * Author: Jude Pereira
 * Copyright (c) 2014
 */

package com.clevertap.android.sdk;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.app.Notification;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.content.pm.PackageInfo;
import android.graphics.Bitmap;
import android.location.Location;
import android.location.LocationManager;
import android.media.RingtoneManager;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.support.v4.app.NotificationCompat;
import com.clevertap.android.sdk.exceptions.CleverTapMetaDataNotFoundException;
import com.clevertap.android.sdk.exceptions.CleverTapPermissionsNotSatisfied;
import org.json.JSONException;
import org.json.JSONObject;

import java.lang.ref.WeakReference;
import java.net.URLDecoder;
import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * Manipulates the CleverTap SDK.
 */
@SuppressWarnings("WeakerAccess")
public final class CleverTapAPI {
    // Static fields
    public static final String CHARGED_EVENT = "Charged";
    public static final String NOTIFICATION_TAG = "wzrk_pn";
    private static final String CACHED_GUIDS_KEY = "cachedGUIDsKey";
    private static final String MULTI_USER_PREFIX = "mt_";
    private static final Handler handlerUsingMainLooper = new Handler(Looper.getMainLooper());
    private static final ExecutorService es = Executors.newFixedThreadPool(1);
    private static int debugLevel = 0;
    private static boolean appForeground = false;
    private static CleverTapAPI ourInstance = null;
    static Runnable pendingInappRunnable = null;
    static String localAccountID = null, localToken = null;
    private static int lastLocationPingTime = 0;
    private InAppNotificationListener inAppNotificationListener;

    static int activityCount = 0;

    private static ArrayList<ValidationResult> pendingValidationResults = new ArrayList<ValidationResult>();
    private static final Boolean pendingValidationResultsLock = true;

    private static String parseInstallationId = null;

    private static boolean appLaunchPushed = false;

    private static final Object appLaunchPushedLock = new Object();

    static boolean isAppLaunchPushed() {
        synchronized (appLaunchPushedLock) {
            return appLaunchPushed;
        }
    }

    static void setAppLaunchPushed(boolean pushed) {
        synchronized (appLaunchPushedLock) {
            appLaunchPushed = pushed;
        }
    }

    private static WeakReference<Activity> currentActivity;

    static Activity getCurrentActivity() {
        return (currentActivity == null) ? null : currentActivity.get();
    }

    static void setCurrentActivity(Activity activity) {
        currentActivity = (activity == null) ? null : new WeakReference<Activity>(activity);
    }

    static String getCurrentActivityName() {
        Activity current = getCurrentActivity();
        return (current != null) ? current.getLocalClassName() : null;
    }

    static String getParseInstallationId() {
        return parseInstallationId;
    }

    private static String getGoogleAdID() {
        return DeviceInfo.getGoogleAdID();
    }

    private static boolean isLimitAdTrackingEnabled() {
        return DeviceInfo.isLimitAdTrackingEnabled();
    }

    // Non static fields
    public final EventHandler event;
    public final DataHandler data;
    public final ProfileHandler profile;
    public final SessionHandler session;
    private Runnable sessionTimeoutRunnable = null;
    private final Context context;
    private long appLastSeen;
    private SyncListener syncListener = null;
    private Location locationFromUser = null;

    Context getContext() {
        return context;
    }

    public void setInAppNotificationListener(InAppNotificationListener inAppNotificationListener) {
        this.inAppNotificationListener = inAppNotificationListener;
    }

    public InAppNotificationListener getInAppNotificationListener() {
        return inAppNotificationListener;
    }

    static ValidationResult popValidationResult() {
        // really a shift
        ValidationResult vr = null;

        synchronized (pendingValidationResultsLock) {
            try {
                if (!pendingValidationResults.isEmpty()) {
                    vr = pendingValidationResults.remove(0);
                }
            } catch (Exception e) {
                // no-op
            }
        }
        return vr;
    }

    static void pushValidationResult(ValidationResult vr) {
        synchronized (pendingValidationResultsLock) {
            try {
                int len = pendingValidationResults.size();
                if (len > 50) {
                    ArrayList<ValidationResult> trimmed = new ArrayList<ValidationResult>();
                    // Trim down the list to 40, so that this loop needn't run for the next 10 events
                    // Hence, skip the first 10 elements
                    for (int i = 10; i < len; i++)
                        trimmed.add(pendingValidationResults.get(i));
                    trimmed.add(vr);
                    pendingValidationResults = trimmed;
                } else {
                    pendingValidationResults.add(vr);
                }
            } catch (Exception e) {
                // no-op
            }
        }
    }

    /**
     * Returns the generic handler object which is used to post
     * runnables. The returned value will never be null.
     *
     * @return The static generic handler
     * @see Handler
     */
    static Handler getHandlerUsingMainLooper() {
        return handlerUsingMainLooper;
    }

    private static long EXECUTOR_THREAD_ID = 0;

    /**
     * Use this to safely post a runnable to the async handler.
     * It adds try/catch blocks around the runnable and the handler itself.
     */
    @SuppressWarnings("UnusedParameters")
    static void postAsyncSafely(final String name, final Runnable runnable) {
        try {
            final boolean executeSync = Thread.currentThread().getId() == EXECUTOR_THREAD_ID;

            if (executeSync) {
                runnable.run();
            } else {
                es.submit(new Runnable() {
                    @Override
                    public void run() {
                        EXECUTOR_THREAD_ID = Thread.currentThread().getId();
                        try {
                            //Logger.logFine("Executor service: Starting task - " + name);
                            //final long start = System.currentTimeMillis();
                            runnable.run();
                            //final long time = System.currentTimeMillis() - start;
                            //Logger.logFine("Executor service: Task completed successfully in " + time + "ms (" + name + ")");
                        } catch (Throwable t) {
                            Logger.logFine("Executor service: Failed to complete the scheduled task", t);
                        }
                    }
                });
            }
        } catch (Throwable t) {
            Logger.logFine("Failed to submit task to the executor service", t);
        }
    }

    /**
     * Returns whether or not the app is in the foreground.
     *
     * @return The foreground status
     */
    static boolean isAppForeground() {
        return appForeground;
    }

    /**
     * This method is used internally.
     */
    public static void setAppForeground(boolean appForeground) {
        CleverTapAPI.appForeground = appForeground;
    }

    public void setSyncListener(SyncListener syncListener) {
        this.syncListener = syncListener;
    }

    public SyncListener getSyncListener() {
        return syncListener;
    }

    private static void handleParse(final Context context) {
        try {
            if (ManifestMetaData.getMetaData(context, Constants.LABEL_PARSE) != null) {
                retrieveIdentifierForParse(context);
            }
        } catch (Throwable t) {
            // Ignore
        }
    }

    private static void retrieveIdentifierForParse(final Context context) {
        postAsyncSafely("parse-id-retriever", new Runnable() {
            @Override
            public void run() {

                // Try to get Parse's installation ID
                String installationId = ParseUtils.getParseInstallationId(context);

                if (installationId != null) {
                    try {
                        installationId = "_p" + installationId.replace("-", "");
                        Logger.logFine("Setting Parse installation ID " + installationId);
                        parseInstallationId = installationId;
                    } catch (Exception e) {
                        Logger.logFine("Error reading Parse installation ID: " + e.toString());
                    }
                }
            }
        });
    }

    public static void changeCredentials(String accountID, String token) {
        localAccountID = accountID;
        localToken = token;
    }

    /**
     * Returns an instance of the CleverTap SDK.
     *
     * @param context The Android context
     * @return The {@link CleverTapAPI} object
     * @throws CleverTapMetaDataNotFoundException
     * @throws CleverTapPermissionsNotSatisfied
     */
    public static synchronized CleverTapAPI getInstance(final Context context)
            throws CleverTapMetaDataNotFoundException, CleverTapPermissionsNotSatisfied {
        if (ourInstance == null && context != null) {

            // wake up the data store - loads the local profile - good to do this first
            LocalDataStore.initializeWithContext(context);

            // handles the guid initialization logic
            DeviceInfo.initializeWithContext(context);

            // check to see if we need to look for the Parse identifier
            handleParse(context);

            ourInstance = new CleverTapAPI(context.getApplicationContext());
        }
        return ourInstance;
    }

    /**
     * @return Application's {@code SharedPreferences}.
     */
    private SharedPreferences getPreferences(Context context) {
        return StorageHelper.getPreferences(context);
    }

    private void testPermissions(final Context context) throws CleverTapMetaDataNotFoundException, CleverTapPermissionsNotSatisfied {
        DeviceInfo.testPermission(context, "android.permission.INTERNET");
        // Test for account ID, and token
        ManifestMetaData.getMetaData(context, Constants.LABEL_ACCOUNT_ID);
        ManifestMetaData.getMetaData(context, Constants.LABEL_TOKEN);
    }

    private CleverTapAPI(final Context context) throws CleverTapMetaDataNotFoundException, CleverTapPermissionsNotSatisfied {
        this.context = context;
        event = new EventHandler(context);
        profile = new ProfileHandler(context);
        data = new DataHandler(context);
        session = new SessionHandler(context);

        testPermissions(context);

        Logger.logFine("New instance of CleverTapAPI created");

    }

    static String getAccountID(final Context context) {
        if (localAccountID == null) {
            try {
                localAccountID = ManifestMetaData.getMetaData(context, Constants.LABEL_ACCOUNT_ID);
            } catch (Exception e) {
                // no-op
                Logger.logFine("Error reading accountID: " + e.toString());
            }
        }

        return localAccountID;
    }

    // multi-user handling

    private JSONObject getCachedGUIDs() {
        JSONObject cache = null;
        String json = StorageHelper.getString(context, CACHED_GUIDS_KEY, null);
        if (json != null) {
            try {
                cache = new JSONObject(json);
            } catch (Throwable t) {
                // no-op
                Logger.logFine("Error reading guid cache: " + t.toString());
            }
        }

        return (cache != null) ? cache : new JSONObject();
    }

    private void setCachedGUIDs(JSONObject cachedGUIDs) {
        if (cachedGUIDs == null) return;
        try {
            StorageHelper.putString(context, CACHED_GUIDS_KEY, cachedGUIDs.toString());
        } catch (Throwable t) {
            // no-op
            Logger.logFine("Error persisting guid cache: " + t.toString());
        }
    }

    private String getGUIDForIdentifier(String key, String identifier) {
        if (key == null || identifier == null) return null;

        String cacheKey = key + "_" + identifier;
        JSONObject cache = getCachedGUIDs();
        try {
            return cache.getString(cacheKey);
        } catch (Throwable t) {
            Logger.logFine("Error reading guid cache: " + t.toString());
            return null;
        }
    }

    void cacheGUIDForIdentifier(String guid, String key, String identifier) {
        if (guid == null || key == null || identifier == null) return;

        String cacheKey = key + "_" + identifier;
        JSONObject cache = getCachedGUIDs();
        try {
            cache.put(cacheKey, guid);
            setCachedGUIDs(cache);
        } catch (Throwable t) {
            // no-op
            Logger.logFine("Error caching guid: " + t.toString());
        }
    }

    private boolean isAnonymousDevice() {
        JSONObject cachedGUIDs = getCachedGUIDs();
        return cachedGUIDs.length() <= 0;
    }

    private boolean deviceIsMultiUser() {
        JSONObject cachedGUIDs = getCachedGUIDs();
        return cachedGUIDs.length() > 1;
    }

    private static String processingUserLoginIdentifier = null;
    private static final Boolean processingUserLoginLock = true;

    private boolean isProcessUserLoginWithIdentifier(String identifier) {
        synchronized (processingUserLoginLock) {
            return processingUserLoginIdentifier!= null && processingUserLoginIdentifier.equals(identifier);
        }
    }

    /**
     * Creates a separate and distinct user profile identified by one or more of Identity, Email, FBID or GPID values,
     * and populated with the key-values included in the profile map argument.
     * <p>
     * If your app is used by multiple users, you can use this method to assign them each a unique profile to track them separately.
     * <p>
     * If instead you wish to assign multiple Identity, Email, FBID and/or GPID values to the same user profile,
     * use profile.push rather than this method.
     * <p>
     * If none of Identity, Email, FBID or GPID is included in the profile map,
     * all profile map values will be associated with the current user profile.
     * <p>
     * When initially installed on this device, your app is assigned an "anonymous" profile.
     * The first time you identify a user on this device (whether via onUserLogin or profilePush),
     * the "anonymous" history on the device will be associated with the newly identified user.
     * <p>
     * Then, use this method to switch between subsequent separate identified users.
     * <p>
     * Please note that switching from one identified user to another is a costly operation
     * in that the current session for the previous user is automatically closed
     * and data relating to the old user removed, and a new session is started
     * for the new user and data for that user refreshed via a network call to CleverTap.
     * In addition, any global frequency caps are reset as part of the switch.
     *
     * @param profile The map keyed by the type of identity, with the value as the identity
     */

    public void onUserLogin(final Map<String, Object> profile) {
        if (profile == null) return;

        try {
            final String currentGUID = getCleverTapID();
            if (currentGUID == null) return;

            boolean haveIdentifier = false;
            String cachedGUID = null;

            // check for valid identifier keys
            // use the first one we find
            for (String key : profile.keySet()) {
                Object value = profile.get(key);
                if (Constants.PROFILE_IDENTIFIER_KEYS.contains(key)) {
                    try {
                        String identifier = value.toString();
                        if (identifier != null && identifier.length() > 0) {
                            haveIdentifier = true;
                            cachedGUID = getGUIDForIdentifier(key, identifier);
                            if (cachedGUID != null) break;
                        }
                    } catch (Throwable t) {
                        // no-op
                    }
                }
            }

            // if no valid identifier provided or there are no identified users on the device; just push on the current profile
            if (!haveIdentifier || isAnonymousDevice()) {
                Logger.logFine("onUserLogin: no identifier provided or device is anonymous, pushing on current user profile");
                this.profile.push(profile);
                return;
            }

            // if identifier maps to current guid, push on current profile
            if (cachedGUID != null && cachedGUID.equals(currentGUID)) {
                Logger.logFine("onUserLogin: " + profile.toString() + " maps to current device id " + currentGUID + " pushing on current profile");
                this.profile.push(profile);
                return;
            }

            // stringify profile to use as dupe blocker
            String profileToString = profile.toString();

            // as processing happens async block concurrent onUserLogin requests with the same profile, as our cache is set async
            if (isProcessUserLoginWithIdentifier(profileToString)) {
                Logger.logFine("Already processing onUserLogin for " + profileToString);
                return;
            }

            // create new guid if necessary and reset
            // block any concurrent onUserLogin call for the same profile
            synchronized (processingUserLoginLock) {
                processingUserLoginIdentifier = profileToString;
            }

            Logger.logFine("onUserLogin: queuing reset profile for " + profileToString
                    + " with Cached GUID " + ((cachedGUID != null) ? cachedGUID : "NULL"));

            final ProfileHandler profileHandler = this.profile;

            final String guid = cachedGUID;

            postAsyncSafely("resetProfile", new Runnable() {
                @Override
                public void run() {
                    try {

                        // unregister the device token on the current user
                        forcePushDeviceToken(false);

                        // try and flush and then reset the queues
                        CommsManager.flushQueueSync(context);
                        QueueManager.clearQueues(context);

                        // clear out the old data
                        LocalDataStore.changeUser(context);
                        InAppFCManager.changeUser(context);
                        activityCount = 1;
                        forceDestroySession();

                        // either force restore the cached GUID or generate a new one
                        if (guid != null) {
                            DeviceInfo.forceUpdateDeviceId(guid);
                        } else {
                            DeviceInfo.forceNewDeviceID();
                        }

                        forcePushAppLaunchedEvent("onLoginUser");
                        profileHandler.push(profile);
                        forcePushDeviceToken(true);
                        synchronized (processingUserLoginLock) {
                            processingUserLoginIdentifier = null;
                        }
                    } catch (Throwable t) {
                        Logger.error("Reset Profile error", t);
                    }
                }
            });

        } catch (Throwable t) {
            Logger.error("onUserLogin failed", t);
        }
    }

    static void notifyUserProfileInitialized(String deviceID) {
        if (ourInstance == null) return;
        ourInstance._notifyProfileInitialized(deviceID);
    }

    private void _notifyProfileInitialized(String deviceID) {
        deviceID = (deviceID != null) ? deviceID : getCleverTapID();

        if (deviceID == null) return;

        final SyncListener sl;
        try {
            sl = getSyncListener();
            if (sl != null) {
                sl.profileDidInitialize(deviceID);
            }
        } catch (Throwable t) {
            // Ignore
        }
    }

    @SuppressLint("CommitPrefEdits")
    private void pushDailyEventsAsync() {
        postAsyncSafely("CleverTapAPI#pushDailyEventsAsync", new Runnable() {
            @Override
            public void run() {
                try {
                    Logger.logFine("Queuing daily events");
                    profile.pushBasicProfile(null);
                } catch (Throwable t) {
                    Logger.error("Daily profile sync failed", t);
                }
            }
        });
    }

    /**
     * Enables or disables debugging. If enabled, see debug messages in Android's logcat utility.
     * Debug messages are tagged as CleverTap.
     *
     * @param level Can be one of the following - 0(disables all debugging),
     *              1(show some debug output)
     */
    public static void setDebugLevel(int level) {
        debugLevel = level;
    }

    public static int getDebugLevel() {
        return debugLevel;
    }

    /**
     * Tells CleverTap that the current Android activity was paused. Add a call to this
     * method in the onPause() method of every activity.
     *
     * @param activity The calling activity
     */
    public void activityPaused(Activity activity) {
        setAppForeground(false);
        // Just update it
        setCurrentActivity(activity);
        appLastSeen = System.currentTimeMillis();
        // Create the runnable, if it is null
        if (sessionTimeoutRunnable == null)
            sessionTimeoutRunnable = new Runnable() {
                @Override
                public void run() {
                    checkTimeoutSession();
                }
            };
        // Remove any existing session timeout runnable object
        getHandlerUsingMainLooper().removeCallbacks(sessionTimeoutRunnable);
        getHandlerUsingMainLooper().postDelayed(sessionTimeoutRunnable,
                Constants.SESSION_LENGTH_MINS * 60 * 1000);
        Logger.logFine("Foreground activity gone to background");
    }


    private static HashSet<String> inappActivityExclude = null;

    /*package*/ boolean canShowInAppOnActivity() {
        updateBlacklistedActivitySet();

        for (String blacklistedActivity : inappActivityExclude) {
            String currentActivityName = getCurrentActivityName();
            if (currentActivityName != null && currentActivityName.contains(blacklistedActivity)) {
                return false;
            }
        }

        return true;
    }

    private void updateBlacklistedActivitySet() {
        if (inappActivityExclude == null) {
            inappActivityExclude = new HashSet<String>();
            try {
                String activities = ManifestMetaData.getMetaData(context, Constants.LABEL_INAPP_EXCLUDE);
                if (activities != null) {
                    String[] split = activities.split(",");
                    for (String a : split) {
                        inappActivityExclude.add(a.trim());
                    }
                }
            } catch (Throwable t) {
                // Ignore
            }
            Logger.log("In-app notifications will not be shown on " + Arrays.toString(inappActivityExclude.toArray()));
        }
    }

    /**
     * Notifies CleverTap that a new activity was created. Add a call to this method
     * in the onCreate() method of every activity.
     */
    private void notifyActivityChanged(Activity activity) {
        setCurrentActivity(activity);
        activityCount++;

        //noinspection ConstantConditions
        if (activity != null) {
            Logger.logFine("Activity changed: " + activity.getLocalClassName());
        }

        if (Constants.ENABLE_INAPP) {
            // We MUST loop through the set and do a contains on each and every entry there
            final boolean canShow = canShowInAppOnActivity();

            if (canShow) {
                if (pendingInappRunnable != null) {
                    Logger.logFine("Found a pending inapp runnable. Scheduling it");
                    getHandlerUsingMainLooper().postDelayed(pendingInappRunnable, 200);
                    pendingInappRunnable = null;
                } else {
                    // Show an in-app notification, if available
                    InAppManager.showNotificationIfAvailable(context);
                }
            } else {
                Logger.log("In-app notifications will not be shown for this activity ("
                        + (activity != null ? activity.getLocalClassName() : "") + ")");
            }
        }

        pushDailyEventsAsync();

        event.pushDeviceDetails();
        SessionManager.activityChanged(getCurrentActivityName());
    }

    /**
     * Tells CleverTap that the current Android activity was resumed. Add a call to this
     * method in the onResume() method of every activity.
     *
     * @param activity The calling activity
     */
    public void activityResumed(Activity activity) {
        setAppForeground(true);
        Activity current = getCurrentActivity();
        boolean newLaunch = current == null;
        // Check if this is a different activity from the last one
        String currentActivityName = getCurrentActivityName();
        if (currentActivityName == null
                || !currentActivityName.equals(activity.getLocalClassName()))
            notifyActivityChanged(activity);

        Logger.logFine("Background activity in foreground");

        // Initialize push and start a lazy App Launched
        if (newLaunch) {
            initPush();

            getHandlerUsingMainLooper().postDelayed(new Runnable() {
                @Override
                public void run() {
                    // Raise the App Launched event
                    pushAppLaunchedEvent("delayed generic handler");
                }
            }, 500);
        }
    }

    /*package*/ JSONObject getAppLaunchedFields() {
        try {
            final JSONObject evtData = new JSONObject();
            PackageInfo pInfo = context.getPackageManager().getPackageInfo(context.getPackageName(), 0);
            evtData.put("Build", pInfo.versionCode + "");
            evtData.put("Version", pInfo.versionName);
            evtData.put("OS Version", android.os.Build.VERSION.RELEASE);
            evtData.put("SDK Version", BuildInfo.SDK_SVN_REVISION);

            if (locationFromUser != null) {
                evtData.put("Latitude", locationFromUser.getLatitude());
                evtData.put("Longitude", locationFromUser.getLongitude());
            }

            // send up parseInstallationId
            if (parseInstallationId != null) {
                evtData.put("prg", parseInstallationId);
            }

            // send up googleAdID
            if (getGoogleAdID() != null) {
                String baseAdIDKey = "GoogleAdID";
                String adIDKey = deviceIsMultiUser() ? MULTI_USER_PREFIX + baseAdIDKey : baseAdIDKey;
                evtData.put(adIDKey, getGoogleAdID());
                evtData.put("GoogleAdIDLimit", isLimitAdTrackingEnabled());
            }

            try {
                // Device data
                String make = Build.MANUFACTURER;
                String model = Build.MODEL;
                model = model.replace(make, "");
                evtData.put("Make", make.trim());
                evtData.put("Model", model.trim());
            } catch (Throwable t) {
                // Ignore
            }
            return evtData;
        } catch (Throwable t) {
            Logger.logFine("Failed to construct App Launched event", t);
            return new JSONObject();
        }
    }

    // SessionManager/session management

    private void checkTimeoutSession() {
        long now = System.currentTimeMillis();
        if (!isAppForeground()
                && (now - appLastSeen) > Constants.SESSION_LENGTH_MINS * 60 * 1000) {
            Logger.logFine("Session Timed Out");
            forceDestroySession();
            setCurrentActivity(null);
        }
    }

    private void forceDestroySession() {
        setAppLaunchPushed(false);
        SessionManager.destroySession();
    }

    private void forcePushAppLaunchedEvent(String source) {
        setAppLaunchPushed(false);
        pushAppLaunchedEvent(source);
    }

    synchronized void pushAppLaunchedEvent(String source) {
        if (isAppLaunchPushed()) {
            Logger.logFine("App Launched has already been triggered. Will not trigger it; source = " + source);
            return;
        } else {
            Logger.logFine("Firing App Launched event; source = " + source);
        }
        setAppLaunchPushed(true);
        JSONObject event = new JSONObject();
        try {
            event.put("evtName", Constants.APP_LAUNCHED_EVENT);
            event.put("evtData", getAppLaunchedFields());
        } catch (Throwable t) {
            // We won't get here
        }
        QueueManager.queueEvent(context, event, Constants.RAISED_EVENT);
    }

    /**
     * Sends all the events in the event queue.
     */
    public void flush() {
        CommsManager.flushQueueAsync(context);
    }


    // Push Handling

    private static ArrayList<PushType> enabledPushTypes = null;

    private void initPush() {
        enabledPushTypes = DeviceInfo.getEnabledPushTypes();
        if (enabledPushTypes == null) return;
        for (PushType pushType : enabledPushTypes) {
            switch (pushType) {
                case GCM:
                    GcmManager.initializeWithContext(context);
                    break;
                case FCM:
                    FcmManager.initializeWithContext(context);
                    break;
                default:
                    //no-op
                    break;
            }
        }
    }

    /**
     * push the device token outside of the normal course
     */
    private void forcePushDeviceToken(final boolean register) {
        pushDeviceToken(register, true);
    }

    private void pushDeviceToken(final boolean register, final boolean force) {
        if (enabledPushTypes == null) return;
        for (PushType pushType : enabledPushTypes) {
            switch (pushType) {
                case GCM:
                    GcmManager.pushDeviceToken(null, register, force);
                    break;
                case FCM:
                    FcmManager.pushDeviceToken(null, register, force);
                    break;
                default:
                    //no-op
                    break;
            }
        }
    }

    /**
     * Checks whether this notification is from CleverTap.
     *
     * @param extras The payload from the GCM intent
     * @return See {@link NotificationInfo}
     */
    public static NotificationInfo getNotificationInfo(final Bundle extras) {
        if (extras == null) return new NotificationInfo(false, false);

        boolean fromCleverTap = extras.containsKey(NOTIFICATION_TAG);
        boolean shouldRender = fromCleverTap && extras.containsKey("nm");
        return new NotificationInfo(fromCleverTap, shouldRender);
    }

    /**
     * Launches an asynchronous task to download the notification icon from CleverTap,
     * and create the Android notification.
     * <p/>
     * If your app is using CleverTap SDK's built in GCM message handling,
     * this method does not need to be called explicitly.
     * <p/>
     * Use this method when implementing your own GCM handling mechanism. Refer to the
     * SDK documentation for usage scenarios and examples.
     *
     * @param context A reference to an Android context
     * @param extras  The {@link Bundle} object received by the broadcast receiver
     */
    public static void createNotification(final Context context, final Bundle extras) {
        //noinspection ConstantConditions
        if (extras == null || extras.get(NOTIFICATION_TAG) == null) {
            return;
        }

        try {
            postAsyncSafely("CleverTapAPI#createNotification", new Runnable() {
                @Override
                public void run() {
                    try {
                        // Check if this is a test notification
                        if (extras.containsKey(Constants.DEBUG_KEY)
                                && "y".equals(extras.getString(Constants.DEBUG_KEY))) {
                            int r = (int) (Math.random() * 10);
                            if (r != 8) {
                                // Discard acknowledging this notif
                                return;
                            }
                            JSONObject event = new JSONObject();
                            try {
                                JSONObject actions = new JSONObject();
                                for (String x : extras.keySet()) {
                                    Object value = extras.get(x);
                                    actions.put(x, value);
                                }
                                event.put("evtName", "wzrk_d");
                                event.put("evtData", actions);
                                QueueManager.queueEvent(context, event, Constants.RAISED_EVENT);
                            } catch (JSONException ignored) {
                                // Won't happen
                            }
                            // Drop further processing
                            return;
                        }
                        String notifTitle = extras.getString("nt");
                        // If not present, set it to the app name
                        notifTitle = (notifTitle != null) ? notifTitle : context.getApplicationInfo().name;
                        String notifMessage = extras.getString("nm");
                        if (notifMessage == null || notifMessage.isEmpty()) {
                            // What else is there to show then?
                            return;
                        }
                        String icoPath = extras.getString("ico");
                        Intent launchIntent;

                        if (extras.containsKey(Constants.DEEP_LINK_KEY)) {
                            launchIntent = new Intent(Intent.ACTION_VIEW, Uri.parse(extras.getString(Constants.DEEP_LINK_KEY)));
                        } else {
                            launchIntent = context.getPackageManager().getLaunchIntentForPackage(context.getPackageName());
                        }

                        PendingIntent pIntent;

                        // Take all the properties from the notif and add it to the intent
                        launchIntent.putExtras(extras);
                        launchIntent.setFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP | Intent.FLAG_ACTIVITY_SINGLE_TOP);
                        pIntent = PendingIntent.getActivity(context, (int) System.currentTimeMillis(),
                                launchIntent, PendingIntent.FLAG_UPDATE_CURRENT);

                        NotificationCompat.Style style;
                        String bigPictureUrl = extras.getString("wzrk_bp");
                        if (bigPictureUrl != null && bigPictureUrl.startsWith("http")) {
                            try {
                                Bitmap bpMap = Utils.getNotificationBitmap(bigPictureUrl, false, context);

                                //noinspection ConstantConditions
                                if (bpMap == null) throw new Exception("Failed to fetch big picture!");

                                style = new NotificationCompat.BigPictureStyle()
                                        .setSummaryText(notifMessage)
                                        .bigPicture(bpMap);
                            } catch (Throwable t) {
                                style = new NotificationCompat.BigTextStyle()
                                        .bigText(notifMessage);
                                Logger.error("Falling back to big text notification, couldn't fetch big picture", t);
                            }
                        } else {
                            style = new NotificationCompat.BigTextStyle()
                                    .bigText(notifMessage);
                        }

                        int smallIcon;
                        try {
                            String x = ManifestMetaData.getMetaData(context, Constants.LABEL_NOTIFICATION_ICON);
                            smallIcon = context.getResources().getIdentifier(x, "drawable", context.getPackageName());
                            if (smallIcon == 0) throw new IllegalArgumentException();
                        } catch (Throwable t) {
                            smallIcon = DeviceInfo.getAppIconAsIntId(context);
                        }

                        NotificationCompat.Builder nb = new NotificationCompat.Builder(context)
                                .setContentTitle(notifTitle)
                                .setContentText(notifMessage)
                                .setLargeIcon(Utils.getNotificationBitmap(icoPath, true, context))
                                .setContentIntent(pIntent)
                                .setAutoCancel(true)
                                .setStyle(style)
                                .setSmallIcon(smallIcon);

                        try {
                            if (extras.containsKey("wzrk_sound")) {
                                Object o = extras.get("wzrk_sound");
                                if ((o instanceof String && o.equals("true")) || (o instanceof Boolean && (Boolean) o)) {
                                    Uri defaultSoundUri = RingtoneManager.getDefaultUri(RingtoneManager.TYPE_NOTIFICATION);
                                    nb.setSound(defaultSoundUri);
                                }
                            }
                        } catch (Throwable t) {
                            Logger.error("Could not process sound parameter", t);
                        }

                        Notification n = nb.build();

                        NotificationManager notificationManager =
                                (NotificationManager) context.getSystemService(Context.NOTIFICATION_SERVICE);

                        notificationManager.notify((int) (Math.random() * 100), n);
                    } catch (Throwable t) {
                        // Occurs if the notification image was null
                        // Let's return, as we couldn't get a handle on the app's icon
                        // Some devices throw a PackageManager* exception too
                        Logger.error("Couldn't render notification!", t);
                    }
                }
            });
        } catch (Throwable t) {
            Logger.error("Failed to process GCM notification", t);
        }
    }

    /**
     * This method is used internally.
     */
    public void pushDeepLink(Uri uri) {
        pushDeepLink(uri, false);
    }

    synchronized void pushDeepLink(Uri uri, boolean install) {
        if (uri == null)
            return;

        try {
            JSONObject referrer = UriHelper.getUrchinFromUri(uri);
            referrer.put("referrer", uri.toString());
            if (install) {
                referrer.put("install", true);
            }
            event.pushDeviceDetailsWithExtras(referrer);
        } catch (Throwable t) {
            Logger.error("Failed to push deep link", t);
        }
    }

    public synchronized void pushInstallReferrer(String source, String medium, String campaign) {
        if (source == null && medium == null && campaign == null) return;
        try {
            // If already pushed, don't send it again
            int status = StorageHelper.getInt(context, "app_install_status", 0);
            if (status != 0) {
                Logger.log("Install referrer has already been set. Will not override it");
                return;
            }
            StorageHelper.putInt(context, "app_install_status", 1);

            if (source != null) source = Uri.encode(source);
            if (medium != null) medium = Uri.encode(medium);
            if (campaign != null) campaign = Uri.encode(campaign);

            String uriStr = "wzrk://track?install=true";
            if (source != null) uriStr += "&utm_source=" + source;
            if (medium != null) uriStr += "&utm_medium=" + medium;
            if (campaign != null) uriStr += "&utm_campaign=" + campaign;

            Uri uri = Uri.parse(uriStr);
            pushDeepLink(uri, true);
        } catch (Throwable t) {
            Logger.error("Failed to push install referrer", t);
        }
    }

    private static final HashMap<String, Integer> installReferrerMap = new HashMap<String, Integer>(8);

    public void enablePersonalization() {
        LocalDataStore.setPersonalisationEnabled(context, true);
    }

    public void disablePersonalization() {
        LocalDataStore.setPersonalisationEnabled(context, false);
    }

    public void pushInstallReferrer(Intent intent) {
        try {
            final Bundle extras = intent.getExtras();
            // Preliminary checks
            if (extras == null || !extras.containsKey("referrer")) {
                return;
            }
            final String url;
            try {
                url = URLDecoder.decode(extras.getString("referrer"), "UTF-8");

                Logger.logFine("Referrer received: " + url);
            } catch (Throwable e) {
                // Could not decode
                return;
            }
            if (url == null) {
                return;
            }
            int now = (int) (System.currentTimeMillis() / 1000);

            if (installReferrerMap.containsKey(url) && now - installReferrerMap.get(url) < 10) {
                Logger.logFine("Skipping install referrer due to duplicate within 10 seconds");
                return;
            }

            installReferrerMap.put(url, now);

            Uri uri = Uri.parse("wzrk://track?install=true&" + url);

            pushDeepLink(uri, true);
        } catch (Throwable t) {
            // Weird
        }
    }

    /**
     * get the current device location
     * requires Location Permission in AndroidManifest e.g. "android.permission.ACCESS_COARSE_LOCATION"
     * You can then use the returned Location value to update the user profile location in CleverTap via {@link #setLocation(Location)}
     *
     * @return android.location.Location
     */

    public Location getLocation() {
        return _getLocation();
    }

    /**
     * set the user profile location in CleverTap
     * location can then be used for geo-segmentation etc.
     *
     * @param location android.location.Location
     */

    public void setLocation(Location location) {
        _setLocation(location);
    }

    /**
     * @deprecated use {@link #setLocation(Location)} ()} instead.
     */

    @Deprecated
    public void updateLocation(Location location) {
        _setLocation(location);
    }

    private void _setLocation(Location location) {
        if (location == null) return;
        locationFromUser = location;
        Logger.logFine("Location updated (" + location.getLatitude() + ", " + location.getLongitude() + ")");

        // only queue the location ping if we are in the foreground
        if (!isAppForeground()) return;

        // Queue the ping event to transmit location update to server
        // min 10 second interval between location pings
        final int now = (int) (System.currentTimeMillis() / 1000);
        if (now > (lastLocationPingTime + Constants.LOCATION_PING_INTERVAL_IN_SECONDS)) {
            QueueManager.queueEvent(context, new JSONObject(), Constants.PING_EVENT);
            lastLocationPingTime = now;
            Logger.logFine("Queuing location ping event for location (" + location.getLatitude() + ", " + location.getLongitude() + ")");
        }
    }

    private Location _getLocation() {
        try {
            LocationManager lm = (LocationManager) context.getSystemService(Context.LOCATION_SERVICE);
            List<String> providers = lm.getProviders(true);
            Location bestLocation = null;
            Location l = null;
            for (String provider : providers) {
                try {
                    l = lm.getLastKnownLocation(provider);
                } catch (SecurityException e) {
                    //no-op
                    Logger.logFine("Location security exception", e);
                }

                if (l == null) {
                    continue;
                }
                if (bestLocation == null || l.getAccuracy() < bestLocation.getAccuracy()) {
                    bestLocation = l;
                }
            }

            return bestLocation;
        } catch (Throwable t) {
            Logger.logFine("Couldn't get user's location", t);
            return null;
        }
    }

    /**
     * Returns a unique identifier by which CleverTap identifies this user.
     *
     * @return The user identifier currently being used to identify this user.
     */
    public String getCleverTapID() {
        return DeviceInfo.getDeviceID();
    }

    /**
     * Returns a unique CleverTap identifier suitable for use with install attribution providers.
     *
     * @return The attribution identifier currently being used to identify this user.
     */
    public String getCleverTapAttributionIdentifier() {
        return DeviceInfo.getAttributionID();
    }
}
