
package com.clevertap.android.sdk;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.SharedPreferences;

import com.google.android.gms.gcm.GoogleCloudMessaging;
import com.google.android.gms.iid.InstanceID;

public final class GcmManager {

    private static Context context = null;
    private static final Object LOCK = new Object();

    /**
     * CleverTapAPI calls this in its singleton constructor.
     */
    static void initializeWithContext(Context context) {
        GcmManager.context = context;
        doGCMRefresh();
    }

    // called from GcmTokenListenerService
    static void onTokenRefresh() {
        doGCMRefresh();
    }

    // all happens serial async

    private static void doGCMRefresh() {
        CleverTapAPI.postAsyncSafely("GcmManager#doGCMRefresh", new Runnable() {
            @Override
            public void run() {
                try {
                    String senderID = getGCMSenderID();
                    if (senderID == null) {
                        Logger.error("GcmManager: GCM Sender ID unknown, unable to request GCM token");
                        return;
                    }

                    if (!isGooglePlayServicesAvailable()) {
                        Logger.error("GcmManager: Play Services unavailable, unable to request GCM token");
                        return;
                    }

                    String freshToken = GCMGetFreshToken(senderID);
                    if (freshToken == null) return;

                    cacheGCMToken(freshToken);

                    // better safe to always force a push from here
                    pushDeviceToken(freshToken, true, true);

                } catch (Throwable t) {
                    Logger.logFine("GcmManager: GCM Token error", t);
                }
            }
        });
    }

    // helpers

    private static boolean isGooglePlayServicesAvailable() {
        return DeviceInfo.isGooglePlayServicesAvailable();
    }

    private static String getGCMSenderID() {
        return DeviceInfo.getGCMSenderID();
    }

    /**
     * request token from GCM
     */
    private static String GCMGetFreshToken(final String senderID) {
        Logger.logFine("GcmManager: Requesting a GCM token for Sender ID - " + senderID);

        String token = null;
        try {
            token = InstanceID.getInstance(context)
                    .getToken(senderID, GoogleCloudMessaging.INSTANCE_ID_SCOPE, null);
        } catch (Throwable t) {
            Logger.error("GcmManager: Error requesting GCM token", t);
        }
        return token;
    }

    private static final String GCM_PROPERTY_REG_ID = "registration_id"; // maintain legacy value

    private static SharedPreferences getPreferences() {
        try {
            return (context == null) ? null : StorageHelper.getPreferences(context);
        } catch (Throwable t) {
            return null;
        }
    }

    private static String getCachedGCMToken() {
        SharedPreferences prefs = getPreferences();
        return (prefs == null) ? null : prefs.getString(GCM_PROPERTY_REG_ID, null);
    }

    @SuppressLint("CommitPrefEdits")
    private static void cacheGCMToken(String token) {
        try {
            if (token == null || alreadyHaveGCMToken(token)) return;

            final SharedPreferences prefs = getPreferences();
            if (prefs == null) return;

            SharedPreferences.Editor editor = prefs.edit();
            editor.putString(GCM_PROPERTY_REG_ID, token);
            StorageHelper.persist(editor);
        } catch (Throwable t) {
            Logger.logFine("GcmManager: Unable to cache GCM Token", t);
        }
    }

    private static boolean alreadyHaveGCMToken(final String newToken) {
        if (newToken == null) return false;

        String cachedToken = getCachedGCMToken();
        return (cachedToken != null && cachedToken.equals(newToken));
    }

    private static boolean havePushedDeviceToken = false;

    static void pushDeviceToken(String token, final boolean register, final boolean forceUpdate) {
        synchronized (LOCK) {
            if (havePushedDeviceToken && !forceUpdate) {
                Logger.logFine("GcmManager: skipping device token push - already sent.");
                return;
            }

            try {
                token = (token != null) ? token : getCachedGCMToken();
                if (token == null) return;
                DataHandler.pushDeviceToken(context, token, register, PushType.GCM);
                havePushedDeviceToken = true;
            } catch (Throwable t) {
                Logger.logFine("GcmManager: pushing device token failed", t);
            }
        }
    }
}