package com.clevertap.android.sdk;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.SharedPreferences;
import android.os.Bundle;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.text.SimpleDateFormat;
import java.util.*;

import static com.clevertap.android.sdk.StorageHelper.getPreferences;

/**
 * User: Jude Pereira
 * Date: 10/03/2016
 * Time: 14:46
 */
@SuppressWarnings("Contract")
final class InAppFCManager {

    private static final SimpleDateFormat ddMMyyyy = new SimpleDateFormat("ddMMyyyy", Locale.US);

    // Storage keys
    private static final String KEY_COUNTS_PER_INAPP = "counts_per_inapp";
    private static final String KEY_COUNTS_SHOWN_TODAY = "istc_inapp";
    private static final String KEY_MAX_PER_DAY = "istmcd_inapp";

    // InApp payload keys
    private static final String INAPP_ID_IN_PAYLOAD = "ti";

    // Session state variables
    private final ArrayList<String> mDismissedThisSession = new ArrayList<String>();
    private final HashMap<String, Integer> mShownThisSession = new HashMap<String, Integer>();
    private int mShownThisSessionCount = 0;

    private static InAppFCManager instance = null;
    private Context context;

    private InAppFCManager(Context context) {
        this.context = context;
    }

    @SuppressLint("CommitPrefEdits")
    private static synchronized InAppFCManager getInstance(final Context context) {
        if (instance == null) {
            instance = new InAppFCManager(context);
        }

        final String today = ddMMyyyy.format(new Date());
        final String lastUpdated = StorageHelper.getString(context, "ict_date", "20140428");
        if (!today.equals(lastUpdated)) {
            StorageHelper.putString(context, "ict_date", today);

            // Reset today count
            StorageHelper.putInt(context, KEY_COUNTS_SHOWN_TODAY, 0);

            // Reset the counts for each inapp
            final SharedPreferences prefs = getPreferences(context, KEY_COUNTS_PER_INAPP);
            final SharedPreferences.Editor editor = prefs.edit();
            final Map<String, ?> all = prefs.getAll();
            for (String inapp : all.keySet()) {
                Object ov = all.get(inapp);
                if (!(ov instanceof String)) {
                    editor.remove(inapp);
                    continue;
                }

                String[] oldValues = ((String) ov).split(",");
                if (oldValues.length != 2) {
                    editor.remove(inapp);
                    continue;
                }

                // protocol: todayCount,lifeTimeCount
                try {
                    editor.putString(inapp, "0," + oldValues[1]);
                } catch (Throwable t) {
                    Logger.error("Failed to reset todayCount for inapp " + inapp, t);
                }
            }

            StorageHelper.persist(editor);
        }
        return instance;
    }

    private int[] getInAppCountsFromPersistentStore(String inappID) {
        final SharedPreferences prefs = getPreferences(context, KEY_COUNTS_PER_INAPP);
        final String str = prefs.getString(inappID, null);
        if (str == null) return new int[]{0, 0};

        try {
            final String[] parts = str.split(",");
            if (parts.length != 2) return new int[]{0, 0};

            // protocol: todayCount,lifeTimeCount
            return new int[]{Integer.parseInt(parts[0]), Integer.parseInt(parts[1])};
        } catch (Throwable t) {
            return new int[]{0, 0};
        }
    }

    @SuppressLint("CommitPrefEdits")
    private void incrementInAppCountsInPersistentStore(String inappID) {
        int[] current = getInAppCountsFromPersistentStore(inappID);
        current[0] = current[0] + 1;
        current[1] = current[1] + 1;

        final SharedPreferences prefs = getPreferences(context, KEY_COUNTS_PER_INAPP);
        final SharedPreferences.Editor editor = prefs.edit();

        // protocol: todayCount,lifeTimeCount
        editor.putString(inappID, current[0] + "," + current[1]);
        StorageHelper.persist(editor);
    }

    private static String getInAppID(JSONObject inapp) {
        if (inapp.has(INAPP_ID_IN_PAYLOAD)) {
            try {
                return inapp.get(INAPP_ID_IN_PAYLOAD).toString();
            } catch (Throwable ignored) {
                return null;
            }
        }
        return null;
    }

    private boolean hasSessionCapacityMaxedOut(JSONObject inapp) {
        final String id = getInAppID(inapp);
        if (id == null) return false;

        // 1. Has this been dismissed?
        if (mDismissedThisSession.contains(id)) return true;

        // 2. Has the session max count for this inapp been breached?
        try {
            final int maxPerSession = inapp.getJSONObject("w").getInt("mdc");

            Integer c = mShownThisSession.get(id);
            if (c != null && c >= maxPerSession) return true;
        } catch (Throwable t) {
            return true;
        }

        // 3. Have we shown enough of in-apps this session?
        final int c = StorageHelper.getInt(context, Constants.INAPP_MAX_PER_SESSION, 1);
        if (mShownThisSessionCount >= c) return true;

        // Not breached
        return false;
    }

    private boolean hasLifetimeCapacityMaxedOut(JSONObject inapp) {
        final String id = getInAppID(inapp);
        if (id == null) return false;

        if (!inapp.has("tlc")) return false;

        try {
            int c = inapp.getInt("tlc");
            if (c == -1) return false;

            final int[] counts = getInAppCountsFromPersistentStore(id);
            if (counts[1] >= c) return true;
        } catch (JSONException e) {
            return true;
        }

        return false;
    }

    private boolean hasDailyCapacityMaxedOut(JSONObject inapp) {
        final String id = getInAppID(inapp);
        if (id == null) return false;

        // 1. Has the daily count maxed out globally?
        int shownTodayCount = StorageHelper.getInt(context, KEY_COUNTS_SHOWN_TODAY, 0);
        int maxPerDayCount = StorageHelper.getInt(context, KEY_MAX_PER_DAY, 1);
        if (shownTodayCount >= maxPerDayCount) return true;

        // 2. Has the daily count been maxed out for this inapp?
        if (!inapp.has("tdc")) return false;
        try {
            int maxPerDay = inapp.getInt("tdc");
            if (maxPerDay == -1) return false;

            final int[] counts = getInAppCountsFromPersistentStore(id);
            if (counts[0] >= maxPerDay) return true;
        } catch (Throwable t) {
            return true;
        }

        return false;
    }

    static synchronized boolean canShow(final Context context, JSONObject inapp) {
        try {
            if (inapp == null) return false;

            final String id = getInAppID(inapp);
            if (id == null) return true;

            // Exclude from all caps?
            if (inapp.has("efc") && inapp.getInt("efc") == 1) return true;

            final InAppFCManager instance = getInstance(context);

            if (!instance.hasSessionCapacityMaxedOut(inapp)
                    && !instance.hasLifetimeCapacityMaxedOut(inapp)
                    && !instance.hasDailyCapacityMaxedOut(inapp)) {
                return true;
            }
        } catch (Throwable t) {
            return false;
        }
        return false;
    }

    static synchronized void didDismiss(final Context context, Bundle b) {
        final Object id = b.get(INAPP_ID_IN_PAYLOAD);
        if (id != null) {
            getInstance(context).mDismissedThisSession.add(id.toString());
        }
    }

    static synchronized void destroySession() {
        instance = null;
    }

    @SuppressLint("CommitPrefEdits")
    static void changeUser(final Context context) {
        // reset counters
        StorageHelper.putInt(context, KEY_COUNTS_SHOWN_TODAY, 0);
        instance.mShownThisSession.clear();
        instance.mShownThisSessionCount = 0;
        instance.mDismissedThisSession.clear();

        final SharedPreferences prefs = getPreferences(context, KEY_COUNTS_PER_INAPP);
        final SharedPreferences.Editor editor = prefs.edit();
        editor.clear();
        StorageHelper.persist(editor);
    }

    static synchronized void didShow(final Context context, JSONObject inapp) {
        final String id = getInAppID(inapp);
        if (id == null) return;

        final InAppFCManager instance = getInstance(context);
        instance.mShownThisSessionCount++;

        Integer count = instance.mShownThisSession.get(id);
        if (count == null) count = 1;

        instance.mShownThisSession.put(id, ++count);

        instance.incrementInAppCountsInPersistentStore(id);

        int shownToday = StorageHelper.getInt(context, KEY_COUNTS_SHOWN_TODAY, 0);
        StorageHelper.putInt(context, KEY_COUNTS_SHOWN_TODAY, ++shownToday);
    }

    static synchronized void updateLimits(final Context context, int perDay, int perSession) {
        StorageHelper.putInt(context, KEY_MAX_PER_DAY, perDay);
        StorageHelper.putInt(context, Constants.INAPP_MAX_PER_SESSION, perSession);
    }

    static synchronized void attachToHeader(final Context context, JSONObject header) {
        try {
            // Trigger reset for dates

            getInstance(context);
            header.put("imp", StorageHelper.getInt(context, KEY_COUNTS_SHOWN_TODAY, 0));

            // tlc: [[targetID, todayCount, lifetime]]
            JSONArray arr = new JSONArray();
            final SharedPreferences prefs = StorageHelper.getPreferences(context, KEY_COUNTS_PER_INAPP);
            final Map<String, ?> all = prefs.getAll();
            for (String inapp : all.keySet()) {
                final Object o = all.get(inapp);
                if (o instanceof String) {
                    final String[] parts = ((String) o).split(",");
                    if (parts.length == 2) {
                        JSONArray a = new JSONArray();
                        a.put(0, inapp);
                        a.put(1, Integer.parseInt(parts[0]));
                        a.put(2, Integer.parseInt(parts[1]));
                        arr.put(a);
                    }
                }
            }

            header.put("tlc", arr);
        } catch (Throwable t) {
            Logger.logFine("Failed to attach FC to header", t);
        }
    }

    @SuppressLint("CommitPrefEdits")
    static synchronized void processResponse(final Context context, final JSONObject response) {
        try {
            if (!response.has("inapp_stale")) return;

            final JSONArray arr = response.getJSONArray("inapp_stale");


            final SharedPreferences prefs = getPreferences(context, KEY_COUNTS_PER_INAPP);
            final SharedPreferences.Editor editor = prefs.edit();

            for (int i = 0; i < arr.length(); i++) {
                final Object o = arr.get(i);
                if (o instanceof Integer) {
                    editor.remove("" + o);
                    Logger.logFine("Purged stale in-app - " + o);
                } else if (o instanceof String) {
                    editor.remove((String) o);
                    Logger.logFine("Purged stale in-app - " + o);
                }
            }

            StorageHelper.persist(editor);
        } catch (Throwable t) {
            Logger.logFine("Failed to purge out stale targets", t);
        }
    }
}