/*
 * Author: Jude Pereira
 * Copyright (c) 2014
 */

package com.clevertap.android.sdk;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.support.v4.content.ContextCompat;
import android.text.BoringLayout;

import com.clevertap.android.sdk.exceptions.CleverTapException;
import com.clevertap.android.sdk.exceptions.CleverTapPermissionsNotSatisfied;
import com.google.android.gms.ads.identifier.AdvertisingIdClient;

import java.util.TooManyListenersException;
import java.util.UUID;

/**
 * Provides various methods to access the device.
 */
final class DeviceInfo {
    private static final String GUID_PREFIX = "__";

    private static String provisionalGUID = null;

    private static final Object deviceIDLock = new Object();
    private static final Object adIDLock = new Object();

    private static Context context = null;

    private static String googleAdID = null;

    private static boolean limitAdTracking = false;

    static String getGoogleAdID() {
        synchronized (adIDLock) {
            return googleAdID;
        }
    }

    static boolean isLimitAdTrackingEnabled() {
        synchronized (adIDLock) {
            return limitAdTracking;
        }
    }

    /**
     * CleverTapAPI calls this in its singleton constructor.
     */
    static void initializeWithContext(Context context) {
        DeviceInfo.context = context;
        initDeviceIDAsync();
    }

    private static void generateProvisionalGUID() {
        synchronized (deviceIDLock) {
            if (provisionalGUID == null) {
                provisionalGUID = generateGUID();
            }
        }
    }

    private static void initDeviceIDAsync() {
        // generate a provisional while we do the rest async
        generateProvisionalGUID();

        CleverTapAPI.postAsyncSafely("DeviceInfo#generateDeviceID", new Runnable() {
            @Override
            public void run() {
                // grab and cache the googleAdID in any event if available
                // if we already have a deviceID we won't user ad id as the guid
                cacheGoogleAdID();

                // if we already have a device ID use it and just notify
                // otherwise generate one, either from ad id if available or the provisional
                String deviceID = getDeviceID();
                if (deviceID != null && deviceID.trim().length() > 2) {
                    notifyNewDeviceID(deviceID);
                } else {
                    generateDeviceID();
                }
            }
        });
    }

    private static void cacheGoogleAdID() {

        String advertisingID;
        try {
            AdvertisingIdClient.Info adInfo;
            adInfo = AdvertisingIdClient.getAdvertisingIdInfo(context);
            advertisingID = adInfo.getId();
            synchronized (adIDLock) {
                limitAdTracking = adInfo.isLimitAdTrackingEnabled();
            }
        } catch (Throwable t) {
            advertisingID = null;
        }

        if (advertisingID != null && advertisingID.trim().length() > 2) {
            synchronized (adIDLock) {
                googleAdID = advertisingID.replace("-", "");
            }
        }
    }

    private static void generateDeviceID() {
        String generatedDeviceID;

        // try google ad id first
        // if no ad id then make provisional guid permanent
        if (googleAdID != null) {
            synchronized (adIDLock) {
                generatedDeviceID = Constants.GUID_PREFIX_GOOGLE_AD_ID + googleAdID;
            }
        } else {
            Logger.logFine("Failed with Advertising ID");

            synchronized (deviceIDLock) {
                generatedDeviceID = provisionalGUID;
                Logger.logFine("Made provisional ID permanent");
            }
        }

        if (generatedDeviceID != null && generatedDeviceID.trim().length() > 2) {
            forceUpdateDeviceId(generatedDeviceID);
        } else {
            Logger.logFine("Unable to generate device ID");
        }
    }

    static String getAttributionID() {
        String deviceID = getDeviceID();
        synchronized (deviceIDLock) {
            return (deviceID != null && deviceID.trim().length() > 2) ? deviceID : provisionalGUID;
        }
    }

    static String getDeviceID() {
        synchronized (deviceIDLock) {
            return StorageHelper.getString(context, Constants.DEVICE_ID_TAG, null);
        }
    }

    private static void notifyNewDeviceID(final String deviceID) {
        CleverTapAPI.postAsyncSafely("DeviceInfo#notifyNewDeviceID", new Runnable() {
            @Override
            public void run() {
                try {
                    CleverTapAPI.notifyUserProfileInitialized(deviceID);
                } catch (Throwable t) {
                    // no-op
                }
            }
        });
    }

    private static String generateGUID() {
        return GUID_PREFIX + UUID.randomUUID().toString().replace("-", "");
    }


    static void forceNewDeviceID() {
        String deviceID = generateGUID();
        forceUpdateDeviceId(deviceID);
    }

    /**
     * Force updates the device ID, with the ID specified.
     * <p>
     * This is used internally by the SDK, there is no need to call this explicitly.
     * </p>
     *
     * @param id      The new device ID
     */
    @SuppressLint("CommitPrefEdits")
    static void forceUpdateDeviceId(String id) {
        Logger.logExtraFine("Force updating the device ID to " + id);
        synchronized (deviceIDLock) {
            StorageHelper.putString(context, Constants.DEVICE_ID_TAG, id);
        }
        notifyNewDeviceID(id);
    }

    /**
     * Tests whether a particular permission is available or not.
     *
     * @param context    The Android {@link Context}
     * @param permission The fully qualified Android permission name
     * @throws CleverTapPermissionsNotSatisfied
     */
    static void testPermission(final Context context, String permission)
            throws CleverTapPermissionsNotSatisfied {
        if (!hasPermission(context, permission))
            throw new CleverTapPermissionsNotSatisfied("Permission required: " + permission);
    }

    @SuppressWarnings("WeakerAccess")
    static boolean hasPermission(final Context context, String permission) {
        try {
            return PackageManager.PERMISSION_GRANTED == ContextCompat.checkSelfPermission(context, permission);
        } catch (Throwable t) {
            return false;
        }
    }

    /**
     * Returns the integer identifier for the default app icon.
     *
     * @param context The Android context
     * @return The integer identifier for the image resource
     */
    static int getAppIconAsIntId(final Context context) {
        ApplicationInfo ai = context.getApplicationInfo();
        return ai.icon;
    }
}
