/*
 * Author: Jude Pereira
 * Copyright (c) 2014
 */

package com.clevertap.android.sdk;

import android.accounts.Account;
import android.annotation.SuppressLint;
import android.app.Activity;
import android.app.Notification;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.content.pm.PackageInfo;
import android.graphics.Bitmap;
import android.location.Location;
import android.location.LocationManager;
import android.media.RingtoneManager;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.support.v4.app.NotificationCompat;
import com.clevertap.android.sdk.exceptions.CleverTapMetaDataNotFoundException;
import com.clevertap.android.sdk.exceptions.CleverTapPermissionsNotSatisfied;
import com.google.android.gms.ads.identifier.AdvertisingIdClient;
import com.google.android.gms.common.ConnectionResult;
import com.google.android.gms.common.GooglePlayServicesUtil;
import com.google.android.gms.gcm.GoogleCloudMessaging;
import com.google.android.gms.iid.InstanceID;
import org.json.JSONException;
import org.json.JSONObject;

import java.net.URLDecoder;
import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import static android.content.Context.MODE_PRIVATE;

/**
 * Manipulates the CleverTap SDK.
 */
public final class CleverTapAPI {
    // Static fields
    public static final String CHARGED_EVENT = "Charged";
    public static final String NOTIFICATION_TAG = "wzrk_pn";
    private static final String PROPERTY_APP_VERSION = "appVersionCode";
    private static final Handler handlerUsingMainLooper = new Handler(Looper.getMainLooper());
    private static final ExecutorService es = Executors.newFixedThreadPool(1);
    private static int debugLevel = 0;
    private static boolean appForeground = false;
    private static CleverTapAPI ourInstance = null;
    static Runnable pendingInappRunnable = null;
    static String localAccountID = null, localToken = null;
    private static int lastLocationPingTime = 0;

    private static boolean usingSegment = false;

    private static boolean usingParse = false;

    static int activityCount = 0;

    private static ArrayList<ValidationResult> pendingValidationResults = new ArrayList<ValidationResult>();
    private static final Boolean pendingValidationResultsLock = true;

    static Activity currentActivity;

    private static String parseInstallationId = null;

    // Non static fields
    public final EventHandler event;
    public final DataHandler data;
    public final ProfileHandler profile;
    public final SessionHandler session;
    private Runnable sessionTimeoutRunnable = null;
    private final Context context;
    private long appLastSeen;
    private SyncListener syncListener = null;
    private Location locationFromUser = null;

    /**
     * Note: You don't need to use a lock to access this variable - everything happens in
     * a serial execution manner with the async handler
     */
    private boolean pushedGcmId = false;

    private boolean pushedProfileDefaults = false;

    static boolean isUsingSegment() {
        return usingSegment;
    }

    static boolean isUsingParse() {
        return usingParse;
    }

    static String getParseInstallationId() {
        return parseInstallationId;
    }

    static ValidationResult popValidationResult() {
        // really a shift
        ValidationResult vr = null;

        synchronized (pendingValidationResultsLock) {
            try {
                if (!pendingValidationResults.isEmpty()) {
                    vr = pendingValidationResults.remove(0);
                }
            } catch (Exception e) {
                // no-op
            }
        }
        return vr;
    }

    static void pushValidationResult(ValidationResult vr) {
        synchronized (pendingValidationResultsLock) {
            try {
                int len = pendingValidationResults.size();
                if (len > 50) {
                    ArrayList<ValidationResult> trimmed = new ArrayList<ValidationResult>();
                    // Trim down the list to 40, so that this loop needn't run for the next 10 events
                    // Hence, skip the first 10 elements
                    for (int i = 10; i < len; i++)
                        trimmed.add(pendingValidationResults.get(i));
                    trimmed.add(vr);
                    pendingValidationResults = trimmed;
                } else {
                    pendingValidationResults.add(vr);
                }
            } catch (Exception e) {
                // no-op
            }
        }
    }

    /**
     * Returns the generic handler object which is used to post
     * runnables. The returned value will never be null.
     *
     * @return The static generic handler
     * @see Handler
     */
    static Handler getHandlerUsingMainLooper() {
        return handlerUsingMainLooper;
    }

    /**
     * Use this to safely post a runnable to the async handler.
     * It adds try/catch blocks around the runnable and the handler itself.
     */
    static void postAsyncSafely(final String name, final Runnable runnable) {
        try {
            es.submit(new Runnable() {
                @Override
                public void run() {
                    try {
                        Logger.logFine("Executor service: Starting task - " + name);
                        final long start = System.currentTimeMillis();
                        runnable.run();
                        final long time = System.currentTimeMillis() - start;
                        Logger.logFine("Executor service: Task completed successfully in " + time + "ms (" + name + ")");
                    } catch (Throwable t) {
                        Logger.logFine("Executor service: Failed to complete the scheduled task", t);
                    }
                }
            });
        } catch (Throwable t) {
            Logger.logFine("Failed to submit task to the executor service", t);
        }
    }

    /**
     * Returns whether or not the app is in the foreground.
     *
     * @return The foreground status
     */
    static boolean isAppForeground() {
        return appForeground;
    }

    /**
     * This method is used internally.
     */
    public static void setAppForeground(boolean appForeground) {
        CleverTapAPI.appForeground = appForeground;
    }

    public void setSyncListener(SyncListener syncListener) {
        this.syncListener = syncListener;
    }

    public SyncListener getSyncListener() {
        return syncListener;
    }

    /**
     * Returns an instance of the CleverTap SDK.
     *
     * @param context The Android context
     * @return The {@link CleverTapAPI} object
     * @throws CleverTapMetaDataNotFoundException
     * @throws CleverTapPermissionsNotSatisfied
     */
    public static synchronized CleverTapAPI getInstance(final Context context)
            throws CleverTapMetaDataNotFoundException, CleverTapPermissionsNotSatisfied {
        if (ourInstance == null && context != null) {
            try {
                usingSegment = ManifestMetaData.getMetaData(context, Constants.LABEL_SEGMENT) != null;
                if (usingSegment) {
                    Logger.log("Personalisation (naked API) and pushing events/profile data is unavailable");
                    retrieveIdentifierForSegment(context);
                }
            } catch (Throwable t) {
                // Ignore
            }

            try {
                usingParse = ManifestMetaData.getMetaData(context, Constants.LABEL_PARSE) != null;
                if (usingParse) {
                    retrieveIdentifierForParse(context);
                }
            } catch (Throwable t) {
                // Ignore
            }

            if (!usingSegment) {
                DeviceInfo.updateDeviceIdIfRequired(context.getApplicationContext());
            }

            ourInstance = new CleverTapAPI(context.getApplicationContext());
        }
        return ourInstance;
    }

    private CleverTapAPI(final Context context)
            throws CleverTapMetaDataNotFoundException, CleverTapPermissionsNotSatisfied {
        this.context = context;
        event = new EventHandler(context);
        profile = new ProfileHandler(context);
        data = new DataHandler(context);
        session = new SessionHandler(context);

        // wake up the data store
        LocalDataStore.initializeWithContext(context);

        DeviceInfo.testPermission(context, "android.permission.INTERNET");
        // Test for account ID, and token
        ManifestMetaData.getMetaData(context, Constants.LABEL_ACCOUNT_ID);
        ManifestMetaData.getMetaData(context, Constants.LABEL_TOKEN);


        Logger.logFine("New instance of CleverTapAPI created");

        // Try to automatically register to GCM
        initPushHandling();
    }

    void notifyUserProfileInitialized() {
        if (this.syncListener != null) {
            String cleverTapID = this.getCleverTapID();

            if (cleverTapID == null) {
                return;
            }

            try {
                Logger.logFine("Notifying UserProfileInitialized with CleverTapId " + cleverTapID);
                this.syncListener.profileDidInitialize(cleverTapID);
            } catch (Throwable t) {
                Logger.error("Execution of sync listener failed", t);
            }
        }
    }

    private static void retrieveIdentifierForSegment(final Context context) {
        postAsyncSafely("segment-id-retriever", new Runnable() {
            @Override
            public void run() {
                // Try to get the advertiser ID first
                String advertisingID = getAdvertisingIDSafelySync(context);
                if (advertisingID != null && !advertisingID.trim().equals("")) {
                    DeviceInfo.forceUpdateDeviceId(context, "___" + advertisingID);
                    Logger.logFine("Using advertiser ID as device identifier (Segment is present)");
                    return;
                } else {
                    Logger.logFine("Unable to retrieve advertising ID as the Segment identifier");
                }

                // Try to get Segment's anonymous ID
                String anonymousId = null;
                SharedPreferences prefs = context.getSharedPreferences("analytics-android", MODE_PRIVATE);
                Map<String, ?> all = prefs.getAll();
                for (String s : all.keySet()) {
                    if (s.startsWith("traits")) {
                        try {
                            JSONObject jsonObject = new JSONObject(prefs.getString(s, "{}"));
                            anonymousId = jsonObject.getString("anonymousId");
                            break;
                        } catch (JSONException e) {
                            // Ignore
                        }
                    }
                }

                if (anonymousId == null || anonymousId.trim().equals("")) {
                    // Try again in a few seconds
                    getHandlerUsingMainLooper().postDelayed(new Runnable() {
                        @Override
                        public void run() {
                            retrieveIdentifierForSegment(context);
                        }
                    }, 5000);
                    return;
                }

                DeviceInfo.forceUpdateDeviceId(context, "___" + anonymousId);
                Logger.logFine("Using Segment's anonymous ID as device identifier (Segment is present)");
            }
        });
    }

    private static void retrieveIdentifierForParse(final Context context) {

        postAsyncSafely("parse-id-retriever", new Runnable() {
            @Override
            public void run() {

                // Try to get Parse's installation ID
                String installationId = ParseUtils.getParseInstallationId(context);

                if (installationId != null) {
                    try {
                        installationId = "_p" + installationId.replace("-", "");
                        Logger.logFine("Setting Parse installation ID " + installationId);
                        parseInstallationId = installationId;
                    } catch (Exception e) {
                        Logger.logFine("Error reading Parse installation ID: " + e.toString());
                    }
                }
            }
        });
    }

    public static void changeCredentials(String accountID, String token) {
        localAccountID = accountID;
        localToken = token;
    }

    static String getAccountID(final Context context) {
        if (localAccountID == null) {
            try {
                localAccountID = ManifestMetaData.getMetaData(context, Constants.LABEL_ACCOUNT_ID);
            } catch (Exception e) {
                // no-op
                Logger.logFine("Error reading accountID: " + e.toString());
            }
        }

        return localAccountID;
    }

    private final String PREFS_LAST_DAILY_PUSHED_EVENTS_DATE = "lastDailyEventsPushedDate";

    private JSONObject getProfileDefaults() {
        JSONObject defaults = new JSONObject();
        String name = null;

        Account account = this.profile.getGoogleAccount();
        if (account != null && account.name != null) {
            ValidationResult vr;
            try {
                vr = Validator.cleanObjectValue(account.name, false, false);
                String email = vr.getObject().toString();
                String[] parts = email.split("\\@");
                name = parts[0];
            } catch (Exception e) {
                // Ignore
            }
        }

        if (name != null) {
            try {
                defaults.put("Name", name);
            } catch (Exception e) {
                // ignore
            }

        }

        return defaults;
    }

    void pushProfileDefaults() {

        postAsyncSafely("CleverTapAPI#pushProfileDefaultsAsync", new Runnable() {
            @Override
            public void run() {

                if (pushedProfileDefaults) {
                    return;
                }

                pushedProfileDefaults = true;

                try {

                    JSONObject defaults = getProfileDefaults();

                    if (defaults.length() > 0) {
                        profile.pushProfileDefaults(context, defaults);
                    }
                } catch (Throwable t) {
                    Logger.error("Daily profile sync failed", t);
                }
            }
        });

    }

    @SuppressLint("CommitPrefEdits")
    private void pushDailyEventsAsync() {
        postAsyncSafely("CleverTapAPI#pushDailyEventsAsync", new Runnable() {
            @Override
            public void run() {
                try {
                    Date d = new Date();
                    SharedPreferences prefs = StorageHelper.getPreferences(context);
                    Calendar cal = Calendar.getInstance();
                    cal.setTime(d);
                    if (hasAppVersionChanged() || prefs.getInt(PREFS_LAST_DAILY_PUSHED_EVENTS_DATE, 0) != cal.get(Calendar.DATE)) {

                        Logger.logFine("Queuing daily events");
                        profile.pushBasicProfile(null);
                        updateAdvertisingID();
                        if (!pushedGcmId) {
                            String gcmRegId = getRegistrationId();
                            if (gcmRegId != null && !gcmRegId.equals(""))
                                data.pushGcmRegistrationId(gcmRegId, GcmManager.isGcmEnabled(context));
                        } else {
                            Logger.logFine("Skipped push of the GCM ID. Somebody already sent it.");
                        }
                        pushedGcmId = true;
                    }
                    SharedPreferences.Editor editor = prefs.edit().putInt(PREFS_LAST_DAILY_PUSHED_EVENTS_DATE, cal.get(Calendar.DATE));
                    StorageHelper.persist(editor);
                    updateAppVersionInPrefs();
                } catch (Throwable t) {
                    Logger.error("Daily profile sync failed", t);
                }
            }
        });
    }

    /**
     * Enables or disables debugging. If enabled, see debug messages in Android's logcat utility.
     * Debug messages are tagged as CleverTap.
     *
     * @param level Can be one of the following - 0(disables all debugging),
     *              1(show some debug output)
     */
    public static void setDebugLevel(int level) {
        debugLevel = level;
    }

    public static int getDebugLevel() {
        return debugLevel;
    }

    /**
     * Tells CleverTap that the current Android activity was paused. Add a call to this
     * method in the onPause() method of every activity.
     *
     * @param activity The calling activity
     */
    public void activityPaused(Activity activity) {
        setAppForeground(false);
        // Just update it
        currentActivity = activity;
        appLastSeen = System.currentTimeMillis();
        // Create the runnable, if it is null
        if (sessionTimeoutRunnable == null)
            sessionTimeoutRunnable = new Runnable() {
                @Override
                public void run() {
                    long now = System.currentTimeMillis();
                    if (!isAppForeground()
                            && (now - appLastSeen) > Constants.SESSION_LENGTH_MINS * 60 * 1000) {
                        Logger.logFine("Session timeout reached");
                        SessionManager.destroySession();

                        Logger.logFine("Current activity set to null");
                        CleverTapAPI.currentActivity = null;
                    }
                }
            };
        // Remove any existing session timeout runnable object
        getHandlerUsingMainLooper().removeCallbacks(sessionTimeoutRunnable);
        getHandlerUsingMainLooper().postDelayed(sessionTimeoutRunnable,
                Constants.SESSION_LENGTH_MINS * 60 * 1000);
        Logger.logFine("Foreground activity gone to background");
    }


    private static HashSet<String> inappActivityExclude = null;

    /**
     * Notifies CleverTap that a new activity was created. Add a call to this method
     * in the onCreate() method of every activity.
     */
    private void notifyActivityChanged(Activity activity) {
        currentActivity = activity;
        activityCount++;

        //noinspection ConstantConditions
        if (activity != null) {
            Logger.logFine("Activity changed: " + activity.getLocalClassName());
        }

        if (inappActivityExclude == null) {
            inappActivityExclude = new HashSet<String>();
            try {
                String activities = ManifestMetaData.getMetaData(context, Constants.LABEL_INAPP_EXCLUDE);
                if (activities != null) {
                    String[] split = activities.split(",");
                    for (String a : split) {
                        inappActivityExclude.add(a.trim());
                    }
                }
            } catch (Throwable t) {
                // Ignore
            }
            Logger.log("In-app notifications will not be shown on " + Arrays.toString(inappActivityExclude.toArray()));
        }

        if (Constants.ENABLE_INAPP) {
            // We MUST loop through the set and do a contains on each and every entry there
            boolean activityBlacklisted = false;

            for (String blacklistedActivity : inappActivityExclude) {
                if (currentActivity != null && currentActivity.getLocalClassName().contains(blacklistedActivity)) {
                    activityBlacklisted = true;
                    break;
                }
            }


            if (!activityBlacklisted) {
                if (pendingInappRunnable != null) {
                    Logger.logFine("Found a pending inapp runnable. Scheduling it");
                    getHandlerUsingMainLooper().postDelayed(pendingInappRunnable, 200);
                    pendingInappRunnable = null;
                } else {
                    // Show an in-app notification, if available
                    InAppManager.showNotificationIfAvailable(context);
                }
            } else {
                Logger.log("In-app notifications will not be shown for this activity ("
                        + (activity != null ? activity.getLocalClassName() : "") + ")");
            }
        }

        pushDailyEventsAsync();

        event.pushDeviceDetails();
        SessionManager.activityChanged(getCurrentActivityName());
    }

    /**
     * Tells CleverTap that the current Android activity was resumed. Add a call to this
     * method in the onResume() method of every activity.
     *
     * @param activity The calling activity
     */
    public void activityResumed(Activity activity) {
        setAppForeground(true);
        boolean newLaunch = currentActivity == null;
        // Check if this is a different activity from the last one
        if (currentActivity == null
                || !currentActivity.getLocalClassName().equals(activity.getLocalClassName()))
            notifyActivityChanged(activity);

        Logger.logFine("Background activity in foreground");

        // Start a lazy App Launched
        if (newLaunch) {
            getHandlerUsingMainLooper().postDelayed(new Runnable() {
                @Override
                public void run() {
                    // Raise the App Launched event
                    pushAppLaunchedEvent("delayed generic handler");
                }
            }, 500);
        }
    }

    private void updateAdvertisingID() {
        postAsyncSafely("CleverTapAPI#updateAdvertisingID", new Runnable() {
            @Override
            public void run() {
                String advertisingID = getAdvertisingIDSafelySync(context);
                if (advertisingID != null) {
                    HashMap<String, Object> m = new HashMap<String, Object>();
                    m.put("GoogleAdID", advertisingID);
                    profile.push(m);
                } else {
                    Logger.logFine("Failed to update advertising ID");
                    StorageHelper.putString(context, Constants.ADVERTISER_ID, "");
                }
            }
        });
    }

    private static String getAdvertisingIDSafelySync(final Context context) {
        try {
            AdvertisingIdClient.Info adInfo;
            adInfo = AdvertisingIdClient.getAdvertisingIdInfo(context);
            final String id = adInfo.getId();
            final boolean dontTrack = adInfo.isLimitAdTrackingEnabled();
            if (dontTrack) {
                return null;
            }
            return id;
        } catch (Throwable t) {
            return null;
        }
    }

    /*package*/ JSONObject getAppLaunchedFields() {
        try {
            final JSONObject evtData = new JSONObject();
            PackageInfo pInfo = context.getPackageManager().getPackageInfo(context.getPackageName(), 0);
            evtData.put("Build", pInfo.versionCode + "");
            evtData.put("Version", pInfo.versionName);
            evtData.put("OS Version", android.os.Build.VERSION.RELEASE);
            evtData.put("SDK Version", BuildInfo.SDK_SVN_REVISION);

            Location loc = getLocation();
            if (loc != null) {
                evtData.put("Latitude", loc.getLatitude());
                evtData.put("Longitude", loc.getLongitude());
            }

            // send up parseInstallationId
            if (parseInstallationId != null) {
                evtData.put("prg", parseInstallationId);
            }

            try {
                // Device data
                String make = Build.MANUFACTURER;
                String model = Build.MODEL;
                model = model.replace(make, "");
                evtData.put("Make", make.trim());
                evtData.put("Model", model.trim());
            } catch (Throwable t) {
                // Ignore
            }
            return evtData;
        } catch (Throwable t) {
            Logger.logFine("Failed to construct App Launched event", t);
            return new JSONObject();
        }
    }

    synchronized void pushAppLaunchedEvent(String source) {
        if (SessionManager.isAppLaunchedBeenPushed()) {
            Logger.logFine("App Launched has already been triggered. Will not trigger it; source = " + source);
            return;
        } else {
            Logger.logFine("Firing App Launched event; source = " + source);
        }
        SessionManager.setAppLaunchedBeenPushed(true);
        JSONObject event = new JSONObject();
        try {
            event.put("evtName", Constants.APP_LAUNCHED_EVENT);
            event.put("evtData", getAppLaunchedFields());
        } catch (Throwable t) {
            // We won't get here
        }
        QueueManager.addToQueue(context, event, Constants.RAISED_EVENT);
    }

    /**
     * Sends all the events in the event queue.
     */
    public void flush() {
        new CommsManager().drainQueueAsync(context);
    }

    /**
     * Check the device to make sure it has the Google Play Services APK.
     *
     * @return The result of the request
     */
    private boolean checkPlayServices() {
        int resultCode;
        try {
            resultCode = GooglePlayServicesUtil.isGooglePlayServicesAvailable(context);
        } catch (Throwable e) {
            resultCode = -1;
        }

        Logger.logFine("Google Play services availability: " +
                (resultCode == ConnectionResult.SUCCESS));
        return resultCode == ConnectionResult.SUCCESS;
    }

    /**
     * Initializes the GCM architecture built into the CleverTap SDK.
     * <p/>
     * If you'd like to handle the messages yourself, please refer our documentation.
     */
    void initPushHandling() {
        String senderID;
        try {
            senderID = ManifestMetaData.getMetaData(context, Constants.LABEL_SENDER_ID);
            if (senderID != null) {
                senderID = senderID.replace("id:", "");
            }
        } catch (CleverTapMetaDataNotFoundException e) {
            // Ignore
            return;
        } catch (Throwable t) {
            Logger.error("Failed to automatically register to GCM", t);
            return;
        }
        Logger.log("Requesting a GCM registration ID for project ID(s) - " + senderID);
        if (checkPlayServices() && getRegistrationId().equals("")) {
            Logger.logFine("Google Play services available, and no valid registration ID found. Initializing");
            doGcmRegistration(senderID);
        }
    }

    /**
     * Gets the current registration ID for application on GCM service.
     * <p/>
     * If the result is empty, the app needs to register.
     *
     * @return registration ID, or empty string if there is no existing
     * registration ID.
     */
    public String getRegistrationId() {
        final SharedPreferences prefs = getPreferences(context);
        String registrationId = prefs.getString(Constants.GCM_PROPERTY_REG_ID, "");
        if (registrationId.equals("")) {
            if (getDebugLevel() >= Constants.DEBUG_FINEST)
                Logger.logFine("GCM registration ID not found");
            return "";
        }
        // Check if app was updated; if so, it must clear the registration ID
        // since the existing regID is not guaranteed to work with the new
        // app version
        int registeredVersion = prefs.getInt(PROPERTY_APP_VERSION, Integer.MIN_VALUE);
        int currentVersion = getAppVersion();
        if (registeredVersion != currentVersion) {
            if (getDebugLevel() >= Constants.DEBUG_FINEST)
                Logger.logFine("App version changed");
            return "";
        }
        return registrationId;
    }

    /**
     * @return Application's {@code SharedPreferences}.
     */
    private SharedPreferences getPreferences(Context context) {
        return StorageHelper.getPreferences(context);
    }

    /**
     * @return Application's version code from the {@code PackageManager}.
     */
    private int getAppVersion() {
        try {
            PackageInfo packageInfo = context.getPackageManager()
                    .getPackageInfo(context.getPackageName(), 0);
            return packageInfo.versionCode;
        } catch (Throwable ignore) {
            // Should never happen
            return 0;
        }
    }

    /**
     * This is meant for use with regard to non-GCM related data.
     * For GCM, use {@code PROPERTY_APP_VERSION}.
     */
    private final String APP_VERSION_CODE_TAG = "avc";

    /**
     * Compares the current app version and the recorded app version.
     *
     * @return Whether the app version was changed - upgraded, or downgraded
     */
    private boolean hasAppVersionChanged() {
        int currentVer = getAppVersion();
        SharedPreferences prefs = getPreferences(context);
        int prefsVer = prefs.getInt(APP_VERSION_CODE_TAG, -1);
        return currentVer != prefsVer;
    }

    /**
     * Updates the stored app version code to the current one.
     */
    @SuppressLint("CommitPrefEdits")
    private void updateAppVersionInPrefs() {
        int currentVer = getAppVersion();
        SharedPreferences prefs = getPreferences(context);
        SharedPreferences.Editor editor = prefs.edit().putInt(APP_VERSION_CODE_TAG, currentVer);
        StorageHelper.persist(editor);
    }

    /**
     * Registers the application with the GCM servers asynchronously.
     * <p/>
     * Stores the registration ID and app versionCode in the application's
     * shared preferences.
     */
    private void doGcmRegistration(final String senderId) {
        postAsyncSafely("CleverTapAPI#doGcmRegistration", new Runnable() {
            @Override
            public void run() {
                try {
                    Logger.logFine("Trying to register against GCM servers");

                    String gcmRegId = InstanceID.getInstance(context)
                            .getToken(senderId, GoogleCloudMessaging.INSTANCE_ID_SCOPE, null);

                    Logger.logFine("Registered successfully");

                    // Send the ID to CleverTap servers
                    data.pushGcmRegistrationId(gcmRegId, true);
                    pushedGcmId = true;

                    // Persist the regID - no need to register again.
                    storeRegistrationId(gcmRegId);
                } catch (Throwable ex) {
                    Logger.logFine("Exception while registering with GCM servers: " + ex.toString());
                    ex.printStackTrace();
                }
            }
        });
    }

    /**
     * Stores the registration ID and app versionCode in the application's
     * {@code SharedPreferences}.
     *
     * @param regId registration ID
     */
    @SuppressLint("CommitPrefEdits")
    private void storeRegistrationId(String regId) {
        final SharedPreferences prefs = getPreferences(context);
        int appVersion = getAppVersion();
        if (CleverTapAPI.getDebugLevel() > Constants.DEBUG_FINEST)
            Logger.logFine("Saving GCM registration ID on app version " + appVersion);
        SharedPreferences.Editor editor = prefs.edit();
        editor.putString(Constants.GCM_PROPERTY_REG_ID, regId);
        editor.putInt(PROPERTY_APP_VERSION, appVersion);
        StorageHelper.persist(editor);
    }

    String getCurrentActivityName() {
        if (currentActivity != null) {
            return currentActivity.getLocalClassName();
        }
        return null;
    }

    /**
     * Checks whether this notification is from CleverTap.
     *
     * @param extras The payload from the GCM intent
     * @return See {@link NotificationInfo}
     */
    public static NotificationInfo getNotificationInfo(final Bundle extras) {
        if (extras == null) return new NotificationInfo(false, false);

        boolean fromCleverTap = extras.containsKey(NOTIFICATION_TAG);
        boolean shouldRender = fromCleverTap && extras.containsKey("nm");
        return new NotificationInfo(fromCleverTap, shouldRender);
    }

    /**
     * Launches an asynchronous task to download the notification icon from CleverTap,
     * and create the Android notification.
     * <p/>
     * If your app is using CleverTap SDK's built in GCM message handling,
     * this method does not need to be called explicitly.
     * <p/>
     * Use this method when implementing your own GCM handling mechanism. Refer to the
     * SDK documentation for usage scenarios and examples.
     *
     * @param context A reference to an Android context
     * @param extras  The {@link Bundle} object received by the broadcast receiver
     */
    public static void createNotification(final Context context, final Bundle extras) {
        //noinspection ConstantConditions
        if (extras == null || extras.get(NOTIFICATION_TAG) == null) {
            return;
        }

        try {
            postAsyncSafely("CleverTapAPI#createNotification", new Runnable() {
                @Override
                public void run() {
                    try {
                        // Check if this is a test notification
                        if (extras.containsKey(Constants.DEBUG_KEY)
                                && "y".equals(extras.getString(Constants.DEBUG_KEY))) {
                            int r = (int) (Math.random() * 10);
                            if (r != 8) {
                                // Discard acknowledging this notif
                                return;
                            }
                            JSONObject event = new JSONObject();
                            try {
                                JSONObject actions = new JSONObject();
                                for (String x : extras.keySet()) {
                                    Object value = extras.get(x);
                                    actions.put(x, value);
                                }
                                event.put("evtName", "wzrk_d");
                                event.put("evtData", actions);
                                QueueManager.addToQueue(context, event, Constants.RAISED_EVENT);
                            } catch (JSONException ignored) {
                                // Won't happen
                            }
                            // Drop further processing
                            return;
                        }
                        String notifTitle = extras.getString("nt");
                        // If not present, set it to the app name
                        notifTitle = (notifTitle != null) ? notifTitle : context.getApplicationInfo().name;
                        String notifMessage = extras.getString("nm");
                        if (notifMessage == null) {
                            // What else is there to show then?
                            return;
                        }
                        String icoPath = extras.getString("ico");
                        Intent launchIntent;

                        if (extras.containsKey(Constants.DEEP_LINK_KEY)) {
                            launchIntent = new Intent(Intent.ACTION_VIEW, Uri.parse(extras.getString(Constants.DEEP_LINK_KEY)));
                        } else {
                            launchIntent = context.getPackageManager().getLaunchIntentForPackage(context.getPackageName());
                        }

                        PendingIntent pIntent;

                        // Take all the properties from the notif and add it to the intent
                        launchIntent.putExtras(extras);
                        launchIntent.putExtra(Constants.NOTIFICATION_RECEIVED_EPOCH_TAG, System.currentTimeMillis());
                        launchIntent.setFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP | Intent.FLAG_ACTIVITY_SINGLE_TOP);
                        pIntent = PendingIntent.getActivity(context, (int) System.currentTimeMillis(),
                                launchIntent, PendingIntent.FLAG_UPDATE_CURRENT);

                        NotificationCompat.Style style;
                        String bigPictureUrl = extras.getString("wzrk_bp");
                        if (bigPictureUrl != null && bigPictureUrl.startsWith("http")) {
                            try {
                                Bitmap bpMap = Utils.getNotificationBitmap(bigPictureUrl, false, context);

                                if (bpMap == null) throw new Exception("Failed to fetch big picture!");

                                style = new NotificationCompat.BigPictureStyle()
                                        .setSummaryText(notifMessage)
                                        .bigPicture(bpMap);
                            } catch (Throwable t) {
                                style = new NotificationCompat.BigTextStyle()
                                        .bigText(notifMessage);
                                Logger.error("Falling back to big text notification, couldn't fetch big picture", t);
                            }
                        } else {
                            style = new NotificationCompat.BigTextStyle()
                                    .bigText(notifMessage);
                        }

                        int smallIcon;
                        try {
                            String x = ManifestMetaData.getMetaData(context, Constants.LABEL_NOTIFICATION_ICON);
                            smallIcon = context.getResources().getIdentifier(x, "drawable", context.getPackageName());
                            if (smallIcon == 0) throw new IllegalArgumentException();
                        } catch (Throwable t) {
                            smallIcon = DeviceInfo.getAppIconAsIntId(context);
                        }

                        NotificationCompat.Builder nb = new NotificationCompat.Builder(context)
                                .setContentTitle(notifTitle)
                                .setContentText(notifMessage)
                                .setLargeIcon(Utils.getNotificationBitmap(icoPath, true, context))
                                .setContentIntent(pIntent)
                                .setAutoCancel(true)
                                .setStyle(style)
                                .setSmallIcon(smallIcon);

                        try {
                            if (extras.containsKey("wzrk_sound")) {
                                Object o = extras.get("wzrk_sound");
                                if ((o instanceof String && o.equals("true")) || (o instanceof Boolean && (Boolean) o)) {
                                    Uri defaultSoundUri = RingtoneManager.getDefaultUri(RingtoneManager.TYPE_NOTIFICATION);
                                    nb.setSound(defaultSoundUri);
                                }
                            }
                        } catch (Throwable t) {
                            Logger.error("Could not process sound parameter", t);
                        }

                        Notification n = nb.build();

                        NotificationManager notificationManager =
                                (NotificationManager) context.getSystemService(Context.NOTIFICATION_SERVICE);

                        notificationManager.notify((int) (Math.random() * 100), n);
                    } catch (Throwable t) {
                        // Occurs if the notification image was null
                        // Let's return, as we couldn't get a handle on the app's icon
                        // Some devices throw a PackageManager* exception too
                        Logger.error("Couldn't render notification!", t);
                    }
                }
            });
        } catch (Throwable t) {
            Logger.error("Failed to process GCM notification", t);
        }
    }

    /**
     * This method is used internally.
     */
    public void pushDeepLink(Uri uri) {
        pushDeepLink(uri, false);
    }

    synchronized void pushDeepLink(Uri uri, boolean install) {
        if (uri == null)
            return;

        try {
            JSONObject referrer = UriHelper.getUrchinFromUri(context, uri);
            referrer.put("referrer", uri.toString());
            if (install) {
                referrer.put("install", true);
            }
            event.pushDeviceDetailsWithExtras(referrer);
        } catch (Throwable t) {
            Logger.error("Failed to push deep link", t);
        }
    }

    public synchronized void pushInstallReferrer(String source, String medium, String campaign) {
        if (source == null && medium == null && campaign == null) return;
        try {
            // If already pushed, don't send it again
            int status = StorageHelper.getInt(context, "app_install_status", 0);
            if (status != 0) {
                Logger.log("Install referrer has already been set. Will not override it");
                return;
            }
            StorageHelper.putInt(context, "app_install_status", 1);

            if (source != null) source = Uri.encode(source);
            if (medium != null) medium = Uri.encode(medium);
            if (campaign != null) campaign = Uri.encode(campaign);

            String uriStr = "wzrk://track?install=true";
            if (source != null) uriStr += "&utm_source=" + source;
            if (medium != null) uriStr += "&utm_medium=" + medium;
            if (campaign != null) uriStr += "&utm_campaign=" + campaign;

            Uri uri = Uri.parse(uriStr);
            pushDeepLink(uri, true);
        } catch (Throwable t) {
            Logger.error("Failed to push install referrer", t);
        }
    }

    private static final HashMap<String, Integer> installReferrerMap = new HashMap<String, Integer>(8);

    public void enablePersonalization() {
        LocalDataStore.setPersonalisationEnabled(context, true);
    }

    public void disablePersonalization() {
        LocalDataStore.setPersonalisationEnabled(context, false);
    }

    public void pushInstallReferrer(Intent intent) {
        try {
            final Bundle extras = intent.getExtras();
            // Preliminary checks
            if (extras == null || !extras.containsKey("referrer")) {
                return;
            }
            final String url;
            try {
                url = URLDecoder.decode(extras.getString("referrer"), "UTF-8");

                Logger.logFine("Referrer received: " + url);
            } catch (Throwable e) {
                // Could not decode
                return;
            }
            if (url == null) {
                return;
            }
            int now = (int) (System.currentTimeMillis() / 1000);

            if (installReferrerMap.containsKey(url) && now - installReferrerMap.get(url) < 10) {
                Logger.logFine("Skipping install referrer due to duplicate within 10 seconds");
                return;
            }

            installReferrerMap.put(url, now);

            Uri uri = Uri.parse("wzrk://track?install=true&" + url);

            pushDeepLink(uri, true);
        } catch (Throwable t) {
            // Weird
        }
    }

    public void updateLocation(Location location) {
        locationFromUser = location;
        Logger.logFine("Location updated (" + location.getLatitude() + ", " + location.getLongitude() + ")");

        // Queue the ping event to transmit location update to server
        // min 10 second interval between location pings
        final int now = (int) (System.currentTimeMillis() / 1000);
        if (now > (lastLocationPingTime + Constants.LOCATION_PING_INTERVAL_IN_SECONDS)) {
            QueueManager.addToQueue(context, new JSONObject(), Constants.PING_EVENT);
            lastLocationPingTime = now;
            Logger.logFine("Queuing location ping event for location (" + location.getLatitude() + ", " + location.getLongitude() + ")");
        }
    }

    private Location getLocation() {
        if (locationFromUser != null) {
            return locationFromUser;
        }

        // Respect disable location privacy
        boolean tryLocation = true;
        try {
            String value = ManifestMetaData.getMetaData(context, Constants.LABEL_PRIVACY_MODE);
            if (value.contains(Constants.PRIVACY_MODE_DISABLE_LOCATION)) {
                tryLocation = false;
            }
        } catch (Throwable t) {
            // Okay cool, so the default behaviour is to capture location
        }

        if (!tryLocation) return null;

        try {
            LocationManager lm = (LocationManager) context.getSystemService(Context.LOCATION_SERVICE);
            List<String> providers = lm.getProviders(true);
            Location bestLocation = null;
            for (String provider : providers) {
                Location l = lm.getLastKnownLocation(provider);
                if (l == null) {
                    continue;
                }
                if (bestLocation == null || l.getAccuracy() < bestLocation.getAccuracy()) {
                    bestLocation = l;
                }
            }
            return bestLocation;
        } catch (Throwable t) {
            Logger.logFine("Couldn't get user's location", t);
            return null;
        }
    }

    Context getContext() {
        return context;
    }

    /**
     * Returns a unique identifier by which CleverTap identifies this user.
     *
     * @return The user identifier if available, else null.
     */

    public String getCleverTapID() {
        return DeviceInfo.getDeviceID(context);
    }
}
