/*
 * Author: Jude Pereira
 * Copyright (c) 2014
 */

package com.clevertap.android.sdk;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.SharedPreferences;
import org.json.JSONArray;
import org.json.JSONObject;

import java.util.ArrayList;

import static com.clevertap.android.sdk.Constants.SESSION_LAST_ACTIVITY_TRAIL;
import static com.clevertap.android.sdk.Constants.SESSION_LAST_PING_EPOCH;

/**
 * Provides various methods to manipulate the current session.
 */
final class SessionManager {
    private static PingEntity pingEntity = null;
    private static Runnable pingTickerRunnable = null;
    private static int currentElapsedPingSeconds = 0;
    private static int currentSessionId = 0;
    private static boolean firstSession = false;
    private static boolean appLaunchedBeenPushed = true;
    private static int lastSessionLength = 0;
    private static final ArrayList<String> activityList = new ArrayList<String>();
    private static JSONArray lastSessionActivityTrail = null;

    private static final Runnable syncLastTickRunnable = new Runnable() {
        @Override
        public void run() {
            final int now = (int) (System.currentTimeMillis() / 1000);

            // Sync session time
            try {
                // Can pass context as null because the instance is already initialized
                StorageHelper.putInt(CleverTapAPI.getInstance(null).getContext(), SESSION_LAST_PING_EPOCH, now);
                Logger.logFine("Synced last ping time");
            } catch (Throwable t) {
                Logger.logFine("Failed to sync last ping time: " + t.getMessage());
            }

            // Sync navigation data
            try {
                JSONArray arr = new JSONArray();
                for (String s : activityList) {
                    arr.put(s);
                }

                // Can pass context as null because the instance is already initialized
                StorageHelper.putString(CleverTapAPI.getInstance(null).getContext(), SESSION_LAST_ACTIVITY_TRAIL, arr.toString());
                Logger.logFine("Synced activity trail");
            } catch (Throwable t) {
                Logger.logFine("Failed to sync activity trail: " + t.getMessage());
            }
        }
    };

    static void activityChanged(final String newActivityName) {
        if (newActivityName == null || newActivityName.trim().equals("")) {
            return;
        }

        // Don't worry about syncing, as it's synced in the syncLastTickRunnable cycle
        activityList.add(newActivityName);
    }

    public static int getLastSessionLength() {
        return lastSessionLength;
    }

    static boolean isAppLaunchedBeenPushed() {
        return appLaunchedBeenPushed;
    }

    static void setAppLaunchedBeenPushed(boolean appLaunchedBeenPushed) {
        SessionManager.appLaunchedBeenPushed = appLaunchedBeenPushed;
    }

    static boolean isFirstSession() {
        return firstSession;
    }

    /**
     * Destroys the current session, a new one will NOT be created..
     */
    static void destroySession() {
        Logger.logFine("Session destroyed; Session ID is now 0");
        InAppFCManager.destroySession();
        currentSessionId = 0;
        SessionHandler.setSource(null);
        SessionHandler.setMedium(null);
        SessionHandler.setCampaign(null);
        activityList.clear();
    }

    /**
     * Returns the current session ID.
     *
     * @return The current session ID
     */
    static int getCurrentSession() {
        return currentSessionId;
    }

    @SuppressLint("CommitPrefEdits")
    static int createSession(final Context context) {
        currentSessionId = (int) (System.currentTimeMillis() / 1000);

        Logger.logFine("Session created with ID: " + currentSessionId);

        SharedPreferences prefs = StorageHelper.getPreferences(context);

        final int lastSessionID = prefs.getInt(Constants.SESSION_ID_LAST, 0);
        final int lastSessionPingEpoch = prefs.getInt(SESSION_LAST_PING_EPOCH, 0);
        if (lastSessionPingEpoch > 0) {
            lastSessionLength = lastSessionPingEpoch - lastSessionID;
        }

        Logger.logFine("Last session length: " + lastSessionLength + " seconds");

        if (lastSessionID == 0) {
            firstSession = true;
        }

        try {
            final String activityTrail = prefs.getString(Constants.SESSION_LAST_ACTIVITY_TRAIL, "[]");
            lastSessionActivityTrail = new JSONArray(activityTrail);
            Logger.logFine("Last activity trail: " + lastSessionActivityTrail.toString());
        } catch (Throwable t) {
            // Ignore
            Logger.logFine("Last activity trail not found");
        }

        final SharedPreferences.Editor editor = prefs.edit().putInt(Constants.SESSION_ID_LAST, currentSessionId);
        StorageHelper.persist(editor);

        // Attach the ping timer to this session
        resetPingTimer(context);

        setAppLaunchedBeenPushed(false);
        return currentSessionId;
    }

    /**
     * Destroys any existing ping timer, and starts a new one.
     *
     * @param context The Android context
     */
    private static void resetPingTimer(final Context context) {
        currentElapsedPingSeconds = 0;

        if (pingTickerRunnable != null) {
            CleverTapAPI.getHandlerUsingMainLooper().removeCallbacks(pingTickerRunnable);
        }

        // Create a new ping entity object and start the timer for it
        pingEntity = new PingEntity();
        if (pingTickerRunnable == null) {
            pingTickerRunnable = new Runnable() {
                @Override
                public void run() {
                    if (pingEntity == null) return;

                    // Perform a tick only if the app is in foreground
                    if (CleverTapAPI.isAppForeground()) {
                        pingEntity.doTick();

                        // Push the last ping sync-er guy ahead
                        CleverTapAPI.getHandlerUsingMainLooper().removeCallbacks(syncLastTickRunnable);
                        CleverTapAPI.getHandlerUsingMainLooper().postDelayed(syncLastTickRunnable, Constants.PING_TICK_INTERVAL + 1000);
                    }

                    // Set the last ping epoch here
                    // The reason it's here and not in CleverTapAPI#setAppForeground is because
                    // activities need to transition quickly - shouldn't write to shared preferences there
                    final int now = (int) (System.currentTimeMillis() / 1000);
                    StorageHelper.putInt(context, SESSION_LAST_PING_EPOCH, now);

                    // Check if our timer has matured - every 30 seconds
                    // 5 seconds during the very first session, 30 seconds for all other
                    final int duration;
                    if (isFirstSession()) {
                        duration = Constants.PING_INTERVAL_FIRST_SESSION_IN_SECONDS;
                    } else {
                        duration = Constants.PING_INTERVAL_IN_SECONDS;
                    }

                    if (pingEntity.hasMatured(currentElapsedPingSeconds + duration)) {
                        currentElapsedPingSeconds += duration;
                        Logger.logFine("Ping timer has matured. Firing ping event. Elapsed=" + currentElapsedPingSeconds);

                        // Queue the ping event
                        QueueManager.addToQueue(context, new JSONObject(), Constants.PING_EVENT);
                    }

                    // Schedule the next call
                    if (pingEntity != null) {
                        CleverTapAPI.getHandlerUsingMainLooper()
                                .postDelayed(pingTickerRunnable, Constants.PING_TICK_INTERVAL);
                    }
                }
            };
        }
        // Schedule the first tick
        CleverTapAPI.getHandlerUsingMainLooper().postDelayed(pingTickerRunnable, Constants.PING_TICK_INTERVAL);

        Logger.logFine("New ping entity associated with this session");
    }

    public static JSONArray getLastSessionActivityTrail() {
        if (lastSessionActivityTrail == null) return new JSONArray();

        return lastSessionActivityTrail;
    }
}
