/*
 * Author: Jude Pereira
 * Copyright (c) 2014
 */

package com.clevertap.android.sdk;

import java.io.UnsupportedEncodingException;
import java.util.Date;

/**
 * Provides methods to validate various entities.
 */
final class Validator {
    private static final String[] eventNameCharsNotAllowed = {".", ":", "$", "'", "\"", "\\"};
    private static final String[] objectKeyCharsNotAllowed = {".", ":", "$", "'", "\"", "\\"};
    private static final String[] objectValueCharsNotAllowed = {"'", "\"", "\\"};
    private static final String[] restrictedNames = {"Stayed", "Notification Clicked",
            "Notification Viewed", "UTM Visited", "Notification Sent", "App Launched", "wzrk_d",
            "App Uninstalled", "Notification Bounced"};

    /**
     * Cleans the event name to the following guidelines:
     * <p>
     * The following characters are removed:
     * dot, colon, dollar sign, single quote, double quote, and backslash.
     * Additionally, the event name is limited to 32 characters.
     * </p>
     *
     * @param name The event name to be cleaned
     * @return The {@link ValidationResult} object containing the object,
     * and the error code(if any)
     */
    static ValidationResult cleanEventName(String name) {
        ValidationResult vr = new ValidationResult();

        name = name.trim();
        for (String x : eventNameCharsNotAllowed)
            name = name.replace(x, "");

        if (name.length() > 32) {
            name = name.substring(0, 31);
            vr.setErrorDesc(name.trim() + "... exceeds the limit of 32 characters. Trimmed");
            vr.setErrorCode(510);
        }

        vr.setObject(name.trim());
        return vr;
    }

    /**
     * Cleans the object key.
     *
     * @param name Name of the object key
     * @return The {@link ValidationResult} object containing the object,
     * and the error code(if any)
     */
    static ValidationResult cleanObjectKey(String name) {
        ValidationResult vr = new ValidationResult();
        name = name.trim();
        for (String x : objectKeyCharsNotAllowed)
            name = name.replace(x, "");

        if (name.length() > 32) {
            name = name.substring(0, 31);
            vr.setErrorDesc(name.trim() + "... exceeds the limit of 32 characters. Trimmed");
            vr.setErrorCode(520);
        }

        vr.setObject(name.trim());

        return vr;
    }

    /**
     * Cleans the object value, only if it is a string, otherwise, it simply returns the object.
     * <p/>
     * It also accepts a {@link java.util.Date} object, and converts it to a CleverTap
     * specific date format.
     * <p/>
     * In addition, if the object represents a number in a string format, it will convert
     * and return an Integer object.
     *
     * @param o Object to be cleaned(only if it is a string)
     * @return The cleaned object
     */
    static ValidationResult cleanObjectValue(Object o, boolean useOldDate, boolean autoConvert)
            throws IllegalArgumentException {
        ValidationResult vr = new ValidationResult();
        // If it's any type of number, send it back
        if (o instanceof Integer
                || o instanceof Float
                || o instanceof Boolean
                || o instanceof Double) {
            // No support for double
            if (o instanceof Double) {
                vr.setObject(((Double) o).floatValue());
            } else {
                vr.setObject(o);
            }
            return vr;
        } else if (o instanceof Long) {
            if (!autoConvert) {
                // This signifies that it's an event attribute - no long support here
                vr.setObject(o.toString());
            } else {
                vr.setObject(o);
            }
            return vr;
        } else if (o instanceof String || o instanceof Character) {
            String value;
            if (o instanceof Character)
                value = String.valueOf(o);
            else
                value = (String) o;
            value = value.trim();
            if (autoConvert) {
                // Try to convert the value to a long first, if not,
                // then treat it as a string and continue validation
                try {
                    // Return a Long object
                    Long i = Long.parseLong(value);
                    if (i.toString().equals(value)) {
                        // It's an integer alright
                        vr.setObject(i);
                        return vr;
                    }
                } catch (NumberFormatException e) {
                    // Okay, proceed
                }

                // Try to convert the value to an double first, if not,
                // then treat it as a string and continue validation
                try {
                    // Return a float object
                    Double d = Double.parseDouble(value);
                    // No support for double, hence float - loss of precision - OK
                    vr.setObject(d.floatValue());
                    return vr;
                } catch (NumberFormatException e) {
                    // Okay, proceed
                }

                // Try to convert the value to an boolean first, if not,
                // then treat it as a string and continue validation
                try {
                    // Return an Boolean object
                    if (value.equalsIgnoreCase("true"))
                        vr.setObject(true);
                    else if (value.equalsIgnoreCase("false"))
                        vr.setObject(false);
                    else
                        throw new Exception();
                    return vr;
                } catch (Exception e) {
                    // Okay, proceed
                }
            }

            for (String x : objectValueCharsNotAllowed) {
                value = value.replace(x, "");
            }

            try {
                if (value.getBytes("UTF-8").length > 120) {
                    value = fastTrim(value, 120);
                    vr.setErrorDesc(value.trim() + "... exceeds the limit of 120 bytes. Trimmed");
                    vr.setErrorCode(521);
                }
            } catch (UnsupportedEncodingException ignore) {
                // We really shouldn't get here
                // Ignore
            }
            vr.setObject(value.trim());
            return vr;
        } else if (o instanceof Date) {
            String date;
            if (useOldDate) {
                date = "$D_" + Constants.DOB_DATE_FORMAT.format(((Date) o));
            } else {
                date = "$D_" + ((Date) o).getTime() / 1000;
            }
            vr.setObject(date);
            return vr;
        } else {
            throw new IllegalArgumentException("Not a String, Boolean, Long, Integer, Float, Double, or Date");
        }
    }

    /**
     * Checks if the phone number is less than 8 characters.
     *
     * @param value The object(usually a {@link String} containing the phone number
     */
    static void validatePhone(Object value) {
        try {
            if (value instanceof Number) {
                if (value.toString().length() >= 8) {
                    return;
                }
            }
        } catch (Throwable e) {
            // Ignore
        }
        Logger.error("Invalid phone number specified - " + value);
        throw new IllegalArgumentException("Invalid phone number");
    }

    /**
     * Checks whether the specified event name is restricted. If it is,
     * then create a pending error, and abort.
     *
     * @param name The event name
     * @return Boolean indication whether the event name is restricted
     */
    static boolean isRestrictedEventName(String name) {
        if (name == null) return false;
        for (String x : restrictedNames)
            if (name.equalsIgnoreCase(x)) {
                // The event name is restricted
                ValidationResult error = new ValidationResult();
                error.setErrorCode(513);
                error.setErrorDesc(name + " is a restricted event name. Last event aborted.");
                CleverTapAPI.pendingValidationResult = error;
                Logger.error(name + " is a restricted system event name. Last event aborted.");
                return true;
            }
        return false;
    }

    static String fastTrim(String input, int byteLen) {
        try {
            byte[] data = input.getBytes("UTF-8");
            if (data.length <= byteLen) return input;

            int total = 0;
            //noinspection ForLoopReplaceableByForEach
            for (int i = 0; i < data.length; i++) {
                int charc = 0;
                if (data[i] >= 0) {
                    charc = 1;
                } else {
                    int mask = data[i] & 0xFF;
                    if (mask >> 4 == 0x0F) { /* 4 bytes */
                        charc = 4;
                    } else if (mask >> 5 == 0x07) { /* 3 bytes */
                        charc = 3;
                    } else if (mask >> 6 == 0x03) { /* 2 bytes */
                        charc = 2;
                    }
                }

                if (total + charc <= byteLen) {
                    total += charc;
                } else {
                    break;
                }
            }

            if (total == data.length) return input;

            byte[] out = new byte[total];
            System.arraycopy(data, 0, out, 0, out.length);

            return new String(out, "UTF-8");
        } catch (UnsupportedEncodingException ex) {
            return "";
        }
    }
}
