/*
 * Author: Jude Pereira
 * Copyright (c) 2014
 */

package com.clevertap.android.sdk;

import android.accounts.Account;
import android.accounts.AccountManager;
import android.content.Context;
import android.telephony.TelephonyManager;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.text.ParseException;
import java.util.*;

/**
 * Handles all push events which describe a user's profile.
 */
public final class ProfileHandler {
    private final Context context;
    private static final String ENUM_PREFIX = "$E_";

    /**
     * Please do not create an object of this explicitly.
     *
     * @param context The Android context
     * @see CleverTapAPI
     */
    ProfileHandler(Context context) {
        this.context = context;
    }

    private enum KnownFields {
        Name(Constants.USER_NAME), Email(Constants.USER_EMAIL), Education(Constants.USER_EDUCATION),
        Married(Constants.USER_RELATIONSHIP_STATUS), DOB(Constants.USER_DOB), Gender(Constants.USER_GENDER),
        Phone(Constants.USER_PHONE), Age(Constants.USER_AGE);

        private final String storageValue;

        KnownFields(String storageValue) {
            this.storageValue = storageValue;
        }
    }

    /**
     * Push a profile update.
     *
     * @param profile A {@link Map}, with keys as strings, and values as {@link String},
     *                {@link Integer}, {@link Long}, {@link Boolean}, {@link Float}, {@link Double},
     *                {@link java.util.Date}, or {@link Character}
     */
    public void push(Map<String, Object> profile) {
        if (profile == null || profile.isEmpty())
            return;
        try {
            ValidationResult vr;
            JSONObject customProfile = new JSONObject();
            for (String key : profile.keySet()) {
                Object value = profile.get(key);

                vr = Validator.cleanObjectKey(key);
                key = vr.getObject().toString();
                // Check for an error
                if (vr.getErrorCode() != 0) {
                    CleverTapAPI.pendingValidationResult = vr;
                }

                try {
                    vr = Validator.cleanObjectValue(value, true, true);
                } catch (Exception e) {
                    // The object was neither a String, Boolean, or any number primitives
                    ValidationResult error = new ValidationResult();
                    error.setErrorCode(512);
                    error.setErrorDesc("Object value wasn't a primitive. Last event aborted.");
                    CleverTapAPI.pendingValidationResult = error;
                    Logger.log("Object value isn't a primitive. Profile update aborted");
                    // Abort
                    return;
                }
                value = vr.getObject();
                // Check for an error
                if (vr.getErrorCode() != 0) {
                    CleverTapAPI.pendingValidationResult = vr;
                }

                // Check for known keys
                if (key.equalsIgnoreCase("Phone")) {
                    try {
                        Validator.validatePhone(value);
                        value = ((Number) value).longValue();
                    } catch (Exception e) {
                        CleverTapAPI.pendingValidationResult = new ValidationResult(512, "Invalid phone number");
                        Logger.log("Invalid phone number");
                        continue;
                    }
                }

                if (key.equalsIgnoreCase("Age")) {
                    if (value instanceof Integer) {
                        int age = (Integer) value;
                        if (!(age > 0 && age < 120)) {
                            Logger.log("Invalid age supplied");
                            CleverTapAPI.pendingValidationResult = new ValidationResult(512, "Invalid age");
                            continue;
                        }
                    } else {
                        Logger.log("Age looks to be of an unsupported data type");
                        CleverTapAPI.pendingValidationResult = new ValidationResult(512, "Invalid age (unknown data type)");
                        continue;
                    }
                }
                try {
                    KnownFields kf = KnownFields.valueOf(key);
                    if (kf == null) throw new Exception();

                    keepProperty(kf.storageValue, value);
                } catch (Throwable t) {
                    customProfile.put(key, value);
                }
            }
            Logger.logFine("Constructed custom profile: " + customProfile.toString());
            pushBasicProfile(customProfile);
        } catch (Throwable t) {
            // Will not happen
            Logger.error("Failed to push profile", t);
        }
    }

    private String getGraphUserPropertySafely(JSONObject graphUser, String key, String def) {
        try {
            String prop = (String) graphUser.get(key);
            if (prop != null)
                return prop;
            else
                return def;
        } catch (Throwable t) {
            return def;
        }
    }

    /**
     * Pushes everything available in the JSON object returned by the Facebook GraphRequest
     *
     * @param graphUser The object returned from Facebook
     */
    public void pushFacebookUser(JSONObject graphUser) {
        try {
            if (graphUser == null)
                return;
            // Note: No validations are required here, as everything is controlled
            String name = getGraphUserPropertySafely(graphUser, "name", "");
            try {
                // Certain users have nasty looking names - unicode chars, validate for any
                // not allowed chars
                ValidationResult vr = Validator.cleanObjectValue(name, false, false);
                name = vr.getObject().toString();

                if (vr.getErrorCode() != 0) {
                    CleverTapAPI.pendingValidationResult = vr;
                }
            } catch (IllegalArgumentException e) {
                // Weird name, wasn't a string, or any number
                // This would never happen with FB
                name = "";
            }

            String gender = getGraphUserPropertySafely(graphUser, "gender", null);
            // Convert to WR format
            if (gender != null) {
                if (gender.toLowerCase().startsWith("m"))
                    gender = "M";
                else if (gender.toLowerCase().startsWith("f"))
                    gender = "F";
                else
                    gender = "";
            } else {
                gender = null;
            }
            String email = getGraphUserPropertySafely(graphUser, "email", "");

            String birthday = getGraphUserPropertySafely(graphUser, "birthday", null);
            if (birthday != null) {
                // Some users don't have the year of birth mentioned
                // FB returns only MM/dd in those cases
                if (birthday.matches("^../..")) {
                    // This means that the year is not available(~30% of the times)
                    // Ignore
                    birthday = "";
                } else {
                    try {
                        Date date = Constants.FB_DOB_DATE_FORMAT.parse(birthday);
                        // WR format = $D_yyyyMMdd
                        birthday = "$D_" + Constants.DOB_DATE_FORMAT.format(date);
                    } catch (ParseException e) {
                        // Differs from the specs
                        birthday = "";
                    }
                }
            }

            String work;
            try {
                JSONArray workArray = graphUser.getJSONArray("work");
                work = (workArray.length() > 0) ? "Y" : "N";
            } catch (Throwable t) {
                work = "";
            }

            String education;
            try {
                JSONArray eduArray = graphUser.getJSONArray("education");
                // FB returns the education levels in a descending order - highest = last entry
                String fbEdu = eduArray.getJSONObject(eduArray.length() - 1).getString("type");
                if (fbEdu.toLowerCase().contains("high school"))
                    education = "School";
                else if (fbEdu.toLowerCase().contains("college"))
                    education = "College";
                else if (fbEdu.toLowerCase().contains("graduate school"))
                    education = "Graduate";
                else
                    education = "";
            } catch (Throwable t) {
                // No education info available
                education = "";
            }

            String id = getGraphUserPropertySafely(graphUser, "id", "");

            String married = getGraphUserPropertySafely(graphUser, "relationship_status", null);
            if (married != null) {
                if (married.equalsIgnoreCase("married")) {
                    married = "Y";
                } else {
                    married = "N";
                }
            }

            keepProperty(Constants.FB_ID, id);
            keepProperty(Constants.FB_NAME, name);
            keepProperty(Constants.FB_EMAIL, email);
            keepProperty(Constants.FB_GENDER, gender);
            keepProperty(Constants.FB_EDUCATION, education);
            keepProperty(Constants.FB_EMPLOYED, work);
            keepProperty(Constants.FB_DOB, birthday);
            keepProperty(Constants.FB_RELATIONSHIP_STATUS, married);

            pushBasicProfile(null);
        } catch (Throwable t) {
            Logger.error("Failed to parse graph user object successfully", t);
        }
    }

    private void keepProperty(String key, Object value) {
        if (key == null || key.equals("")
                || value == null) {
            return;
        }
        if (value instanceof String) {
            StorageHelper.putString(context, key, (String) value);
        } else if (value instanceof Integer) {
            StorageHelper.putInt(context, key, (Integer) value);
        } else if (value instanceof Long) {
            StorageHelper.putLong(context, key, (Long) value);
        } else {
            Logger.log("Ignored value due to unsupported type: " + value.getClass().getName());
        }
    }

    /**
     * Pushes basic user information, if available.
     * <p/>
     * Basic information is pushed daily. There is no need to call this method
     * explicitly.
     */
    void pushBasicProfile(JSONObject baseProfile) {
        try {
            JSONObject profileEvent = new JSONObject();

            if (baseProfile != null && baseProfile.length() > 0) {
                Iterator i = baseProfile.keys();
                while (i.hasNext()) {
                    String next = i.next().toString();
                    try {
                        profileEvent.put(next, baseProfile.get(next));
                    } catch (JSONException e) {
                        // Ignore
                    }
                }
            }

            try {
                Account account = getGoogleAccount();
                if (account != null && account.name != null) {
                    ValidationResult vr;
                    try {
                        vr = Validator.cleanObjectValue(account.name, false, false);
                        keepProperty(Constants.SYSTEM_EMAIL, vr.getObject().toString());
                    } catch (IllegalArgumentException e) {
                        // The object was neither a String, or any number primitives
                        // Ignore
                    }
                }

                String carrier = getCarrier();
                if (carrier != null && !carrier.equals("")) {
                    keepProperty(Constants.SYSTEM_CARRIER, carrier);
                }

                String cc = getCountryCode();
                if (cc != null && !cc.equals("")) {
                    keepProperty(Constants.SYSTEM_COUNTRY_CODE, cc);
                }

                // Add the user's timezone
                keepProperty(Constants.SYSTEM_TIMEZONE, TimeZone.getDefault().getID());

                addKeptProfileProperties(context, profileEvent);

                // Attempt to send all other email addresses too
                try {
                    JSONArray allAccounts = getAllAccounts();
                    if (allAccounts != null && allAccounts.length() > 0) {
                        profileEvent.put("Alternate Emails", allAccounts);
                    }
                } catch (Throwable t) {
                    // Ignore
                }

                JSONObject event = new JSONObject();
                event.put("profile", profileEvent);
                QueueManager.addToQueue(context, event, Constants.PROFILE_EVENT);
            } catch (JSONException e) {
                // We won't get here
                Logger.logFine("FATAL: Creating basic profile update event failed!");
            }
        } catch (Throwable t) {
            Logger.error("Basic profile sync", t);
        }
    }

    private void addPropertyFromStoreIfExists(Context context, String profileEventKey, JSONObject profile, DataType dataType, String... preferenceKeys) {
        for (String pk : preferenceKeys) {
            try {
                Object value;
                switch (dataType) {
                    case Integer:
                        try {
                            value = StorageHelper.getInt(context, pk, Integer.MIN_VALUE);
                            if ((Integer) value == Integer.MIN_VALUE) continue;
                        } catch (ClassCastException c) {
                            value = StorageHelper.getLong(context, pk, Long.MIN_VALUE);
                            if ((Long) value == Long.MIN_VALUE) continue;
                        }
                        break;
                    case Long:
                        try {
                            value = StorageHelper.getLong(context, pk, Long.MIN_VALUE);
                            if ((Long) value == Long.MIN_VALUE) continue;
                        } catch (ClassCastException c) {
                            value = StorageHelper.getInt(context, pk, Integer.MIN_VALUE);
                            if ((Integer) value == Integer.MIN_VALUE) continue;
                        }
                        break;
                    case String:
                        value = StorageHelper.getString(context, pk, null);
                        if (value == null) continue;
                        break;
                    default:
                        value = null;
                }

                //noinspection ConstantConditions
                if (value == null) continue;

                try {
                    profile.put(profileEventKey, value);
                    return;
                } catch (JSONException e) {
                    // Ignore
                }
            } catch (Exception e) {
                // Ignore
            }
        }
    }

    /**
     * Adds the stored user populated/FB/G+ data, if available.
     * Preference is given to user populated, followed by FB, and then G+, if all are available.
     * <p/>
     * Adds:
     * 1. Name
     * 2. Gender
     * 3. Education
     * 4. Employed
     * 5. Married
     * 6. DOB
     * 7. FBID
     * 8. GPID
     * 10. Phone
     * 11. Age
     * 12. Email
     * 13. tz
     * 14. Carrier
     * 15. cc
     */
    private void addKeptProfileProperties(Context context, JSONObject profile) {
        addPropertyFromStoreIfExists(context, "Name", profile, DataType.String, Constants.USER_NAME, Constants.FB_NAME, Constants.GP_NAME);
        addPropertyFromStoreIfExists(context, "Gender", profile, DataType.String, Constants.USER_GENDER, Constants.FB_GENDER, Constants.GP_GENDER);
        addPropertyFromStoreIfExists(context, "Education", profile, DataType.String, Constants.USER_EDUCATION, Constants.FB_EDUCATION);
        addPropertyFromStoreIfExists(context, "Employed", profile, DataType.String, Constants.USER_EMPLOYED, Constants.FB_EMPLOYED, Constants.GP_EMPLOYED);
        addPropertyFromStoreIfExists(context, "Married", profile, DataType.String, Constants.USER_RELATIONSHIP_STATUS, Constants.FB_RELATIONSHIP_STATUS, Constants.GP_RELATIONSHIP_STATUS);
        addPropertyFromStoreIfExists(context, "DOB", profile, DataType.String, Constants.USER_DOB, Constants.FB_DOB, Constants.GP_DOB);
        addPropertyFromStoreIfExists(context, "FBID", profile, DataType.String, Constants.FB_ID);
        addPropertyFromStoreIfExists(context, "GPID", profile, DataType.String, Constants.GP_ID);
        addPropertyFromStoreIfExists(context, "Phone", profile, DataType.Long, Constants.USER_PHONE);
        addPropertyFromStoreIfExists(context, "Age", profile, DataType.Integer, Constants.USER_AGE);
        addPropertyFromStoreIfExists(context, "Email", profile, DataType.String, Constants.USER_EMAIL, Constants.FB_EMAIL, Constants.SYSTEM_EMAIL);
        addPropertyFromStoreIfExists(context, "tz", profile, DataType.String, Constants.SYSTEM_TIMEZONE);
        addPropertyFromStoreIfExists(context, "Carrier", profile, DataType.String, Constants.SYSTEM_CARRIER);
        addPropertyFromStoreIfExists(context, "cc", profile, DataType.String, Constants.SYSTEM_COUNTRY_CODE);

        // add the secondary email
        addSecondaryEmail(context, profile);
    }

    private void addSecondaryEmail(Context context, JSONObject profile) {
        try {
            String originalEmail = profile.getString("Email");
            if (originalEmail == null || originalEmail.equals("")) {
                addPropertyFromStoreIfExists(context, "Email", profile, DataType.String, Constants.FB_EMAIL);
            } else {
                String email2 = StorageHelper.getString(context, Constants.FB_EMAIL, "");
                if (email2 == null || email2.equals(originalEmail) || email2.equals("")) {
                    return;
                }
                profile.put("Email2", email2);
            }
        } catch (JSONException ignore) {
            // Ignore
        }
    }

    /**
     * Pushes an enum type of property for a user.
     * <p/>
     * Enum types show up in the CleverTap dashboard, when filtering down to a certain
     * group of people.
     * <p/>
     * You can have up to four enums, with eight values for each of them.
     * <p/>
     * For example, if you have different levels amongst your customer,
     * say, Gold, Silver, and Platinum, then you could use the key, "Type",
     * and the value corresponding to the type of customer, i.e. Gold, Silver, or Platinum.
     *
     * @param key   The enum key
     * @param value The value for this profile
     */
    @Deprecated
    public void pushEnum(String key, String value) {
        JSONObject enumEvent = new JSONObject();
        try {
            ValidationResult vr;
            vr = Validator.cleanObjectKey(key);
            key = vr.getObject().toString();
            if (key == null || key.equals(""))
                // Abort
                return;

            vr = Validator.cleanObjectValue(value, false, false);
            value = vr.getObject().toString();
            if (value == null || value.equals(""))
                // Abort
                return;

            enumEvent.put(key, ENUM_PREFIX + value);

            JSONObject event = new JSONObject();
            event.put("profile", enumEvent);
            QueueManager.addToQueue(context, event, Constants.PROFILE_EVENT);
        } catch (Throwable e) {
            // We won't get here
            Logger.logFine("FATAL: Creating enum for profile update event failed!");
        }
    }

    private String getCarrier() {
        try {
            TelephonyManager tm = (TelephonyManager) context.getSystemService(Context.TELEPHONY_SERVICE);
            String carrier = tm.getSimOperatorName();
            ValidationResult vr = Validator.cleanObjectValue(carrier, false, false);
            if (vr.getErrorCode() == 0)
                return (String) vr.getObject();
        } catch (Throwable ignore) {
        }
        return "";
    }

    private Account getGoogleAccount() {
        if (context == null) return null;

        // Respect disable email privacy
        try {
            String value = ManifestMetaData.getMetaData(context, Constants.LABEL_PRIVACY_MODE);
            if (value.contains(Constants.PRIVACY_MODE_DISABLE_EMAIL)) {
                return null;
            }
        } catch (Throwable t) {
            // Okay cool, so the default behaviour is to capture Google account
        }

        try {
            AccountManager manager = AccountManager.get(context);
            Account[] accounts = manager.getAccountsByType("com.google");
            if (accounts != null && accounts.length > 0)
                return accounts[0];
            else
                return null;
        } catch (Throwable ignore) {
            return null;
        }
    }

    private JSONArray getAllAccounts() {
        if (context == null) return null;

        // Respect disable email privacy
        try {
            String value = ManifestMetaData.getMetaData(context, Constants.LABEL_PRIVACY_MODE);
            if (value.contains(Constants.PRIVACY_MODE_DISABLE_EMAIL)) {
                return null;
            }
        } catch (Throwable t) {
            // Okay cool, so the default behaviour is to capture Google account
        }

        try {
            final JSONArray all = new JSONArray();
            AccountManager am = AccountManager.get(context);
            Account[] accounts = am.getAccounts();
            for (Account account : accounts) {
                final String name = account.name;
                if (name != null && name.contains("@")) {
                    all.put(name);
                }
            }
            return all;
        } catch (Throwable t) {
            return null;
        }
    }

    private String getCountryCode() {
        try {
            TelephonyManager tm = (TelephonyManager) context.getSystemService(Context.TELEPHONY_SERVICE);
            return tm.getSimCountryIso();
        } catch (Throwable ignore) {
            return "";
        }
    }

    /**
     * Pushes everything useful within the Google Plus
     * {@link com.google.android.gms.plus.model.people.Person} object.
     *
     * @param person The {@link com.google.android.gms.plus.model.people.Person} object
     * @see com.google.android.gms.plus.model.people.Person
     */
    public void pushGooglePlusPerson(com.google.android.gms.plus.model.people.Person person) {
        if (person == null) {
            return;
        }
        try {
            // Note: No validations are required here, as everything is controlled
            String name = "";
            if (person.hasDisplayName()) {
                try {
                    // Certain users have nasty looking names - unicode chars, validate for any
                    // not allowed chars
                    name = person.getDisplayName();
                    ValidationResult vr = Validator.cleanObjectValue(name, false, false);
                    name = vr.getObject().toString();

                    if (vr.getErrorCode() != 0) {
                        CleverTapAPI.pendingValidationResult = vr;
                    }
                } catch (Throwable t) {
                    // Weird name, wasn't a string, or any number
                    // This would never happen with G+
                    name = "";
                }
            }

            String gender = "";
            if (person.hasGender()) {
                if (person.getGender() == com.google.android.gms.plus.model.people.Person.Gender.MALE) {
                    gender = "M";
                } else if (person.getGender() == com.google.android.gms.plus.model.people.Person.Gender.FEMALE) {
                    gender = "F";
                }
            }

            String birthday = "";

            if (person.hasBirthday()) {
                // We have the string as YYYY-MM-DD
                try {
                    Date date = Constants.GP_DOB_DATE_FORMAT.parse(person.getBirthday());
                    // WR format = $D_yyyyMMdd
                    birthday = "$D_" + Constants.DOB_DATE_FORMAT.format(date);
                } catch (Throwable t) {
                    // Differs from the specs
                    birthday = "";
                }
            }

            String work = "";
            if (person.hasOrganizations()) {
                List<com.google.android.gms.plus.model.people.Person.Organizations> organizations = person.getOrganizations();
                for (com.google.android.gms.plus.model.people.Person.Organizations o : organizations) {
                    if (o.getType() == com.google.android.gms.plus.model.people.Person.Organizations.Type.WORK) {
                        work = "Y";
                        break;
                    }
                }
            }

            String id = "";
            if (person.hasId()) {
                id = person.getId();
            }

            String married = "";
            if (person.hasRelationshipStatus()) {
                if (person.getRelationshipStatus() == com.google.android.gms.plus.model.people.Person.RelationshipStatus.MARRIED) {
                    married = "Y";
                } else {
                    married = "N";
                }
            }

            // Construct json object from the data
            keepProperty(Constants.GP_ID, id);
            keepProperty(Constants.GP_NAME, name);
            keepProperty(Constants.GP_GENDER, gender);
            keepProperty(Constants.GP_EMPLOYED, work);
            keepProperty(Constants.GP_DOB, birthday);
            keepProperty(Constants.GP_RELATIONSHIP_STATUS, married);
            pushBasicProfile(null);
        } catch (Throwable t) {
            // We won't get here
            Logger.logFine("FATAL: Creating G+ profile update event failed!");
        }
    }

    public String getProperty(String name) {
        return LocalDataStore.getProfileProperty(context, name);
    }
}
