/*
 * Decompiled with CFR 0.152.
 */
package com.bugsnag.android;

import androidx.annotation.NonNull;
import com.bugsnag.android.Async;
import com.bugsnag.android.DeliveryParams;
import com.bugsnag.android.DeliveryStatus;
import com.bugsnag.android.EventFilenameInfo;
import com.bugsnag.android.EventPayload;
import com.bugsnag.android.FileStore;
import com.bugsnag.android.ImmutableConfig;
import com.bugsnag.android.Logger;
import com.bugsnag.android.Notifier;
import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Locale;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.Semaphore;

class EventStore
extends FileStore {
    private static final long LAUNCH_CRASH_TIMEOUT_MS = 2000L;
    private static final int LAUNCH_CRASH_POLL_MS = 50;
    volatile boolean flushOnLaunchCompleted = false;
    private final Semaphore semaphore = new Semaphore(1);
    private final ImmutableConfig config;
    private final Logger logger;
    private final FileStore.Delegate delegate;
    private final Notifier notifier;
    static final Comparator<File> EVENT_COMPARATOR = new Comparator<File>(){

        @Override
        public int compare(File lhs, File rhs) {
            if (lhs == null && rhs == null) {
                return 0;
            }
            if (lhs == null) {
                return 1;
            }
            if (rhs == null) {
                return -1;
            }
            return lhs.compareTo(rhs);
        }
    };

    EventStore(@NonNull ImmutableConfig config, @NonNull Logger logger, Notifier notifier, FileStore.Delegate delegate) {
        super(new File(config.getPersistenceDirectory(), "bugsnag-errors"), config.getMaxPersistedEvents(), EVENT_COMPARATOR, logger, delegate);
        this.config = config;
        this.logger = logger;
        this.delegate = delegate;
        this.notifier = notifier;
    }

    void flushOnLaunch() {
        if (this.config.getLaunchCrashThresholdMs() != 0L) {
            List<File> storedFiles = this.findStoredFiles();
            final List<File> crashReports = this.findLaunchCrashReports(storedFiles);
            storedFiles.removeAll(crashReports);
            this.cancelQueuedFiles(storedFiles);
            if (!crashReports.isEmpty()) {
                this.flushOnLaunchCompleted = false;
                this.logger.i("Attempting to send launch crash reports");
                try {
                    Async.run(new Runnable(){

                        @Override
                        public void run() {
                            EventStore.this.flushReports(crashReports);
                            EventStore.this.flushOnLaunchCompleted = true;
                        }
                    });
                }
                catch (RejectedExecutionException ex) {
                    this.logger.w("Failed to flush launch crash reports", ex);
                    this.flushOnLaunchCompleted = true;
                }
                long waitMs = 0L;
                while (!this.flushOnLaunchCompleted && waitMs < 2000L) {
                    try {
                        Thread.sleep(50L);
                        waitMs += 50L;
                    }
                    catch (InterruptedException exception) {
                        this.logger.w("Interrupted while waiting for launch crash report request");
                    }
                }
                this.logger.i("Continuing with Bugsnag initialisation");
            }
        }
        this.flushAsync();
    }

    void flushAsync() {
        try {
            Async.run(new Runnable(){

                @Override
                public void run() {
                    EventStore.this.flushReports(EventStore.this.findStoredFiles());
                }
            });
        }
        catch (RejectedExecutionException exception) {
            this.logger.w("Failed to flush all on-disk errors, retaining unsent errors for later.");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void flushReports(Collection<File> storedReports) {
        if (!storedReports.isEmpty() && this.semaphore.tryAcquire(1)) {
            try {
                this.logger.i(String.format(Locale.US, "Sending %d saved error(s) to Bugsnag", storedReports.size()));
                for (File eventFile : storedReports) {
                    this.flushEventFile(eventFile);
                }
            }
            finally {
                this.semaphore.release(1);
            }
        }
    }

    private void flushEventFile(File eventFile) {
        try {
            EventFilenameInfo eventInfo = EventFilenameInfo.Companion.fromFile(eventFile, this.config);
            String apiKey = eventInfo.getApiKey();
            EventPayload payload = new EventPayload(apiKey, null, eventFile, this.notifier, this.config);
            DeliveryParams deliveryParams = this.config.getErrorApiDeliveryParams(payload);
            DeliveryStatus deliveryStatus = this.config.getDelivery().deliver(payload, deliveryParams);
            switch (deliveryStatus) {
                case DELIVERED: {
                    this.deleteStoredFiles(Collections.singleton(eventFile));
                    this.logger.i("Deleting sent error file " + eventFile.getName());
                    break;
                }
                case UNDELIVERED: {
                    this.cancelQueuedFiles(Collections.singleton(eventFile));
                    this.logger.w("Could not send previously saved error(s) to Bugsnag, will try again later");
                    break;
                }
                case FAILURE: {
                    RuntimeException exc = new RuntimeException("Failed to deliver event payload");
                    this.handleEventFlushFailure(exc, eventFile);
                    break;
                }
            }
        }
        catch (Exception exception) {
            this.handleEventFlushFailure(exception, eventFile);
        }
    }

    private void handleEventFlushFailure(Exception exc, File eventFile) {
        if (this.delegate != null) {
            this.delegate.onErrorIOFailure(exc, eventFile, "Crash Report Deserialization");
        }
        this.deleteStoredFiles(Collections.singleton(eventFile));
    }

    private List<File> findLaunchCrashReports(Collection<File> storedFiles) {
        ArrayList<File> launchCrashes = new ArrayList<File>();
        for (File file : storedFiles) {
            EventFilenameInfo filenameInfo = EventFilenameInfo.Companion.fromFile(file, this.config);
            if (!filenameInfo.isLaunchCrashReport()) continue;
            launchCrashes.add(file);
        }
        return launchCrashes;
    }

    @Override
    @NonNull
    String getFilename(Object object) {
        EventFilenameInfo eventInfo = EventFilenameInfo.Companion.fromEvent(object, null, this.config);
        String encodedInfo = eventInfo.encode();
        return String.format(Locale.US, "%s.json", encodedInfo);
    }

    String getNdkFilename(Object object, String apiKey) {
        EventFilenameInfo eventInfo = EventFilenameInfo.Companion.fromEvent(object, apiKey, this.config);
        String encodedInfo = eventInfo.encode();
        return String.format(Locale.US, "%s.json", encodedInfo);
    }
}

