/*
 * Decompiled with CFR 0.152.
 */
package com.bugsnag.android;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import com.bugsnag.android.Async;
import com.bugsnag.android.BaseObservable;
import com.bugsnag.android.CallbackState;
import com.bugsnag.android.Client;
import com.bugsnag.android.DateUtils;
import com.bugsnag.android.Delivery;
import com.bugsnag.android.DeliveryParams;
import com.bugsnag.android.DeliveryStatus;
import com.bugsnag.android.ForegroundDetector;
import com.bugsnag.android.ImmutableConfig;
import com.bugsnag.android.Logger;
import com.bugsnag.android.Session;
import com.bugsnag.android.SessionStore;
import com.bugsnag.android.StateEvent;
import com.bugsnag.android.User;
import java.io.File;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.Semaphore;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;

class SessionTracker
extends BaseObservable {
    private static final int DEFAULT_TIMEOUT_MS = 30000;
    private final Collection<String> foregroundActivities = new ConcurrentLinkedQueue<String>();
    private final long timeoutMs;
    private final ImmutableConfig configuration;
    private final CallbackState callbackState;
    private final Client client;
    final SessionStore sessionStore;
    private final AtomicLong lastExitedForegroundMs = new AtomicLong(0L);
    private final AtomicLong lastEnteredForegroundMs = new AtomicLong(0L);
    private final AtomicReference<Session> currentSession = new AtomicReference();
    private final Semaphore flushingRequest = new Semaphore(1);
    private final ForegroundDetector foregroundDetector;
    final Logger logger;

    SessionTracker(ImmutableConfig configuration, CallbackState callbackState, Client client, SessionStore sessionStore, Logger logger) {
        this(configuration, callbackState, client, 30000L, sessionStore, logger);
    }

    SessionTracker(ImmutableConfig configuration, CallbackState callbackState, Client client, long timeoutMs, SessionStore sessionStore, Logger logger) {
        this.configuration = configuration;
        this.callbackState = callbackState;
        this.client = client;
        this.timeoutMs = timeoutMs;
        this.sessionStore = sessionStore;
        this.foregroundDetector = new ForegroundDetector(client.getAppContext());
        this.logger = logger;
        this.notifyNdkInForeground();
    }

    @Nullable
    @VisibleForTesting
    Session startNewSession(@NonNull Date date, @Nullable User user, boolean autoCaptured) {
        String id = UUID.randomUUID().toString();
        Session session = new Session(id, date, user, autoCaptured, this.client.getNotifier(), this.logger);
        this.currentSession.set(session);
        this.trackSessionIfNeeded(session);
        return session;
    }

    Session startSession(boolean autoCaptured) {
        return this.startNewSession(new Date(), this.client.getUser(), autoCaptured);
    }

    void pauseSession() {
        Session session = this.currentSession.get();
        if (session != null) {
            session.isPaused.set(true);
            this.notifyObservers(StateEvent.PauseSession.INSTANCE);
        }
    }

    boolean resumeSession() {
        boolean resumed;
        Session session = this.currentSession.get();
        if (session == null) {
            session = this.startSession(false);
            resumed = false;
        } else {
            resumed = session.isPaused.compareAndSet(true, false);
        }
        if (session != null) {
            this.notifySessionStartObserver(session);
        }
        return resumed;
    }

    private void notifySessionStartObserver(Session session) {
        String startedAt = DateUtils.toIso8601(session.getStartedAt());
        this.notifyObservers(new StateEvent.StartSession(session.getId(), startedAt, session.getHandledCount(), session.getUnhandledCount()));
    }

    @Nullable
    Session registerExistingSession(@Nullable Date date, @Nullable String sessionId, @Nullable User user, int unhandledCount, int handledCount) {
        Session session = null;
        if (date != null && sessionId != null) {
            session = new Session(sessionId, date, user, unhandledCount, handledCount, this.client.getNotifier(), this.logger);
            this.notifySessionStartObserver(session);
        } else {
            this.notifyObservers(StateEvent.PauseSession.INSTANCE);
        }
        this.currentSession.set(session);
        return session;
    }

    private void trackSessionIfNeeded(final Session session) {
        this.logger.d("SessionTracker#trackSessionIfNeeded() - session captured by Client");
        boolean notifyForRelease = this.configuration.shouldNotifyForReleaseStage();
        session.setApp(this.client.getAppDataCollector().generateApp());
        session.setDevice(this.client.getDeviceDataCollector().generateDevice());
        boolean deliverSession = this.callbackState.runOnSessionTasks(session, this.logger);
        if (deliverSession && notifyForRelease && (this.configuration.getAutoTrackSessions() || !session.isAutoCaptured()) && session.isTracked().compareAndSet(false, true)) {
            this.notifySessionStartObserver(session);
            try {
                Async.run(new Runnable(){

                    @Override
                    public void run() {
                        SessionTracker.this.flushStoredSessions();
                        try {
                            SessionTracker.this.logger.d("SessionTracker#trackSessionIfNeeded() - attempting initial delivery");
                            DeliveryStatus deliveryStatus = SessionTracker.this.deliverSessionPayload(session);
                            switch (deliveryStatus) {
                                case UNDELIVERED: {
                                    SessionTracker.this.logger.w("Storing session payload for future delivery");
                                    SessionTracker.this.sessionStore.write(session);
                                    break;
                                }
                                case FAILURE: {
                                    SessionTracker.this.logger.w("Dropping invalid session tracking payload");
                                    break;
                                }
                            }
                        }
                        catch (Exception exception) {
                            SessionTracker.this.logger.w("Session tracking payload failed", exception);
                        }
                    }
                });
            }
            catch (RejectedExecutionException exception) {
                this.sessionStore.write(session);
            }
        }
    }

    @Nullable
    Session getCurrentSession() {
        Session session = this.currentSession.get();
        if (session != null && !session.isPaused.get()) {
            return session;
        }
        return null;
    }

    Session incrementUnhandledAndCopy() {
        Session session = this.getCurrentSession();
        if (session != null) {
            return session.incrementUnhandledAndCopy();
        }
        return null;
    }

    Session incrementHandledAndCopy() {
        Session session = this.getCurrentSession();
        if (session != null) {
            return session.incrementHandledAndCopy();
        }
        return null;
    }

    void flushAsync() {
        try {
            Async.run(new Runnable(){

                @Override
                public void run() {
                    SessionTracker.this.flushStoredSessions();
                }
            });
        }
        catch (RejectedExecutionException ex) {
            this.logger.w("Failed to flush session reports", ex);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void flushStoredSessions() {
        if (this.flushingRequest.tryAcquire(1)) {
            try {
                List<File> storedFiles = this.sessionStore.findStoredFiles();
                for (File storedFile : storedFiles) {
                    this.flushStoredSession(storedFile);
                }
            }
            finally {
                this.flushingRequest.release(1);
            }
        }
    }

    void flushStoredSession(File storedFile) {
        this.logger.d("SessionTracker#flushStoredSession() - attempting delivery");
        Session payload = new Session(storedFile, this.client.getNotifier(), this.logger);
        if (!payload.isV2Payload()) {
            payload.setApp(this.client.getAppDataCollector().generateApp());
            payload.setDevice(this.client.getDeviceDataCollector().generateDevice());
        }
        DeliveryStatus deliveryStatus = this.deliverSessionPayload(payload);
        switch (deliveryStatus) {
            case DELIVERED: {
                this.sessionStore.deleteStoredFiles(Collections.singletonList(storedFile));
                break;
            }
            case UNDELIVERED: {
                this.sessionStore.cancelQueuedFiles(Collections.singletonList(storedFile));
                this.logger.w("Leaving session payload for future delivery");
                break;
            }
            case FAILURE: {
                this.logger.w("Deleting invalid session tracking payload");
                this.sessionStore.deleteStoredFiles(Collections.singletonList(storedFile));
                break;
            }
        }
    }

    DeliveryStatus deliverSessionPayload(Session payload) {
        DeliveryParams params = this.configuration.getSessionApiDeliveryParams();
        Delivery delivery = this.configuration.getDelivery();
        return delivery.deliver(payload, params);
    }

    void onActivityStarted(String activityName) {
        this.updateForegroundTracker(activityName, true, System.currentTimeMillis());
    }

    void onActivityStopped(String activityName) {
        this.updateForegroundTracker(activityName, false, System.currentTimeMillis());
    }

    void updateForegroundTracker(String activityName, boolean activityStarting, long nowMs) {
        if (activityStarting) {
            long noActivityRunningForMs = nowMs - this.lastExitedForegroundMs.get();
            if (this.foregroundActivities.isEmpty()) {
                this.lastEnteredForegroundMs.set(nowMs);
                if (noActivityRunningForMs >= this.timeoutMs && this.configuration.getAutoTrackSessions()) {
                    this.startNewSession(new Date(nowMs), this.client.getUser(), true);
                }
            }
            this.foregroundActivities.add(activityName);
        } else {
            this.foregroundActivities.remove(activityName);
            if (this.foregroundActivities.isEmpty()) {
                this.lastExitedForegroundMs.set(nowMs);
            }
        }
        this.notifyNdkInForeground();
    }

    private void notifyNdkInForeground() {
        Boolean inForeground = this.isInForeground();
        boolean foreground = inForeground != null ? inForeground : false;
        this.notifyObservers(new StateEvent.UpdateInForeground(foreground, this.getContextActivity()));
    }

    @Nullable
    Boolean isInForeground() {
        return this.foregroundDetector.isInForeground();
    }

    @Nullable
    Long getDurationInForegroundMs(long nowMs) {
        long durationMs = 0L;
        long sessionStartTimeMs = this.lastEnteredForegroundMs.get();
        Boolean inForeground = this.isInForeground();
        if (inForeground == null) {
            return null;
        }
        if (inForeground.booleanValue() && sessionStartTimeMs != 0L) {
            durationMs = nowMs - sessionStartTimeMs;
        }
        return durationMs > 0L ? durationMs : 0L;
    }

    @Nullable
    String getContextActivity() {
        if (this.foregroundActivities.isEmpty()) {
            return null;
        }
        int size = this.foregroundActivities.size();
        String[] activities = this.foregroundActivities.toArray(new String[size]);
        return activities[size - 1];
    }
}

