/*
 * Decompiled with CFR 0.152.
 */
package com.bugsnag.android;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import com.bugsnag.android.BackgroundTaskService;
import com.bugsnag.android.CallbackState;
import com.bugsnag.android.Delivery;
import com.bugsnag.android.DeliveryParams;
import com.bugsnag.android.DeliveryStatus;
import com.bugsnag.android.Event;
import com.bugsnag.android.EventFilenameInfo;
import com.bugsnag.android.EventPayload;
import com.bugsnag.android.FileStore;
import com.bugsnag.android.Logger;
import com.bugsnag.android.MarshalledEventSource;
import com.bugsnag.android.Notifier;
import com.bugsnag.android.TaskType;
import com.bugsnag.android.internal.ImmutableConfig;
import java.io.File;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

class EventStore
extends FileStore {
    private static final long LAUNCH_CRASH_TIMEOUT_MS = 2000L;
    private final ImmutableConfig config;
    private final FileStore.Delegate delegate;
    private final Notifier notifier;
    private final BackgroundTaskService bgTaskSevice;
    private final CallbackState callbackState;
    final Logger logger;
    static final Comparator<File> EVENT_COMPARATOR = new Comparator<File>(){

        @Override
        public int compare(File lhs, File rhs) {
            if (lhs == null && rhs == null) {
                return 0;
            }
            if (lhs == null) {
                return 1;
            }
            if (rhs == null) {
                return -1;
            }
            return lhs.compareTo(rhs);
        }
    };
    private static long oneMegabyte = 0x100000L;

    EventStore(@NonNull ImmutableConfig config, @NonNull Logger logger, Notifier notifier, BackgroundTaskService bgTaskSevice, FileStore.Delegate delegate, CallbackState callbackState) {
        super(new File((File)config.getPersistenceDirectory().getValue(), "bugsnag-errors"), config.getMaxPersistedEvents(), EVENT_COMPARATOR, logger, delegate);
        this.config = config;
        this.logger = logger;
        this.delegate = delegate;
        this.notifier = notifier;
        this.bgTaskSevice = bgTaskSevice;
        this.callbackState = callbackState;
    }

    void flushOnLaunch() {
        if (!this.config.getSendLaunchCrashesSynchronously()) {
            return;
        }
        Future<?> future = null;
        try {
            future = this.bgTaskSevice.submitTask(TaskType.ERROR_REQUEST, new Runnable(){

                @Override
                public void run() {
                    EventStore.this.flushLaunchCrashReport();
                }
            });
        }
        catch (RejectedExecutionException exc) {
            this.logger.d("Failed to flush launch crash reports, continuing.", exc);
        }
        try {
            if (future != null) {
                future.get(2000L, TimeUnit.MILLISECONDS);
            }
        }
        catch (InterruptedException | ExecutionException | TimeoutException exc) {
            this.logger.d("Failed to send launch crash reports within 2s timeout, continuing.", exc);
        }
    }

    void flushLaunchCrashReport() {
        List<File> storedFiles = this.findStoredFiles();
        File launchCrashReport = this.findLaunchCrashReport(storedFiles);
        if (launchCrashReport != null) {
            storedFiles.remove(launchCrashReport);
        }
        this.cancelQueuedFiles(storedFiles);
        if (launchCrashReport != null) {
            this.logger.i("Attempting to send the most recent launch crash report");
            this.flushReports(Collections.singletonList(launchCrashReport));
            this.logger.i("Continuing with Bugsnag initialisation");
        } else {
            this.logger.d("No startupcrash events to flush to Bugsnag.");
        }
    }

    @Nullable
    File findLaunchCrashReport(Collection<File> storedFiles) {
        ArrayList<File> launchCrashes = new ArrayList<File>();
        for (File file : storedFiles) {
            EventFilenameInfo filenameInfo = EventFilenameInfo.fromFile(file, this.config);
            if (!filenameInfo.isLaunchCrashReport()) continue;
            launchCrashes.add(file);
        }
        Collections.sort(launchCrashes, EVENT_COMPARATOR);
        return launchCrashes.isEmpty() ? null : (File)launchCrashes.get(launchCrashes.size() - 1);
    }

    void flushAsync() {
        try {
            this.bgTaskSevice.submitTask(TaskType.ERROR_REQUEST, new Runnable(){

                @Override
                public void run() {
                    List<File> storedFiles = EventStore.this.findStoredFiles();
                    if (storedFiles.isEmpty()) {
                        EventStore.this.logger.d("No regular events to flush to Bugsnag.");
                    }
                    EventStore.this.flushReports(storedFiles);
                }
            });
        }
        catch (RejectedExecutionException exception) {
            this.logger.w("Failed to flush all on-disk errors, retaining unsent errors for later.");
        }
    }

    void flushReports(Collection<File> storedReports) {
        if (!storedReports.isEmpty()) {
            int size = storedReports.size();
            this.logger.i("Sending " + size + " saved error(s) to Bugsnag");
            for (File eventFile : storedReports) {
                this.flushEventFile(eventFile);
            }
        }
    }

    private void flushEventFile(File eventFile) {
        try {
            EventFilenameInfo eventInfo = EventFilenameInfo.fromFile(eventFile, this.config);
            String apiKey = eventInfo.getApiKey();
            EventPayload payload = this.createEventPayload(eventFile, apiKey);
            if (payload == null) {
                this.deleteStoredFiles(Collections.singleton(eventFile));
            } else {
                this.deliverEventPayload(eventFile, payload);
            }
        }
        catch (Exception exception) {
            this.handleEventFlushFailure(exception, eventFile);
        }
    }

    private void deliverEventPayload(File eventFile, EventPayload payload) {
        DeliveryParams deliveryParams = this.config.getErrorApiDeliveryParams(payload);
        Delivery delivery = this.config.getDelivery();
        DeliveryStatus deliveryStatus = delivery.deliver(payload, deliveryParams);
        switch (deliveryStatus) {
            case DELIVERED: {
                this.deleteStoredFiles(Collections.singleton(eventFile));
                this.logger.i("Deleting sent error file " + eventFile.getName());
                break;
            }
            case UNDELIVERED: {
                if (this.isTooBig(eventFile)) {
                    this.logger.w("Discarding over-sized event (" + eventFile.length() + ") after failed delivery");
                    this.deleteStoredFiles(Collections.singleton(eventFile));
                    break;
                }
                if (this.isTooOld(eventFile)) {
                    this.logger.w("Discarding historical event (from " + this.getCreationDate(eventFile) + ") after failed delivery");
                    this.deleteStoredFiles(Collections.singleton(eventFile));
                    break;
                }
                this.cancelQueuedFiles(Collections.singleton(eventFile));
                this.logger.w("Could not send previously saved error(s) to Bugsnag, will try again later");
                break;
            }
            case FAILURE: {
                RuntimeException exc = new RuntimeException("Failed to deliver event payload");
                this.handleEventFlushFailure(exc, eventFile);
                break;
            }
        }
    }

    @Nullable
    private EventPayload createEventPayload(File eventFile, String apiKey) {
        MarshalledEventSource eventSource = new MarshalledEventSource(eventFile, apiKey, this.logger);
        try {
            if (!this.callbackState.runOnSendTasks(eventSource, this.logger)) {
                return null;
            }
        }
        catch (Exception ioe) {
            eventSource.clear();
        }
        Event processedEvent = eventSource.getEvent();
        if (processedEvent != null) {
            apiKey = processedEvent.getApiKey();
            return new EventPayload(apiKey, processedEvent, null, this.notifier, this.config);
        }
        return new EventPayload(apiKey, null, eventFile, this.notifier, this.config);
    }

    private void handleEventFlushFailure(Exception exc, File eventFile) {
        if (this.delegate != null) {
            this.delegate.onErrorIOFailure(exc, eventFile, "Crash Report Deserialization");
        }
        this.deleteStoredFiles(Collections.singleton(eventFile));
    }

    @Override
    @NonNull
    String getFilename(Object object) {
        EventFilenameInfo eventInfo = EventFilenameInfo.fromEvent(object, null, this.config);
        return eventInfo.encode();
    }

    String getNdkFilename(Object object, String apiKey) {
        EventFilenameInfo eventInfo = EventFilenameInfo.fromEvent(object, apiKey, this.config);
        return eventInfo.encode();
    }

    public boolean isTooBig(File file) {
        return file.length() > oneMegabyte;
    }

    public boolean isTooOld(File file) {
        Calendar cal = Calendar.getInstance();
        cal.add(5, -60);
        return EventFilenameInfo.findTimestampInFilename(file) < cal.getTimeInMillis();
    }

    public Date getCreationDate(File file) {
        return new Date(EventFilenameInfo.findTimestampInFilename(file));
    }
}

