/**
 * Copyright (C) 2013 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.aniways.volley.toolbox;

import java.util.ArrayList;
import java.util.List;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.drawable.BitmapDrawable;
import android.graphics.drawable.Drawable;
import android.text.TextUtils;
import android.util.AttributeSet;
import android.view.ViewGroup.LayoutParams;
import android.widget.ImageView;

import com.aniways.IconData;
import com.aniways.Log;
import com.aniways.analytics.NonThrowingRunnable;
import com.aniways.data.AniwaysNetworkStateChecker;
import com.aniways.data.AniwaysPrivateConfig;
import com.aniways.volley.VolleyError;
import com.aniways.volley.toolbox.ImageLoader.ImageContainer;
import com.aniways.volley.toolbox.ImageLoader.ImageListener;

import pl.droidsonroids.gif.GifDrawable;
import pl.droidsonroids.gif.GifDrawableBuilder;

/**
 * Handles fetching an image from a URL as well as the life-cycle of the
 * associated request.
 */
public class NetworkImageView extends ImageView {
    protected static final String TAG = "AniwaysNetworkImageView";

	/** The URL of the network image to load */
    private String mUrl;

    /**
     * Resource ID of the image to be used as a placeholder until the network image is loaded.
     */
    private int mDefaultImageId;

    /**
     * Resource ID of the image to be used if the network response fails.
     */
    private int mErrorImageId;

    /** Local copy of the ImageLoader. */
    private ImageLoader mImageLoader;

    /** Current ImageContainer. (either in-flight or finished) */
    private ImageContainer mImageContainer;

	private List<IResponseListener> mRequestListeners = new ArrayList<IResponseListener>();

	private int mMaxWidth;

	private int mMaxHeight;

	private boolean mGetDiskCacheOnMainThread;

	private IconData mIconData;

    public NetworkImageView(Context context) {
        this(context, null);
    }

    public NetworkImageView(Context context, AttributeSet attrs) {
        this(context, attrs, 0);
    }

    public NetworkImageView(Context context, AttributeSet attrs, int defStyle) {
        super(context, attrs, defStyle);
    }

    /**
     * Sets URL of the image that should be loaded into this view. Note that calling this will
     * immediately either set the cached image (if available) or the default image specified by
     * {@link com.aniways.volley.toolbox.NetworkImageView#setDefaultImageResId(int)} on the view.
     *
     * NOTE: If applicable, {@link com.aniways.volley.toolbox.NetworkImageView#setDefaultImageResId(int)} and
     * {@link com.aniways.volley.toolbox.NetworkImageView#setErrorImageResId(int)} should be called prior to calling
     * this function.
     *
     * @param url The URL that should be loaded into this ImageView.
     * @param imageLoader ImageLoader that will be used to make the request.
     */
    public void setImageUrl(String url, ImageLoader imageLoader) {
      setImageUrl(url, null, imageLoader, 0, 0, false);    
    }

	public void setImageUrl(String url, IconData iconData, ImageLoader imageLoader, Integer maxWidth, Integer maxHeight, boolean getDiskCacheOnMainThread) {
		mUrl = url;
		mImageLoader = imageLoader;
		mMaxWidth = maxWidth;
		mMaxHeight = maxHeight;
		mGetDiskCacheOnMainThread = getDiskCacheOnMainThread;
		mIconData = iconData;
		// The URL has potentially changed. See if we need to load it.
		loadImageIfNecessary(false);
	}

    /**
     * Sets the default image resource ID to be used for this view until the attempt to load it
     * completes.
     */
    public void setDefaultImageResId(int defaultImage) {
        mDefaultImageId = defaultImage;
    }

    /**
     * Sets the error image resource ID to be used for this view in the event that the image
     * requested fails to load.
     */
    public void setErrorImageResId(int errorImage) {
        mErrorImageId = errorImage;
    }

    public void registerResponseListener(IResponseListener responseListener){
    	mRequestListeners.add(responseListener);
    }

    public void unregisterPreviousListeners() {
        this.mRequestListeners.clear();
    }

    /**
     * Loads the image for the view if it isn't already loaded.
     * @param isInLayoutPass True if this was invoked from a layout pass, false otherwise.
     */
    void loadImageIfNecessary(final boolean isInLayoutPass) {
    	
        int width = mMaxWidth == 0 ? getWidth() : mMaxWidth;
        int height = mMaxHeight == 0 ? getHeight() : mMaxHeight;

        boolean wrapWidth = false, wrapHeight = false;
        if (getLayoutParams() != null) {
            wrapWidth = getLayoutParams().width == LayoutParams.WRAP_CONTENT;
            wrapHeight = getLayoutParams().height == LayoutParams.WRAP_CONTENT;
        }

        // if the view's bounds aren't known yet, and this is not a wrap-content/wrap-content
        // view, hold off on loading the image.
        boolean isFullyWrapContent = wrapWidth && wrapHeight;
        if (width == 0 && height == 0 && !isFullyWrapContent) {
            return;
        }

        // if the URL to be loaded in this view is empty, cancel any old requests and clear the
        // currently loaded image.
        if (TextUtils.isEmpty(mUrl)) {
            if (mImageContainer != null) {
                mImageContainer.cancelRequest();
                mImageContainer = null;
            }
            setDefaultImageOrNull();
            return;
        }

        // if there was an old request in this view, check if it needs to be canceled.
        if (mImageContainer != null && mImageContainer.getRequestUrl() != null) {
            if (mImageContainer.getRequestUrl().equals(mUrl)) {
                // if the request is from the same URL, return.
                return;
            } else {
                // if there is a pre-existing request, cancel it if it's fetching a different URL.
                mImageContainer.cancelRequest();
                setDefaultImageOrNull();
            }
        }

        // Calculate the max image width / height to use while ignoring WRAP_CONTENT dimens.
        final int maxWidth = wrapWidth ? 0 : width;
        final int maxHeight = wrapHeight ? 0 : height;

        // The pre-existing content of this view didn't match the current URL. Load the new image
        // from the network.
        ImageContainer newContainer = mImageLoader.get(mUrl,
                new ImageListener() {
                    @Override
                    public void onErrorResponse(final VolleyError error) {
                    	
                    	AniwaysNetworkStateChecker.checkInternet(new AniwaysNetworkStateChecker.AniwaysNetworkSateCheckCallback() {
							
							@Override
							public void run(boolean connectionAvailable) {
								if (connectionAvailable){
		    						Log.e(true, TAG, "Error loading image from url: " + mUrl + ". Error: " + error.getMessage(), error);
		    					}
		    					else{
		    						Log.w(false, TAG, "Error loading image, but its OK since there us no internet connection. Url: " + mUrl + ". Error: " + error.getMessage(), error);
		    					}
							}
						});
                    	
                        if (mErrorImageId != 0) {
                            setImageResource(mErrorImageId);
                        }
                        for(IResponseListener listener : mRequestListeners){
                        	listener.onError();
                        }
                    }

                    @Override
                    public void onResponse(final ImageContainer response, final boolean isImmediate) {
                        // If this was an immediate response that was delivered inside of a layout
                        // pass do not set the image immediately as it will trigger a requestLayout
                        // inside of a layout. Instead, defer setting the image by posting back to
                        // the main thread.
                        if (isImmediate && isInLayoutPass) {
                            post(new NonThrowingRunnable("Aniways volley network image view", "onResponse", null) {
                                @Override
                                public void innerRun() {
                                    onResponse(response, isImmediate);
                                }
                            });
                            return;
                        }
                        onResponseInternal(response, isImmediate);
                    }
                    public void onResponseInternal(final ImageContainer response, final boolean cameFromImmediate) {
                        Drawable dr = null;
                        if(mIconData != null && mIconData.isAnimated() && AniwaysPrivateConfig.getInstance().animateInImageView){
                        	Object data = response.getRawData();
                        	if(data == null){
                        		if(cameFromImmediate){
                        			Log.v(TAG, "Received immediate null response for animated gif: " + mUrl);
                        		}
                        		else{
                        			Log.w(true, TAG, "Received non-immediate null response for animated gif: " + mUrl);
                        			for(IResponseListener listener : mRequestListeners){
                                    	listener.onError();
                                    }
                        		}
                        	}
                        	else{
                        		if(data instanceof byte[]){
                        			dr = mIconData.setDrawableBounds(IconData.generateAnimatedDrawable((byte[])data), maxWidth, maxHeight);
                        		}
                        		else{
                        			Log.e(true, TAG, "Received non byte array for animated gif: " + mUrl);
                        			for(IResponseListener listener : mRequestListeners){
                                    	listener.onError();
                                    }
                        		}
                        	}
                        }
                        else if (response.getRawData() != null){ //Yay supporting animated gifs :)
                            //todo: move the methos from IconData to AniwaysImageUtil
                            dr = IconData.generateAnimatedDrawable(response.getRawData());
                        }
                        else if (response.getBitmap() != null) {
                            dr = new BitmapDrawable(getContext().getResources(), response.getBitmap());
                        }
                        else if(response.getBitmap() == null){
                            if(cameFromImmediate){
                                Log.v(TAG, "Received immediate null response for bitmap: " + mUrl);
                                if(mDefaultImageId != 0) {
                                    setImageResource(mDefaultImageId);
                                }
                                return;
                            }
                            Log.e(true, TAG, "Bitmap is null. Url: " + mUrl);
                        	for(IResponseListener listener : mRequestListeners){
                            	listener.onError();
                            }
                        }
                        if(dr != null){
                            if(dr instanceof GifDrawable){
                                ((GifDrawable)dr).stop();
                            }
                        	setImageDrawable(dr);
                            for(IResponseListener listener : mRequestListeners){
                            	listener.onSuccess();
                            }
                        } 
                        else if(mDefaultImageId != 0) {
                            Log.e(true,TAG,"Setting default resource");
                            setImageResource(mDefaultImageId);
                        }
                    }
                }, maxWidth, maxHeight, mGetDiskCacheOnMainThread);

        // update the ImageContainer to be the new bitmap container.
        mImageContainer = newContainer;
    }

    private void setDefaultImageOrNull() {
        if(mDefaultImageId != 0) {
            setImageResource(mDefaultImageId);
        }
        else {
            setImageBitmap(null);
        }
    }

    @Override
    protected void onLayout(boolean changed, int left, int top, int right, int bottom) {
        super.onLayout(changed, left, top, right, bottom);
        loadImageIfNecessary(true);
    }

    @Override
    protected void onDetachedFromWindow() {
        recycle(false, true, true);
        super.onDetachedFromWindow();
    }

    @Override
    protected void drawableStateChanged() {
        super.drawableStateChanged();
        invalidate();
    }

    public void stopAnimation() {
        Drawable dr = this.getDrawable();
        if(dr == null){
            return;
        }
        if(dr instanceof  GifDrawable) {
            ((GifDrawable) dr).stop();
        }
    }

    public void startAnimation() {
        Drawable dr = this.getDrawable();
        if(dr == null){
            return;
        }
        if(dr instanceof  GifDrawable) {
            ((GifDrawable) dr).start();
        }
    }

    public void recycle(boolean recycleBitmap, boolean recycleAnimatedGif, boolean keepUrl) {
        if (mImageContainer != null) {
            Drawable dr = this.getDrawable();
            if(dr != null){
                if(dr instanceof GifDrawable) {
                    if(recycleAnimatedGif) {
                        ((GifDrawable) dr).recycle();
                        Log.v(TAG, "recycled gif");
                    }
                }
                else if(dr instanceof BitmapDrawable){
                    if(recycleBitmap){
                        Bitmap b = ((BitmapDrawable)dr).getBitmap();
                        if (b != null) {
                            b.recycle();
                            Log.v(TAG, "recycled bitmap");
                        }
                    }
                }
            }

            // If the view was bound to an image request, cancel it and clear
            // out the image from the view.
            mImageContainer.cancelRequest();
            setImageBitmap(null);
            // also clear out the container so we can reload the image if necessary.
            mImageContainer = null;
            if(!keepUrl){
                mUrl = null;
            }
        }
    }

    @Override
    protected void onDraw(Canvas canvas) {
        try {
            super.onDraw(canvas);
        }
        catch (Throwable ex){
            //TODO:..
        }
    }
}
