package com.aniways.ui.views;

import android.content.Context;
import android.content.res.TypedArray;
import android.preference.Preference;
import android.util.AttributeSet;
import android.view.View;
import android.view.ViewGroup;
import android.view.ViewParent;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.SeekBar;

import com.aniways.Log;
import com.aniways.R;
import com.aniways.ui.AniwaysUiUtil;

/**
 * Created by moran on 12/10/2014.
 */
public class AniwaysSeekBarPreference extends Preference implements SeekBar.OnSeekBarChangeListener {

    private final String TAG = getClass().getName();

    private static final String ANDROIDNS="http://schemas.android.com/apk/res/android";
    private static final String APPLICATIONNS="http://aniways.com";
    private static final int DEFAULT_VALUE = 60;

    private int mMaxValue      = 80;
    private int mMinValue      = 20;
    private int mInterval      = 1;
    private int mCurrentValue;
    private SeekBar mSeekBar;
    private ImageView mEmoticonSizeIndicatorImage;

    public int getPersistedValue() {
        return getPersistedInt(DEFAULT_VALUE);
    }

    public AniwaysSeekBarPreference(Context context, AttributeSet attrs) {
        super(context, attrs);
        initPreference(context, attrs);
    }

    public AniwaysSeekBarPreference(Context context, AttributeSet attrs, int defStyle) {
        super(context, attrs, defStyle);
        initPreference(context, attrs);
    }

    private void initPreference(Context context, AttributeSet attrs) {
        setValuesFromXml(attrs);
        mSeekBar = new SeekBar(context, attrs);
        mSeekBar.setMax(mMaxValue - mMinValue);
        mSeekBar.setOnSeekBarChangeListener(this);

        setWidgetLayoutResource(R.layout.aniways_seek_bar_preference);
    }

    private void setValuesFromXml(AttributeSet attrs) {
        mMaxValue = attrs.getAttributeIntValue(ANDROIDNS, "max", 80);
        mMinValue = attrs.getAttributeIntValue(APPLICATIONNS, "min", 20);

        try {
            String newInterval = attrs.getAttributeValue(APPLICATIONNS, "interval");
            if(newInterval != null)
                mInterval = Integer.parseInt(newInterval);
        }
        catch(Exception e) {
            Log.e(true,TAG, "Invalid interval value", e);
        }

    }

    @Override
    protected View onCreateView(ViewGroup parent) {
        try {
            View view = super.onCreateView(parent);

            // The basic preference layout puts the widget frame to the right of the title and summary,
            // so we need to change it a bit - the seekbar should be under them.
            LinearLayout layout = (LinearLayout) view;
            layout.setOrientation(LinearLayout.VERTICAL);

            return view;
        } catch (Exception e) {
            Log.e(true,TAG,"Caught exception at onCreateView in AniwaysSeekbarPrefernce", e);
            return null;
        }
    }

    @Override
    public void onBindView(View view) {
        super.onBindView(view);

        try {
            // move our seekbar to the new view we've been given
            ViewParent oldContainer = mSeekBar.getParent();
            ViewGroup newContainer = (ViewGroup) view.findViewById(R.id.seekBarPrefBarContainer);

            if (oldContainer != newContainer) {
                // remove the seekbar from the old view
                if (oldContainer != null) {
                    ((ViewGroup) oldContainer).removeView(mSeekBar);
                }
                // remove the existing seekbar (there may not be one) and add ours
                newContainer.removeAllViews();
                newContainer.addView(mSeekBar, ViewGroup.LayoutParams.MATCH_PARENT,
                        ViewGroup.LayoutParams.WRAP_CONTENT);
            }
        }
        catch(Exception ex) {
            Log.e(true,TAG, "Error binding view: " + ex.toString());
        }

        //if dependency is false from the beginning, disable the seek bar
        if (view != null && !view.isEnabled())
        {
            mSeekBar.setEnabled(false);
        }

        updateView(view);
    }

    /**
     * Update a SeekBarPreference view with our current state
     * @param view
     */
    protected void updateView(View view) {
        try {
            int progress = mCurrentValue - mMinValue;

            mEmoticonSizeIndicatorImage = (ImageView) view.findViewById(R.id.emoticon_seek_bar_indication);
            mSeekBar.setProgress(progress);
            onProgressChanged(mSeekBar, progress,false);
        }
        catch(Exception e) {
            Log.e(true,TAG, "Error updating seek bar preference", e);
        }
    }

    @Override
    public void onProgressChanged(SeekBar seekBar, int progress, boolean fromUser) {
        int newValue = progress + mMinValue;

        if(newValue > mMaxValue)
            newValue = mMaxValue;
        else if(newValue < mMinValue)
            newValue = mMinValue;
        else if(mInterval != 1 && newValue % mInterval != 0)
            newValue = Math.round(((float)newValue)/mInterval)*mInterval;

        // change rejected, revert to the previous value
        if(!callChangeListener(newValue)){
            seekBar.setProgress(mCurrentValue - mMinValue);
            return;
        }

        // change accepted, store it
        mCurrentValue = newValue;
        setImageIndicatorSize(mCurrentValue);
        persistInt(newValue);
    }

    private void setImageIndicatorSize(int newSize){
        if (mEmoticonSizeIndicatorImage == null)
            return;

        int newSizeInDp = AniwaysUiUtil.convertDipsToPixels(newSize);

        mEmoticonSizeIndicatorImage.getLayoutParams().width = newSizeInDp;
        mEmoticonSizeIndicatorImage.getLayoutParams().height = newSizeInDp;

        Log.d("setImageIndicatorSize", "new size: " + newSize);

        mEmoticonSizeIndicatorImage.requestLayout();
    }

    @Override
    public void onStartTrackingTouch(SeekBar seekBar) {
    }

    @Override
    public void onStopTrackingTouch(SeekBar seekBar) {
        //notifyChanged();
    }

    @Override
    protected Object onGetDefaultValue(TypedArray ta, int index){
        int defaultValue = ta.getInt(index, DEFAULT_VALUE);
        return defaultValue;
    }

    @Override
    protected void onSetInitialValue(boolean restoreValue, Object defaultValue) {
        if(restoreValue) {
            mCurrentValue = getPersistedInt(mCurrentValue);
        }
        else {
            int temp = 60;
            try {
                temp = (Integer)defaultValue;
            }
            catch(Exception ex) {
                Log.e(true,TAG, "Invalid default value: " + defaultValue.toString());
            }

            persistInt(temp);
            mCurrentValue = temp;
        }
    }

    /**
     * make sure that the seekbar is disabled if the preference is disabled
     */
    @Override
    public void setEnabled(boolean enabled) {
        super.setEnabled(enabled);
        mSeekBar.setEnabled(enabled);
    }

    @Override
    public void onDependencyChanged(Preference dependency, boolean disableDependent) {
        super.onDependencyChanged(dependency, disableDependent);

        //Disable movement of seek bar when dependency is false
        if (mSeekBar != null)
        {
            mSeekBar.setEnabled(!disableDependent);
        }
    }
}