package com.aniways.ui.views;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.Context;
import android.content.res.Configuration;
import android.graphics.Color;
import android.graphics.Rect;
import android.graphics.drawable.ColorDrawable;
import android.support.annotation.NonNull;
import android.view.Gravity;
import android.view.LayoutInflater;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewTreeObserver;
import android.view.WindowManager;
import android.widget.PopupWindow;

import com.aniways.Log;
import com.aniways.Utils;
import com.aniways.analytics.AnalyticsReporter;
import com.aniways.data.AniwaysStatics;
import com.aniways.quick.action.ContextualPopupCreationContext;

import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;

/**
 * A popup implementation which allows delaying and canceling its dismissal.
 */
public abstract class AniwaysPopupWindow extends PopupWindow implements ViewTreeObserver.OnGlobalLayoutListener
{
    // Holds the logging tag.
    private static final String TAG = "AniwaysPopupWindow";

    protected ContextualPopupCreationContext creationContext;

    private static final List<AniwaysPopupWindow> openPopups = new ArrayList<>();

    // The context under which this instance operates.
    protected final Context context;

    // Holds the layout inflater service.
    protected final LayoutInflater layoutInflater;

    private final Object syncObject;

    // The timer used for delaying the touch-outside-dismissal of this popup instance.
    private Timer touchOutsideDismissalTimer;

    // The timer used for delaying the auto-dismissal of this popup instance.
    private Timer autoDismissalTimer;

    // Holds the timing options for this popup instance.
    private final PopupTimingOptions timingOptions;

    // Holds the popup anchor.
    protected WeakReference<View> anchorRef;

    // Holds a value indicating whether this instance was dismissed.
    private boolean dismissed;

    // Holds a reference to all who wish to intercept this instance's touch events.
    private final List<View.OnTouchListener> touchInterceptors;

    /**
     * Constructor.
     *
     * @param context          Context
     * @param matchParentWidth A value indicating whether this popup instance should default to matching the parent window's width.
     * @param timingOptions    Popup timing options for this popup instance.
     */
    protected AniwaysPopupWindow(Context context, final ContextualPopupCreationContext creationContext, boolean matchParentWidth, final PopupTimingOptions timingOptions)
    {
        super(context);
        this.creationContext = creationContext;
        this.context = context;
        this.timingOptions = timingOptions;
        this.layoutInflater = (LayoutInflater)context.getSystemService(Context.LAYOUT_INFLATER_SERVICE);
        this.syncObject = new Object();
        this.touchInterceptors = new ArrayList<>();

        setBackgroundDrawable(new ColorDrawable(Color.TRANSPARENT));
        setWidth(matchParentWidth ? WindowManager.LayoutParams.MATCH_PARENT : WindowManager.LayoutParams.WRAP_CONTENT);
        setHeight(WindowManager.LayoutParams.WRAP_CONTENT);
        setTouchable(true);

        // This makes sure that the popup will dispatch the touch listener when the user touches outside this instance.
        setFocusable(false);
        setInputMethodMode(PopupWindow.INPUT_METHOD_NOT_NEEDED);
        setOutsideTouchable(true);

        super.setTouchInterceptor(new View.OnTouchListener()
        {
            @Override
            public boolean onTouch(View v, MotionEvent event)
            {
                boolean retProcessed = false;

                for (View.OnTouchListener onTouchListener : AniwaysPopupWindow.this.touchInterceptors)
                {
                    retProcessed |= onTouchListener.onTouch(v, event);
                }

                return retProcessed;
            }
        });

        addTouchInterceptor(new View.OnTouchListener()
        {
            @Override
            public boolean onTouch(View v, MotionEvent event)
            {
                try
                {
                    switch (event.getAction())
                    {
                        case MotionEvent.ACTION_OUTSIDE:
                            // In case of a touch outside the popup, either delay the dismissal or immediately dismiss.
                            if (timingOptions.delayTouchOutsideDismissal)
                            {
                                touchOutsideDismissalTimer = new Timer();
                                touchOutsideDismissalTimer.schedule(new DismissPopupTask(AniwaysPopupWindow.this), timingOptions.touchOutsideDismissalDelay);
                            }
                            else
                            {
                                updatePopupStateBeforeClosing("tap-outside-popup");
                                dismiss();
                            }

                            return true;
                        case MotionEvent.ACTION_DOWN:
                            // In case of an auto-dismissing popup, extend the auto-dismissal timer, if desired.
                            if (timingOptions.autoDismiss && timingOptions.extendAutoDismissOnTouch)
                            {
                                Timer autoDismissalTimer = AniwaysPopupWindow.this.autoDismissalTimer;
                                AniwaysPopupWindow.this.autoDismissalTimer = null;

                                if (autoDismissalTimer != null)
                                {
                                    autoDismissalTimer.cancel();
                                }
                            }

                            return false;
                        case MotionEvent.ACTION_CANCEL:
                        case MotionEvent.ACTION_UP:

                            scheduleAutoDismissalTimer();

                            return false;
                    }
                }
                catch (Throwable ex)
                {
                    Log.e(true, TAG, "Caught Exception in onTouch", ex);
                }

                return false;
            }
        });
    }

    private void updatePopupStateBeforeClosing() {
        updatePopupStateBeforeClosing("popup-closed");
    }

    private void updatePopupStateBeforeClosing(String exitMethodDescription) {
        creationContext.popupStateHolder.exitMethod = exitMethodDescription;
        creationContext.popupStateHolder.nextScreen = "popup-closed";
    }

    private void scheduleAutoDismissalTimer()
    {
        this.autoDismissalTimer = new Timer();
        this.autoDismissalTimer.schedule(new DismissPopupTask(this), this.timingOptions.autoDismissalDelay);
    }

    public void addTouchInterceptor(@NonNull View.OnTouchListener onTouchListener)
    {
        this.touchInterceptors.add(onTouchListener);
    }

    @Override
    public void setTouchInterceptor(View.OnTouchListener onTouchListener)
    {
        addTouchInterceptor(onTouchListener);
    }

    /**
     * <p>Dispose of the popup window. This method can be invoked only after
     * {@link #showAsDropDown(android.view.View)} has been executed. Failing that, calling
     * this method will have no effect.</p>
     *
     * @see #showAsDropDown(android.view.View)
     */
    @SuppressLint("NewApi")
    @Override
    public void dismiss()
    {
        synchronized (this.syncObject)
        {
            if (!dismissed)
            {
                dismissed = true;
                touchOutsideDismissalTimer = null;
                autoDismissalTimer = null;
                openPopups.remove(this);

                View anchor = this.anchorRef.get();
                this.anchorRef.clear();

                if (anchor != null && Utils.isAndroidVersionAtLeast(16))
                {
                    anchor.getViewTreeObserver().removeOnGlobalLayoutListener(this);
                }
            }
            else
            {
                return;
            }
        }

        // Because view detached exception might be thrown
        try
        {
            AnalyticsReporter.reportScreenEvent(this.creationContext.popupStateHolder);
            super.dismiss();
        }
        catch (Exception e)
        {
            Log.w(true, TAG, "caught Exception while dismissing popup", e);
        }
    }

    /**
     * Tries to cancel the dismissal of this instance.
     */
    public void tryCancelDismissal()
    {
        if (!this.timingOptions.delayTouchOutsideDismissal || dismissed) { return; }

        Timer activeTimer = touchOutsideDismissalTimer;

        if (activeTimer != null)
        {
            activeTimer.cancel();
        }
    }

    @SuppressLint("NewApi")
    @Override
    public void onGlobalLayout()
    {
        synchronized (this.syncObject)
        {
            if (!this.dismissed)
            {
                if (AniwaysStatics.getApplicationContext().getResources().getConfiguration().orientation == Configuration.ORIENTATION_LANDSCAPE)
                {
                    updatePopupStateBeforeClosing("orientation-changed");
                    dismiss();
                    return;
                }

                Rect bounds = calculateTargetBounds(this.anchorRef.get());
                try {
                    if(this.isShowing()) {
                        update(bounds.left, bounds.top, bounds.width(), bounds.height(), true);
                    }
                }
                catch(IllegalArgumentException ex){
                    Log.e(true, TAG, "Caught popup not attached to window error when updating popup", ex);
                    try {
                        if(Utils.isAndroidVersionAtLeast(16)) {
                            View anchor = anchorRef.get();
                            if(anchor != null) {
                                anchor.getViewTreeObserver().removeOnGlobalLayoutListener(this);
                            }
                        }
                    }
                    catch(Throwable ex2){

                    }
                }
            }
        }
    }

    protected abstract Rect calculateTargetBounds(View anchor);

    /**
     * Dismisses any open popups.
     *
     * @return A value indicating whether any popup was actually dismissed by this call.
     */
    public static boolean dismissAllOpenPopups()
    {
        boolean retClosedSomething = false;

        HashSet<AniwaysPopupWindow> windows = new HashSet<>(openPopups);

        for (AniwaysPopupWindow popup : windows)
        {
            if (popup.isShowing())
            {
                retClosedSomething = true;
                popup.dismiss();
            }
        }

        return retClosedSomething;
    }

    public static void cancelAutoDismissOfAllOpenQuickActions()
    {
        HashSet<AniwaysPopupWindow> windows = new HashSet<>(openPopups);

        for (AniwaysPopupWindow popup : windows)
        {
            popup.tryCancelDismissal();
        }
    }

    @SuppressLint("NewApi")
    public void show(View anchor)
    {
        synchronized (this.syncObject)
        {
            this.anchorRef = new WeakReference<>(anchor);
            anchor.getViewTreeObserver().addOnGlobalLayoutListener(this);

            Rect bounds = calculateTargetBounds(anchor);
            setHeight(bounds.height());
            setWidth(bounds.width());
            try
            {
                super.showAtLocation(anchor, Gravity.TOP | Gravity.START, bounds.left, bounds.top);
                creationContext.startTimeUserInPopup = System.currentTimeMillis();
                openPopups.add(this);
                scheduleAutoDismissalTimer();
            }
            catch (WindowManager.BadTokenException ex)
            {
                Log.e(true, TAG, "Caught bad token Exception while trying to show popup", ex);
                try {
                    if(Utils.isAndroidVersionAtLeast(16)) {
                        anchor.getViewTreeObserver().removeOnGlobalLayoutListener(this);
                    }
                }
                catch(Throwable ex2){

                }
            }
        }
    }

    /**
     * A timer task that dismisses a specified popup window.
     */
    private class DismissPopupTask extends TimerTask
    {
        // Holds the popup window to be dismissed.
        private final AniwaysPopupWindow mPopupToDismiss;

        public DismissPopupTask(AniwaysPopupWindow popupToDismiss)
        {
            mPopupToDismiss = popupToDismiss;
        }

        /**
         * The task to run should be specified in the implementation of the {@code run()}
         * method.
         */
        @Override
        public void run()
        {
            ((Activity)mPopupToDismiss.context).runOnUiThread(new Runnable()
            {
                @Override
                public void run()
                {
                    updatePopupStateBeforeClosing("popup-closed-timeout");
                    mPopupToDismiss.dismiss();
                }
            });
        }
    }

    /**
     * Holds information regarding how the popup should behave, in terms of timing and user interaction.
     */
    protected static class PopupTimingOptions
    {
        // Defines the default time, in milliseconds, during which a touch-outside-dismissal can be cancelled.
        private static final int DefaultTouchOutsideDismissalDelay = 400;

        // Defines the default time, in milliseconds, after which the popup will auto-dismiss.
        private static final int DefaultAutoDismissalDelay = 60 * 1000;

        // Holds a value indicating whether the popup should delay before dismissing when the user touches outside of the popup bounds.
        public final boolean delayTouchOutsideDismissal;

        // In case touch-outside dismissal is desired, defines how long that delay should be, in Milliseconds.
        public final int touchOutsideDismissalDelay;

        // A value indicating whether the popup should automatically dismiss after a specified auto-dismissal delay.
        public final boolean autoDismiss;

        // In case auto-dismissal is desired, defines how long should the popup wait before dismissing.
        public final int autoDismissalDelay;

        // A value indicating whether user interaction with the popup should reset the auto-dismissal time frame.
        public final boolean extendAutoDismissOnTouch;

        /**
         * Initializes a new instance of PopupTimingOptions.
         */
        public PopupTimingOptions()
        {
            this(true, true, true);
        }

        /**
         * Initializes a new instance of PopupTimingOptions.
         *
         * @param delayTouchOutsideDismissal A value indicating whether the popup should delay before dismissing when the user touches outside of the popup bounds.
         * @param autoDismiss A value indicating whether the popup should automatically dismiss after a specified auto-dismissal delay.
         * @param extendAutoDismissOnTouch A value indicating whether user interaction with the popup should reset the auto-dismissal time frame.
         */
        public PopupTimingOptions(boolean delayTouchOutsideDismissal, boolean autoDismiss, boolean extendAutoDismissOnTouch)
        {
            this(delayTouchOutsideDismissal, DefaultTouchOutsideDismissalDelay, autoDismiss, DefaultAutoDismissalDelay, extendAutoDismissOnTouch);
        }

        /**
         * Initializes a new instance of PopupTimingOptions.
         *
         * @param delayTouchOutsideDismissal A value indicating whether the popup should delay before dismissing when the user touches outside of the popup bounds.
         * @param touchOutsideDismissalDelay In case touch-outside dismissal is desired, defines how long that delay should be, in Milliseconds.
         * @param autoDismiss A value indicating whether the popup should automatically dismiss after a specified auto-dismissal delay.
         * @param autoDismissalDelay In case auto-dismissal is desired, defines how long should the popup wait before dismissing.
         * @param extendAutoDismissOnTouch A value indicating whether user interaction with the popup should reset the auto-dismissal time frame.
         */
        public PopupTimingOptions(boolean delayTouchOutsideDismissal, int touchOutsideDismissalDelay, boolean autoDismiss, int autoDismissalDelay, boolean extendAutoDismissOnTouch)
        {
            this.delayTouchOutsideDismissal = delayTouchOutsideDismissal;
            this.touchOutsideDismissalDelay = touchOutsideDismissalDelay;
            this.autoDismiss = autoDismiss;
            this.autoDismissalDelay = autoDismissalDelay;
            this.extendAutoDismissOnTouch = extendAutoDismissOnTouch;
        }
    }
}