package com.aniways.settings;

import android.app.AlertDialog;
import android.content.DialogInterface;
import android.content.SharedPreferences;
import android.os.Bundle;
import android.preference.CheckBoxPreference;
import android.preference.Preference;
import android.preference.PreferenceActivity;
import android.preference.SwitchPreference;

import com.aniways.Aniways;
import com.aniways.Log;
import com.aniways.R;
import com.aniways.Utils;
import com.aniways.data.AniwaysClientConfig;
import com.aniways.data.AniwaysPrivateConfig;
import com.aniways.service.utils.AniwaysAction;
import com.aniways.service.utils.AniwaysServiceUtils;
import com.aniways.ui.views.AniwaysSeekBarPreference;

//TODO: add 'reset to defaults option'
public class AniwaysSettingsActivity extends PreferenceActivity implements SharedPreferences.OnSharedPreferenceChangeListener {

    private static final String TAG = "AniwaysSettingsActivity";
    private Preference contextualSuggestionSwitch;

    private Preference autoSuggestionSwitch;
    private AniwaysSeekBarPreference seekBarPreference;

    private String contextualSuggestionsEnableKey;
    private String autoSuggestionsEnableKey;
    private String iconSizeKey;
    private boolean iconSizeDialogAlreadyDisplayed;


    @Override
    @SuppressWarnings("deprecation")
    protected void onPostCreate(Bundle savedInstanceState) {
        try {
            super.onPostCreate(savedInstanceState);

            //Use our sharedPreferences file.
            getPreferenceManager().setSharedPreferencesName(AniwaysServiceUtils.SHARED_PREFERENCES);

            //inflate the aniways_pref view
            setContentView(R.layout.aniways_settnigs_main_layout);
            addPreferencesFromResource(R.xml.aniways_pref);

            initMembers();
            setTheme(R.style.Base_Theme_AppCompat_Light);


        } catch (Exception ex) {
            Log.e(true,TAG, "Caught an Exception in onPostCreate of AniwaysSettingsActivity", ex);
        }
    }

    @SuppressWarnings("deprecation")
    private void initMembers() {
        //TODO: the initial values need to be taken from what is defined in the Aniways config
        //TODO: why do we need the keys from the strings?
        contextualSuggestionsEnableKey = getResources().getString(R.string.contextual_suggestion_key);
        autoSuggestionsEnableKey = getResources().getString(R.string.auto_suggest_key);
        iconSizeKey = getResources().getString(R.string.in_message_icon_size_key);

        contextualSuggestionSwitch = findPreference(contextualSuggestionsEnableKey);
        autoSuggestionSwitch = findPreference(autoSuggestionsEnableKey);
        seekBarPreference = (AniwaysSeekBarPreference) findPreference(iconSizeKey);

        setTitles();
        setValues();
    }

    private void setValues() {
        setSwitchMode(autoSuggestionSwitch, AniwaysPrivateConfig.getInstance().suggestionMode == AniwaysPrivateConfig.SuggestionMode.AutoDisplaySuggestions);
        setSwitchMode(contextualSuggestionSwitch, AniwaysPrivateConfig.getInstance().contextualIconSuggestionsEnabled);

        seekBarPreference.setDefaultValue(AniwaysPrivateConfig.getInstance().iconInTextViewHeight);
    }

    private void setTitles() {
        //override our titles if exist on the client
        AniwaysClientConfig clientConfig = AniwaysClientConfig.getInstance();

        contextualSuggestionSwitch.setTitle(clientConfig.getResourceString("is_aniways_enabled_title"));
        autoSuggestionSwitch.setTitle(clientConfig.getResourceString("is_auto_suggest_enabled_title"));
    }

    @SuppressWarnings("deprecation")
    @Override
    protected void onResume() {
        try {
            super.onResume();
            getPreferenceManager()
                    .getSharedPreferences()
                    .registerOnSharedPreferenceChangeListener(this);
            iconSizeDialogAlreadyDisplayed = false;
        } catch (Exception e) {
            Log.e(true, TAG, "Caught an Exception in onResume of AniwaysSettingsActivity", e);
        }
    }

    @SuppressWarnings("deprecation")
    @Override
    protected void onPause() {
        try {
            super.onPause();
            getPreferenceManager()
                    .getSharedPreferences()
                    .unregisterOnSharedPreferenceChangeListener(this);

        } catch (Exception e) {
            Log.e(true, TAG, "Caught an Exception in onPause of AniwaysSettingsActivity", e);
        }
    }

    @Override
    public void onSharedPreferenceChanged(SharedPreferences sharedPreferences, String key) {
        if (key.equals(contextualSuggestionsEnableKey)) {
            onContextualSuggestionSettingChanged();
        }
        else if (key.equals(autoSuggestionsEnableKey)){
            onAutoSuggestionSettingChanged();

        }else if (key.equals(iconSizeKey)){
            onIconSizeSettingChanged();
        }

        Log.d("AniwaysSettingsActivity", "key = " + key);
    }


    private void onIconSizeSettingChanged() {
        if (seekBarPreference == null){
            return;
        }

        int iconSizeInDp = seekBarPreference.getPersistedValue();
        Aniways.getConfiguration(getApplicationContext()).setIconsInTextViewSize(iconSizeInDp);
        if(!iconSizeDialogAlreadyDisplayed) {
            showChangesWillTakeEffectOnceYouLeaveAndReenterConversation();
        }
    }

    private void onAutoSuggestionSettingChanged() {
        final boolean isEnabledByUser = getUserSelection(autoSuggestionSwitch);
        Aniways.getConfiguration(getApplicationContext()).enableAutoPopup(isEnabledByUser);
    }

    private void onContextualSuggestionSettingChanged() {
        final boolean isEnabledByUser = getUserSelection(contextualSuggestionSwitch);

        if (!isEnabledByUser){
            showHowToActivateDialog(new AniwaysAction<Boolean>() {
                @Override
                public void call(Boolean isEnabledAfterDialog) {
                    Aniways.getConfiguration(getApplicationContext()).enableContextualSuggestions(false);
                }
            });
        }
        else {
            Aniways.getConfiguration(getApplicationContext()).enableContextualSuggestions(true);
        }
    }

    private boolean
    getUserSelection(Preference preference) {
        if ((Utils.isAndroidVersionAtLeast(14))) {
            return ((SwitchPreference)preference).isChecked();
        }
        else {
            return ((CheckBoxPreference)preference).isChecked();
        }
    }

    private void showHowToActivateDialog(final AniwaysAction<Boolean> callback) {
        AlertDialog.Builder builder = new AlertDialog.Builder(this);

        builder.setMessage(getString(R.string.disable_contextual_dialog_content))
                .setTitle(getString(R.string.disable_contextual_dialog_title));

        builder.setPositiveButton("Ok", new DialogInterface.OnClickListener() {
            public void onClick(DialogInterface dialog, int id) {
                //contextual is disabled
                callback.call(false);
            }
        });
        builder.setNegativeButton("Cancel", new DialogInterface.OnClickListener() {
            public void onClick(DialogInterface dialog, int id) {
                setSwitchMode(contextualSuggestionSwitch, true);
            }
        });

        AlertDialog dialog = builder.create();
        dialog.show();
    }

    private void showChangesWillTakeEffectOnceYouLeaveAndReenterConversation() {
        AlertDialog.Builder builder = new AlertDialog.Builder(this);

        builder.setMessage(getString(R.string.changes_will_take_effect_after_reentering_conversation_dialog_content))
                .setTitle(getString(R.string.changes_will_take_effect_after_reentering_conversation_dialog_title));

        builder.setPositiveButton("Ok", new DialogInterface.OnClickListener() {
            public void onClick(DialogInterface dialog, int id) {}
        });

        AlertDialog dialog = builder.create();
        dialog.show();
        iconSizeDialogAlreadyDisplayed = true;
    }

    private void setSwitchMode(Preference switchController, boolean isChecked){
        if (Utils.isAndroidVersionAtLeast(14)) {
            ((SwitchPreference)switchController).setChecked(isChecked);
        }
        else {
            ((CheckBoxPreference)switchController).setChecked(isChecked);
        }
    }
}
