package com.aniways.service.utils;

import com.aniways.Utils;
import com.aniways.data.AniwaysPhraseReplacementData;
import com.aniways.data.AniwaysPrivateConfig;
import com.aniways.service.AniwaysIntentService;

import android.Manifest;
import android.app.ActivityManager;
import android.app.ActivityManager.RunningServiceInfo;
import android.content.Context;
import android.content.SharedPreferences;
import android.content.SharedPreferences.Editor;
import android.content.pm.PackageManager;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.util.Log;

/**
 * Defines app-wide constants and utilities
 */
public final class AniwaysServiceUtils {

	public static final String SDK_VERSION = "2.3.09";

	// Name of shared preferences repository that stores persistent state 
	public static final String SHARED_PREFERENCES =
			"com.aniways.SHARED_PREFERENCES";

	// Name of the key in the shared preferences. that indicate that the 
	// keywords update has completed and the application need to update
	// its data structures to fit the new keywords
	public static final String KEY_UPDATE_REQUIRED = 
			"com.aniways.UPDATE_REQUIRED";

	// Name of the key in the shared preferences which holds the App id for the service
	public static final String KEY_APP_ID = 
			"com.aniways.APP_ID";

	// Name of the key in the shared preferences which holds the App id for the service
	public static final String KEY_LOGS_VERBOSITY = 
			"com.aniways.LOGS_VERBOSITY";

	// Name of the key in the shared preferences which holds the App id for the service
	public static final String KEY_TRY_USE_EXTERNAL_STORAGE_TO_CACHE_ICONS = 
			"com.aniways.TRY_USE_EXTERNAL_STORAGE_TO_CACHE_ICONS";

	// Name of the key in the shared preferences that contains the keywords file latest updated version.
	// This is updated by the backend service. This version will become the parsed new version after the data parser
	// parses it, and only then it will be used by the app
	public static final String KEY_KEYWORDS_VERSION =
			"com.aniways.KEYWORDS_VERSION";

	public static final String KEY_KEYWORDS_ETAG =
			"com.aniways.KEYWORDS_ETAG";

	// Name of the key in the shared preferences that contains the last parsed keywords version.
	public static final String KEY_PARSED_KEYWORDS_VERSION =
			"com.aniways.PARSED_KEYWORDS_VERSION";

	public static final String KEY_CONFIG_VERSION =
			"com.aniways.CONFIG_VERSION";

	public static final String KEY_CONFIG_ETAG =
			"com.aniways.CONFIG_ETAG";

	// Name of the key in the shared preferences that contains the last successful executed alarm date in milliseconds.
	public static final String KEY_LAST_SUCCESSFUL_CHECK_FOR_UPDATES = 
			"com.aniways.LAST_SUCCESSFUL_CHECK_FOR_UPDATES_UPDATES";

	// Name of the key in the shared preferences that contains the last user activity date in milliseconds.
	// Right now we consider sending a message as a user activity.
	// This is used to determine whether to sync with the server for new definitions
	public static final String KEY_LAST_USER_ACTIVITY = 
			"com.aniways.LAST_USER_ACTIVITY";

	private static final String KEY_LAST_SUCCESSFUL_ANALYTICS_FLUSH = 
			"com.aniways.LAST_SUCCESSFUL_ANALYTICS_FLUSH";

	// Name of the key in the shared preferences that contains data for the backend sync thread to manipulate from the
	// app process to trigger a call to the Shared Perfs changed listener in that process and update the data structures if necessary
	public static final String KEY_BACKEND_SYNC = 
			"com.aniways.BACKEND_SYNC";

	public static final String KEY_ARE_EMOTICONS_STORED_ON_EXTERNAL_STORAGE = "com.aniways.ARE_EMOTICONS_STORED_ON_EXTERNAL_STORAGE";

	// Just an empty string, because java doesn't have it as a constant
	public static final String EMPTY_STRING = "";



	// Aniways configuration file url
	public static String getConfigUrl(){
		return AniwaysPrivateConfig.getInstance().baseApiUrl + "configuration";
	}

	// Aniways keywords file url
	public static String getKeywordsUrl(){
		return AniwaysPrivateConfig.getInstance().baseApiUrl + "keywords";
	}

	// Keywords dir in the internal storage
	//public static final String KEYWORDS_DIR = "";
	public static final String KEYWORDS_DIR = "/aniways/keywords";

	// Keywords file name in the internal storage
	public static final String KEYWORDS_FILE = "keywords";

	// Old assets dir in the internal storage
	public static final String OLD_ASSETS_DIR = "/assets";

	// Download files thread pool size (how many file download will the application do in parallel)
	public static final int THREAD_POOL_SIZE = 5;

	// Number of retries in case of a file download failure
	public static final int NUM_OF_DOWNLOAD_RETRIES = 10;

	// Interval between each file download retry in milliseconds
	public static final long INTERVAL_BETWEEN_DOWNLOAD_RETRY_MILLIS = 2500;

	// Interval of half a minute in milliseconds
	public static final long INTERVAL_HALF_MINUTE = 30000;

	public static final String NO_CHANGE_JSON_RESPONSE_STRING = "304";

	static final String NON_EXISTANT_APP_ID = "not set!!";

	private static final String TAG = "AniwaysServiceUtils";

	// Used to request the service to perform a sync (outside of the regular alarms)
	public static final String ACTION_SYNC_WITH_SERVER = "com.aniways.SYNC_WITH_SERVER";

	public static final long CONNECTION_MANAGER_TIMEOUT = 15000;

	public static final int READ_TIMEOUT = 15000;

	public static final int CONNECTION_TIMEOUT = 15000;

	private static final String KEY_APP_INSTALL_TIME = "com.aniways.APP_INSTALL_TIME";

	/**
	 * Update the UPDATE_REQUIRED value in the shared preferences with the given value
	 * @param context Application context
	 * @param isUpdateRequired New value to store
	 */
	public static synchronized void setUpdateRequired(Context context, boolean isUpdateRequired) {
		// open the shared preferences
		SharedPreferences prefs = 
				context.getSharedPreferences(SHARED_PREFERENCES, Utils.getSharedPreferencesFlags());
		Editor edit = prefs.edit();
		// write the new value
		edit.putBoolean(KEY_UPDATE_REQUIRED, isUpdateRequired);
		edit.commit();
	}

	/**
	 * return the value of UPDATE_REQUIRED from the shared preferences
	 * @param context Application context
	 * @return UPDATE_REQUIRED value from the shared preferences
	 */
	public static synchronized boolean getIsUpdateRequired(Context context) {
		// open the shared preferences
		SharedPreferences prefs = 
				context.getSharedPreferences(SHARED_PREFERENCES, Utils.getSharedPreferencesFlags());

		// return the values that stored there, in case that no value
		// is stored return false
		return prefs.getBoolean(KEY_UPDATE_REQUIRED, false);
	}

	public static boolean isServiceRunning(Context context) {
		ActivityManager manager = (ActivityManager) context.getSystemService(Context.ACTIVITY_SERVICE);
		for (RunningServiceInfo service : manager.getRunningServices(Integer.MAX_VALUE)) {
			if (AniwaysIntentService.class.getName().equals(service.service.getClassName()) && context.getPackageName().equals(service.service.getPackageName())) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Updates the last successful update time in the shared preferences.
	 * @param context Application context.
	 * @param time successful update time.
	 */
	public synchronized static void setLastSuccessfulUpdate(Context context, long time) {
		Log.i(TAG, "Setting last successful update time to: " + time);

		// open the shared preferences
		SharedPreferences prefs = 
				context.getSharedPreferences(AniwaysServiceUtils.SHARED_PREFERENCES, Utils.getSharedPreferencesFlags());
		Editor edit = prefs.edit();
		// update the new version
		edit.putLong(AniwaysServiceUtils.KEY_LAST_SUCCESSFUL_CHECK_FOR_UPDATES, time);
		edit.commit();
	}

	/**
	 * Retrieve the last successful update time from the shared preferences.
	 * @param context Application context.
	 * @return The assets file version as string.
	 */
	public synchronized static long getLastSuccessfulUpdate(Context context) {
		// open the shared preferences.
		SharedPreferences prefs = 
				context.getSharedPreferences(AniwaysServiceUtils.SHARED_PREFERENCES, Utils.getSharedPreferencesFlags());

		// return the last successful update time(0 if the value doesn't exists yet).
		return prefs.getLong(AniwaysServiceUtils.KEY_LAST_SUCCESSFUL_CHECK_FOR_UPDATES, 0);
	}
	
	/**
	 * Updates that the app install time in the shared preferences.
	 * @param context Application context.
	 */
	public synchronized static void setAppInstallTime(Context context, long installTime) {
		// open the shared preferences
		SharedPreferences prefs = 
				context.getSharedPreferences(AniwaysServiceUtils.SHARED_PREFERENCES, Utils.getSharedPreferencesFlags());
		Editor edit = prefs.edit();
		// update the new version
		edit.putLong(AniwaysServiceUtils.KEY_APP_INSTALL_TIME, installTime);
		edit.commit();
	}

	/**
	 * Retrieve the app install time from the shared preferences.
	 * @param context Application context.
	 * @return The app install time, 0 if was not installed before
	 */
	public synchronized static long getAppInstallTime(Context context, long now) {
		// open the shared preferences.
		SharedPreferences prefs = 
				context.getSharedPreferences(AniwaysServiceUtils.SHARED_PREFERENCES, Utils.getSharedPreferencesFlags());

		// return the app install time
		return prefs.getLong(AniwaysServiceUtils.KEY_APP_INSTALL_TIME, now);
	}
	
	public synchronized static void setLastSuccessAnalyticsFlush(Context context, long time) {
		Log.i(TAG, "Setting last successful analytics flush time to: " + time);

		// open the shared preferences
		SharedPreferences prefs = 
				context.getSharedPreferences(AniwaysServiceUtils.SHARED_PREFERENCES, Utils.getSharedPreferencesFlags());
		Editor edit = prefs.edit();
		// update the new version
		edit.putLong(AniwaysServiceUtils.KEY_LAST_SUCCESSFUL_ANALYTICS_FLUSH, time);
		edit.commit();
	}

	public synchronized static long getLastUserActivityTime(Context context) {
		// open the shared preferences.
		SharedPreferences prefs = 
				context.getSharedPreferences(AniwaysServiceUtils.SHARED_PREFERENCES, Utils.getSharedPreferencesFlags());
		
		// We would like to consider a user that has not done anything yet, as an active user.
		if(!prefs.contains(AniwaysServiceUtils.KEY_LAST_USER_ACTIVITY)){
			setLastUserActivityTime(context, System.currentTimeMillis());
		}
		
		// return the last successful user activity time(0 if the value doesn't exists yet).
		return prefs.getLong(AniwaysServiceUtils.KEY_LAST_USER_ACTIVITY, System.currentTimeMillis());
	}
	
	public synchronized static void setLastUserActivityTime(Context context, long time) {
		Log.i(TAG, "Setting last user activity time to: " + time);

		// open the shared preferences
		SharedPreferences prefs = 
				context.getSharedPreferences(AniwaysServiceUtils.KEY_LAST_USER_ACTIVITY, Utils.getSharedPreferencesFlags());
		Editor edit = prefs.edit();
		// update the new version
		edit.putLong(AniwaysServiceUtils.KEY_LAST_USER_ACTIVITY, time);
		edit.commit();
	}

	public synchronized static long getLastSuccessAnalyticsFlush(Context context) {
		// open the shared preferences.
		SharedPreferences prefs = 
				context.getSharedPreferences(AniwaysServiceUtils.SHARED_PREFERENCES, Utils.getSharedPreferencesFlags());

		// return the last successful update time(0 if the value doesn't exists yet).
		return prefs.getLong(AniwaysServiceUtils.KEY_LAST_SUCCESSFUL_ANALYTICS_FLUSH, 0);
	}

	/**
	 * Will sync if the service is not already running one of the following happens:
	 * 	1. We are set to force update (can be for debug) OR
	 *  2. The user is active (He sent a message within the past configured time) AND EITHER
	 *  	a. Keywords version is zero OR 
	 *  	b. Time from last update is bigger than defined interval (changes whether we are configured to save data and whether WIFI is used).
	 * @param context
	 * @return
	 */
	public static boolean shouldPerformSync(Context context) {

		AniwaysPrivateConfig aniwaysPrivateConfig = AniwaysPrivateConfig.getInstance();
		
		// If service is running don't update
		if(isServiceRunning(context)) {
			Log.i(TAG, "Not performing sync since service is running");
			return false;
		}
		
		// If configured to force update then update
		if(aniwaysPrivateConfig.forceUpdate) {
			Log.i(TAG, "Performing sync since forceUpdate = true");
			return true;
		}

		long timeFromLastUserActivity = System.currentTimeMillis() - getLastUserActivityTime(context);
		boolean isActiveUser = timeFromLastUserActivity <= aniwaysPrivateConfig.timeFromLastMessageBeforeStopSyncing;
		
		// If user is not active then don't update
		if(!isActiveUser){
			Log.i(TAG, "Not performing sync since user is not active. Time from last user activity: " + timeFromLastUserActivity + ". Threshold: " + aniwaysPrivateConfig.timeFromLastMessageBeforeStopSyncing);
			return false;
		}
		
		boolean keywordsVersionIsZero = AniwaysPhraseReplacementData.EMPTY_PARSER_VERSION.equalsIgnoreCase(AniwaysPhraseReplacementData.getParsedKeywordsVersion(context));
		//boolean keywordsVersionIsZero = false;
		//if(AniwaysPhraseReplacementData.isInit()){
		//	// It can be not initialized if we are currently after boot and aniways is inited as a service, or if the alarm created the process
		//	keywordsVersionIsZero = AniwaysPhraseReplacementData.getDataParser().getKeywordsVersion().equalsIgnoreCase(AniwaysPhraseReplacementData.EMPTY_PARSER_VERSION);
		//}
		
		// If keywords version is zero then update
		if(keywordsVersionIsZero){
			Log.i(TAG, "Performing sync since keywords version is zero");
			return true;
		}
		
		// Get the update interval, based on whether we are configured for restricted usage or not
		long updateInterval = aniwaysPrivateConfig.syncAlarmScheduleInterval;
		if(aniwaysPrivateConfig.restrictDataUsage && !isConnectedToWifi(context)){
			Log.v(TAG, "Restricted data usage update interval used");
			updateInterval = aniwaysPrivateConfig.syncAlarmScheduleIntervalRestricted;
		}
		
		// If last update is more than configured interval then update
		long elapsedTimeFromLastUpdate = System.currentTimeMillis() - getLastSuccessfulUpdate(context);
		boolean shouldSync = elapsedTimeFromLastUpdate >= updateInterval;
		
		if(!shouldSync){
			Log.i(TAG, "Not performing sync since not enough time passed since last update: " + elapsedTimeFromLastUpdate + ". In hours: " + (elapsedTimeFromLastUpdate / (1000 * 60 * 60)) + ". Interval: " + updateInterval + ". In hours: " + (updateInterval / (1000 * 60 * 60)));
		}
		else{
			Log.i(TAG, "Performing sync since enough time passed since last update: " + elapsedTimeFromLastUpdate + " . In hours: " + (elapsedTimeFromLastUpdate / (1000 * 60 * 60)) + ". Interval: " + updateInterval + ". In hours: " + (updateInterval / (1000 * 60 * 60)));
		}
		
		return shouldSync;
		
	}

	public static boolean shouldPerformFlush(Context context) {
		AniwaysPrivateConfig aniwaysPrivateConfig = AniwaysPrivateConfig.getInstance();

		if(aniwaysPrivateConfig.restrictDataUsage){
			long elapsedTimeFromLastFlush = System.currentTimeMillis() - getLastSuccessAnalyticsFlush(context);
			if(isConnectedToWifi(context) || elapsedTimeFromLastFlush > aniwaysPrivateConfig.analyticsFlushIntervalRestricted){
				return true;			
			} else {
				return false;
			}
		}

		return true;

	}

	private static boolean isConnectedToWifi(Context context) {
		if(PackageManager.PERMISSION_GRANTED == context.checkCallingOrSelfPermission(Manifest.permission.ACCESS_NETWORK_STATE)){
			ConnectivityManager connManager = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
			NetworkInfo mWifi = connManager.getNetworkInfo(ConnectivityManager.TYPE_WIFI);
			return mWifi.isConnected();
		} else{
			Log.e(TAG, "ACCESS_NETWORK_STATE permission not granted");
			return false;
		}
	}
}
