package com.aniways.service.task;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Field;
import java.net.HttpURLConnection;
import java.net.InetAddress;
import java.net.MalformedURLException;
import java.net.ProtocolException;
import java.net.Socket;
import java.net.URL;
import java.nio.charset.Charset;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.TimeZone;
import java.util.Map.Entry;
import java.util.zip.GZIPInputStream;

import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.StatusLine;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.params.HttpClientParams;
import org.apache.http.conn.params.ConnManagerParams;
import org.apache.http.conn.scheme.LayeredSocketFactory;
import org.apache.http.conn.scheme.Scheme;
import org.apache.http.conn.scheme.SocketFactory;
import org.apache.http.conn.ssl.SSLSocketFactory;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.params.HttpConnectionParams;
import org.apache.http.params.HttpParams;
import org.apache.http.params.HttpProtocolParams;
import org.json.JSONException;
import org.json.JSONObject;

import com.aniways.Log;
import com.aniways.Utils;
import com.aniways.analytics.AnalyticsReporter;
import com.aniways.analytics.info.Location;
import com.aniways.analytics.models.EasyJSONObject;
import com.aniways.data.AniwaysConfiguration.Verbosity;
import com.aniways.data.AniwaysPrivateConfig;
import com.aniways.data.Installation;
import com.aniways.service.helper.KeywordsHelper;
import com.aniways.service.utils.AniwaysServiceUtils;
import com.aniways.service.utils.FileUtils;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager.NameNotFoundException;
import android.net.http.AndroidHttpClient;
import android.os.AsyncTask;
import android.provider.Settings.Secure;

/**
 * Task to download and save json files from the server to the given path
 */
public class DownloadJSONFileTask extends AsyncTask<Void, Void, Map<String, Object>> {

	private static final String TAG = "AniwaysDownloadJSONFileTask";

	// The key that is used to extract the version string from the downloaded json files
	private static final String JSON_VERSION_KEY = "version";

	// Result map key for the status of the task (success or failure)
	public static final String STATUS_KEY = "status";
	// Result map key for the user to know if he should update the keywords file or not
	public static final String NEED_TO_UPDATE_KEY = "need_to_upadte";
	// Result map key for the version of the downloaded json file
	public static final String VERSION_KEY = "version";
	
	public static final String ETAG_KEY = "etag";

	public static final String ANDROID = "Android";

	// The url to download the json file from
	private String mUrl;
	// The destination path to save the json file to
	private String mDestFile;
	// The current version of the json file in the device
	private String mCurrentVersion;
	// The current etag of the json in the device
	private String mCurrentEtag;
	private String mNewEtag = "";
	
	private Context mContext;

	private class HttpRequestPreperationParams{
		public HttpClient httpClient = null;
		public String responseHeaders = null;
		public long responseContentLength = -1;
		public boolean isContentGziped = false;
		public String responseStatusLine = null;
		public InputStream inputStream = null;
		public InputStream deflatedInputream = null;
		public HttpURLConnection urlConnection = null;
		public String contentType = null;
		public int responseCode = -1;
		public String contentEncoding = null;
		public Charset responseCharset = Charset.defaultCharset();
		public Charset requestCharset = Charset.defaultCharset();
		public String userAgent;
		public String responseError = null;
		public boolean encounteredErrorCreatingZipStream = false;
		public int retryNumber = 0;
		public String responseEtag = null;

		public HttpRequestPreperationParams(String userAgent, int retryNumber, boolean encounteredErrorCreatingZipStream) {
			this.userAgent = userAgent;
			this.retryNumber = retryNumber;
			this.encounteredErrorCreatingZipStream = encounteredErrorCreatingZipStream;
		}
	}

	/**
	 * Constructor for the DownloadJSONFileTask.
	 * @param url The url to download the json file from.
	 * @param currentVersion The current version of the json file in the device.
	 * @throws java.io.IOException
	 */
	public DownloadJSONFileTask(Context context, String url, String destFileDir, String destFileName, String currentVersion, String currentEtag) throws IOException {
		mUrl = url;
		mDestFile = destFileDir + "/" + destFileName;
		mCurrentVersion = currentVersion;
		mCurrentEtag = currentEtag;
		mContext = context;

		// Create the dest dir
		File destDirFile = new File(destFileDir);
		if (!destDirFile.exists()) {
			if(!destDirFile.mkdirs()){
				throw new IOException("Could not create dest dir for Json file: " + destDirFile.getAbsolutePath());
			}
			Log.i(TAG, "Created dest dir for json file: " + destDirFile.getAbsolutePath());
		}
	}

	/**
	 * Download the json file from the given url and saves it to the given path
	 * @return Map that contains info about the task status (succeeded or failed),
	 * whether the user should update the keywords file, and what is the version of the downloaded json file.
	 */
	@Override
	protected Map<String, Object> doInBackground(Void... params) {

		// Initialize the result map with default values (failure values)
		Map<String, Object> resultMap = new HashMap<String, Object>();
		resultMap.put(STATUS_KEY, false);
		resultMap.put(NEED_TO_UPDATE_KEY, false);
		resultMap.put(VERSION_KEY, mCurrentVersion);
		resultMap.put(ETAG_KEY, mCurrentEtag);

		try{

			// Download the json file
			String jsonString = null;
			try{
				jsonString = getJsonString();
			}
			catch (Exception ex){
				Log.e(true, TAG, "Error downloading Json " + mUrl, ex);
				return resultMap;
			}

			// Check if json string is null - if it is then return an empty result map - make the log silent cause there has got to be some other log with an error sent to analytics
			if(jsonString == null){
				Log.e(false, TAG, "Returning empty result map cause received a null Json string for Url: " + mUrl);
				return resultMap;
			}

			// Check if the json string is not empty in order to be sure that the download succeeded
			if (jsonString.equals(AniwaysServiceUtils.EMPTY_STRING)) {
				Log.e(true, TAG, "Returning empty result map cause received an empty Json string for Url: " + mUrl);
				return resultMap;
			}

			// Check if received empty response from the server
			if (jsonString.equals(AniwaysServiceUtils.NO_CHANGE_JSON_RESPONSE_STRING)) {
				Log.v(TAG, "Received empty response from server, so has the same version. Returning result that doesn't need update");
				resultMap.put(STATUS_KEY, true);
				return resultMap;
			}
			else {
				JSONObject json;
				String newVersion = mCurrentVersion;

				try {
					// Check if the json string is actually a json, in order to know
					// if we got a good response from the server
					json = new JSONObject(jsonString);
					// extract the new json version
					newVersion = json.getString(JSON_VERSION_KEY);
					Log.v(TAG, "Comparing Json versions. Existing: " + mCurrentVersion + ". New: " + newVersion);
					// TODO: make sure the client version is not more advanced than the server and issue error if it is..
					if (newVersion.equals(mCurrentVersion)) {
						// we already have the file so the task succeeded
						resultMap.put(STATUS_KEY, true);
						// TODO: after change in which there is check of version of server b4 download this should be changed into an error!!
						Log.i(TAG, "Downloaded Json with current version so returning result map saying no need to update.  Json Url: " + mUrl);
						return resultMap;
					}
					else{
						Log.v(TAG, "Versions not the same, so need to update");
					}
					// We did not got a valid json string, the task failed
				} catch (JSONException e) {
					Log.w(true, TAG, "Bad Json string " + jsonString, e);
					return resultMap;
				}

				// Save the json string into the given file
				if (FileUtils.saveDataToFile(new File(mDestFile), jsonString)) {
					// put success values in the result map
					resultMap.put(STATUS_KEY, true);
					resultMap.put(NEED_TO_UPDATE_KEY, true);
					resultMap.put(VERSION_KEY, newVersion);
					resultMap.put(ETAG_KEY, mNewEtag );
					Log.i(TAG, "Saved Json from Url: " + mUrl + " to file: " + mDestFile);
					return resultMap;
				} else {
					// saving to the file operation failed, return the map
					// with the failure values.
					Log.e(false, TAG, "Save Json failed from Url: " + mUrl + " to file: " + mDestFile);
					return resultMap;
				}
			}
		}
		catch(Throwable ex){
			Log.e(true, TAG, "Encountered and unexpected error while downloading a Json file: " + mUrl + " to location on device: " + mDestFile, ex);
			return resultMap;
		}
	}

	/**
	 * Request from the server the newest json version, and returns it as a string
	 * @return String with the json response that we got from the server
	 */
	@SuppressLint("NewApi")
	private String getJsonString() {
		int numOfRetries = 1;
		String jsonString = null;
		String userAgent = createUserAgentString();
		// This init is never used, but needed for use in the catch clauses, if nothing is created.
		HttpRequestPreperationParams preperationParams = new HttpRequestPreperationParams(userAgent, 0, false);

		// try to download the json file few times if failed
		while (AniwaysServiceUtils.NUM_OF_DOWNLOAD_RETRIES >= numOfRetries) {
			try {
				Log.i(TAG, "Downloading JSON file: " + mUrl + ". Retry number: " + numOfRetries + " out of possible " + AniwaysServiceUtils.NUM_OF_DOWNLOAD_RETRIES);
				long startTime = System.currentTimeMillis();

				preperationParams = new HttpRequestPreperationParams(userAgent, numOfRetries, preperationParams.encounteredErrorCreatingZipStream);

				// Use Use Apache client for older Android versions and HttpURLConnection for newer ones
				if(Utils.isAndroidVersionAtLeast(9)){
					performHttpUrlRequest(preperationParams);
				}
				else{
					performHttpClientRequest(preperationParams);
				}

				String responseParams = createResponseParamsString(preperationParams);
				Log.v(TAG, "Response: " + responseParams);

				// Get the new Json from the server if necessary
				if(preperationParams.responseCode == HttpURLConnection.HTTP_NOT_MODIFIED){
					Log.i(TAG, "Received 304 from server");
					// TODO: find a better way to convey to upper levels that update in not necessary
					jsonString = AniwaysServiceUtils.NO_CHANGE_JSON_RESPONSE_STRING;
				}
				else{
					if(preperationParams.responseCode != HttpURLConnection.HTTP_OK){
						throw new IOException("Received response code which is not 200. params are: " + responseParams);
					}

					jsonString = this.getJsonString(preperationParams.deflatedInputream, preperationParams.responseCharset);
					if (jsonString.equals(AniwaysServiceUtils.EMPTY_STRING)) {
						throw new IOException("Received empty Json string: " + jsonString + ". Response params: " + responseParams);
					}
					mNewEtag = preperationParams.responseEtag == null ? "" : preperationParams.responseEtag; 
					AnalyticsReporter.ReportTiming(Verbosity.Statistical, startTime, "Performance", "Download Json and Read Response Time", String.valueOf(jsonString.length()), TAG, "number of chars");
				}
				break;
			} 
			catch (MalformedURLException e) {
				Log.e(numOfRetries == AniwaysServiceUtils.NUM_OF_DOWNLOAD_RETRIES, TAG, "Malformed URL: " + mUrl + " Retry number: " + (numOfRetries) + " out of: " + AniwaysServiceUtils.NUM_OF_DOWNLOAD_RETRIES, e);
			} 
			catch (ProtocolException e) {
				Log.e(numOfRetries == AniwaysServiceUtils.NUM_OF_DOWNLOAD_RETRIES, TAG, "Protocol Exception for url: " + mUrl + " Retry number: " + (numOfRetries) + " out of: " + AniwaysServiceUtils.NUM_OF_DOWNLOAD_RETRIES, e);
			}
			catch (IOException e) {
				String networkInterface = Utils.isInternetAvailable(mContext);
				if (networkInterface != null){
					String responseParams = createResponseParamsString(preperationParams);
					Log.w(numOfRetries == AniwaysServiceUtils.NUM_OF_DOWNLOAD_RETRIES, TAG, "IO Exception on URL " + mUrl + " Retry number: " + (numOfRetries) + " out of: " + AniwaysServiceUtils.NUM_OF_DOWNLOAD_RETRIES + " Network interface: " + networkInterface + "Response params: " + responseParams, e);
				}
				else{
					Log.w(false,TAG, "IO Exception, but there is no internet connection, so that's expected. URL: " + mUrl + " Retry number: " + (numOfRetries) + " out of: " + AniwaysServiceUtils.NUM_OF_DOWNLOAD_RETRIES, e);
				}
			} catch (IllegalArgumentException e) {
				Log.e(true, TAG, "Illegal arg Exception on URL " + mUrl + " Retry number: " + (numOfRetries) + " out of: " + AniwaysServiceUtils.NUM_OF_DOWNLOAD_RETRIES, e);
			} finally{
				if(preperationParams.deflatedInputream != null){
					try {
						preperationParams.deflatedInputream.close();
					} catch (IOException e) {
						Log.e(true, TAG, "Error while closing the deflated input string. is content zipped = " + preperationParams.isContentGziped, e);
					}
				}
				if (preperationParams.isContentGziped && preperationParams.inputStream != null){
					try {
						preperationParams.inputStream.close();
					} catch (IOException e) {
						Log.e(true, TAG, "Error while closing the input string. is content zipped = " + preperationParams.isContentGziped, e);
					}
				}
				if(preperationParams.httpClient != null && preperationParams.httpClient instanceof AndroidHttpClient){
					((AndroidHttpClient)preperationParams.httpClient).close();
				}
				if(preperationParams.urlConnection != null){
					try{
						preperationParams.urlConnection.disconnect();
					}
					catch(Throwable e){
						Log.e(true, TAG, "Error while disconnection HttpUrlConnection", e);
					}
				}
			}

			// We are here only after Exception

			// Could not download the file, retry.
			numOfRetries++;

			// wait between each retry
			try {
				Thread.sleep(AniwaysServiceUtils.INTERVAL_BETWEEN_DOWNLOAD_RETRY_MILLIS);
			} catch (InterruptedException e) {
				Log.w(true, TAG, "Thread interrupted while waiting for next retry downloading Json: " + mUrl, e);
			}
		}

		// Return the result that we got from the server, or null
		return jsonString;
	}

	/**
	 * @return
	 * @throws java.io.IOException
	 * @throws java.net.MalformedURLException
	 */
	private void performHttpUrlRequest(HttpRequestPreperationParams preperationParams) throws IOException {
		// If the HTTP response code is -1, then something went wrong with connection and response handling. The HttpURLConnection implementation is somewhat buggy with keeping connections alive. You may want to turn it off by setting the http.keepAlive system property to false. You can do this programmatically in the beginning of your application by:
		System.setProperty("http.keepAlive", "false"); 
		HttpURLConnection urlConnection = (HttpURLConnection) new URL(mUrl).openConnection();
		preperationParams.urlConnection = urlConnection;
		urlConnection.setRequestProperty("Accept-Charset", preperationParams.requestCharset.displayName(Locale.US));
		//urlConnection.setRequestProperty("Accept-Encoding", "gzip");
		
		if(this.shouldRequestNonZippedContent(preperationParams.retryNumber, preperationParams.encounteredErrorCreatingZipStream)){
			// TODO: Only if not requiring gzip, otherwise will set this to require gzip by default
			urlConnection.setRequestProperty("Accept-Encoding", "");
		}
		urlConnection.setRequestProperty("User-Agent", preperationParams.userAgent);
		if(mCurrentEtag.length() > 0){
			urlConnection.setRequestProperty("If-None-Match", mCurrentEtag);
		}
		urlConnection.setRequestProperty("Cache-Control", "max-age=0, no-cache, no-store");
		urlConnection.setRequestProperty("Pragma", "no-cache");
		urlConnection.setReadTimeout(AniwaysServiceUtils.READ_TIMEOUT);
		urlConnection.setConnectTimeout(AniwaysServiceUtils.CONNECTION_TIMEOUT);
		HttpURLConnection.setFollowRedirects(true);
		//if (Utils.isAndroidVersionAtLeast(8)){
			// Fix bug prior to Froyo
			//urlConnection.setRequestProperty("Connection", "Keep-Alive");
		//}
		//else{
			System.setProperty("http.keepAlive", "false");
		///}

		try{// All these can throw Exceptions
			long startTime = System.currentTimeMillis();
			urlConnection.connect();
			
			preperationParams.responseCode = urlConnection.getResponseCode();
			if(preperationParams.responseCode == HttpURLConnection.HTTP_OK){
				try{
					preperationParams.deflatedInputream = urlConnection.getInputStream();
					String contentLengthString = "-1";
					// This is in try-catch that swallws because this is only for logging and we parse the entity later, after getting more info, and then we allow the exception to be thrown
					try{
						contentLengthString = String.valueOf(urlConnection.getContentLength());
					}
					catch(Throwable ex){
						Log.i(TAG, "Caught Exception while getting entity for logging: " + ex.getStackTrace());
					}
					AnalyticsReporter.ReportTiming(Verbosity.Statistical, startTime, "Performance", "Download Json Time", contentLengthString, TAG, "number of chars");
				}
				catch(IOException ex){
					/// This might not necessarily be because of deflation error, but can't know for sure, so assuming it is and then
					// will eventually ask for non compressed version sometime
					preperationParams.encounteredErrorCreatingZipStream = true;
					throw ex;
				}
			}
		}
		finally{
			try{
				if(urlConnection.getHeaderFields() != null && urlConnection.getHeaderFields().entrySet() != null){
					StringBuilder responseHeadersSb = new StringBuilder();
					for (Entry<String, List<String>> header : urlConnection.getHeaderFields().entrySet()) {
						responseHeadersSb.append("\n" + header.getKey() + ":" + header.getValue());
					}
					preperationParams.responseHeaders = responseHeadersSb.toString();
				}

				preperationParams.responseStatusLine = urlConnection.getHeaderField(0);
				preperationParams.contentEncoding = urlConnection.getContentEncoding();
				preperationParams.contentType = urlConnection.getContentType();
				preperationParams.responseContentLength = urlConnection.getContentLength();
				populateResponseCharset(preperationParams);
				populateIsContentGzipped(preperationParams);
				preperationParams.responseEtag = urlConnection.getHeaderField("ETag");
			}
			catch(Throwable ex){
				Log.e(true, TAG, "Error while getting response params", ex);
			}
			try{
				InputStream errorStream = urlConnection.getErrorStream();
				populateResponseError(preperationParams, errorStream);
			}
			catch(Throwable ex){
				Log.e(true, TAG, "Error while getting response error", ex);
			}
		}
	}

	@SuppressLint("NewApi")
	private void performHttpClientRequest(HttpRequestPreperationParams preperationParams) throws IllegalStateException, IOException{
		HttpClient httpClient = null;
		if(Utils.isAndroidVersionAtLeast(8)){
			httpClient = AndroidHttpClient.newInstance(preperationParams.userAgent);
		}
		else{
			httpClient = new DefaultHttpClient();
			HttpProtocolParams.setUserAgent(httpClient.getParams(), preperationParams.userAgent);
		}

		ConnManagerParams.setTimeout(httpClient.getParams(), AniwaysServiceUtils.CONNECTION_MANAGER_TIMEOUT);
		HttpConnectionParams.setSoTimeout(httpClient.getParams(), AniwaysServiceUtils.READ_TIMEOUT); 
		HttpConnectionParams.setConnectionTimeout(httpClient.getParams(), AniwaysServiceUtils.CONNECTION_TIMEOUT); 
		HttpClientParams.setRedirecting(httpClient.getParams(), true);

		preperationParams.httpClient = httpClient;
		System.setProperty("http.keepAlive", "false");
		HttpGet request = new HttpGet(mUrl);

		// Every 3 attempts request a non zipped content (if we had an IO Exception while creating the zip stream)
		if(shouldRequestNonZippedContent(preperationParams.retryNumber, preperationParams.encounteredErrorCreatingZipStream)){
			request.removeHeaders("Accept-Encoding");
			Log.w(true, TAG, "Requesting non zipped content. Retry number: " + preperationParams.retryNumber);
		}else{
			if(Utils.isAndroidVersionAtLeast(8)){
				AndroidHttpClient.modifyRequestToAcceptGzipResponse(request);
			}
			else{
				request.setHeader("Accept-Encoding", "gzip");
			}
		}
		request.setHeader("Accept-Charset", preperationParams.requestCharset.displayName(Locale.US));
		if(mCurrentEtag.length() > 0){
			request.setHeader("If-None-Match", mCurrentEtag);
		}
		request.setHeader("Cache-Control", "max-age=0, no-cache, no-store");
		request.setHeader("Pragma", "no-cache");

		if(Utils.isAndroidVersionAtLeast(11)){
			// should never happen, cause then we use the HttpUrlConnection class and this method is not called
			throw new IOException("Using apache client for API version >= 9");
		}
		else{
			// Fix https reverse dns bug..
			workAroundReverseDnsBugInHoneycombAndEarlier(httpClient);
		}

		// execute the request
		long startTime = System.currentTimeMillis();
		HttpResponse response = httpClient.execute(request);
		String contentLengthString = "-1";
		// This is in try-catch that swallws because this is only for logging and we parse the entity later, after getting more info, and then we allow the exception to be thrown
		try{
			HttpEntity entity = response.getEntity();
			if(entity != null){
				contentLengthString = String.valueOf(entity.getContentLength());
			}
		}
		catch(Throwable ex){
			Log.i(TAG, "Caught Exception while getting entity for logging: " + ex.getStackTrace());
		}
		AnalyticsReporter.ReportTiming(Verbosity.Statistical, startTime, "Performance", "Download Json Time", contentLengthString, TAG, "number of chars");

		Header contentEncoding = response.getFirstHeader("Content-Encoding");
		if (contentEncoding != null){
			preperationParams.contentEncoding = contentEncoding.getValue();
		}
		populateIsContentGzipped(preperationParams);
		StatusLine line = response.getStatusLine();
		if (line != null){
			preperationParams.responseStatusLine = String.format(Locale.US, "%s %d %s", line.getProtocolVersion() == null ? "null" : line.getProtocolVersion().getProtocol(), line.getStatusCode(), line.getReasonPhrase());
		}
		preperationParams.responseCode = response.getStatusLine() == null ? -1 : response.getStatusLine().getStatusCode(); 
		Header contentType = response.getFirstHeader("Content-Type");
		if (contentType != null){
			preperationParams.contentType = contentType.getValue();
		}
		
		Header etag = response.getFirstHeader("ETag");
		if (contentType != null){
			preperationParams.responseEtag  = etag.getValue();
		}
		
		populateResponseCharset(preperationParams);

		Header[] headers = response.getAllHeaders();
		StringBuilder responseHeadersSb = new StringBuilder();
		if(headers != null && headers.length > 0){
			for(Header i : headers) {
				responseHeadersSb.append("\n" + i.getName() + ":" + i.getValue());
			}
		}
		preperationParams.responseHeaders = responseHeadersSb.toString();

		if(preperationParams.responseCode == HttpURLConnection.HTTP_OK){
			HttpEntity entity = response.getEntity();
			if(entity == null){
				Log.w(false, TAG, "Null entity");
				throw new IllegalArgumentException("Null entity");
			}
			preperationParams.responseContentLength = entity.getContentLength();

			preperationParams.inputStream = entity.getContent();
			preperationParams.deflatedInputream = preperationParams.inputStream;

			// Unzip the jSon
			if (preperationParams.isContentGziped){
				try{
					preperationParams.deflatedInputream = new GZIPInputStream(preperationParams.inputStream, 1024);
				}
				catch(IOException e){
					preperationParams.encounteredErrorCreatingZipStream = true;
					throw e;
				}
			}
		}
	}

	private boolean shouldRequestNonZippedContent(int retryNumber, boolean encounteredErrorCreatingZipStream) {
		return encounteredErrorCreatingZipStream && (retryNumber % 3 == 0);
	}

	private void populateIsContentGzipped(HttpRequestPreperationParams preperationParams) {
		if (preperationParams.contentEncoding != null && preperationParams.contentEncoding.equalsIgnoreCase("gzip")){
			preperationParams.isContentGziped = true;
		}
	}

	/**
	 * @param preperationParams
	 */
	private void populateResponseCharset(HttpRequestPreperationParams preperationParams) {
		if(preperationParams.contentType != null){
			for (String param : preperationParams.contentType.replace(" ", "").split(";")) {
				if (param.startsWith("charset=")) {
					
					try{
						String charsetName = param.split("=", 2)[1];
						if (Charset.isSupported(charsetName)){
							preperationParams.responseCharset = Charset.forName(charsetName);
						}
					}
					catch(Throwable ex){
						Log.w(true, TAG, "Could not parse charset from: " + preperationParams.contentType, ex);
					}
					 
					break;
				}
			}
		}
	}

	/**
	 * @param preperationParams
	 */
	@SuppressLint("NewApi") private void populateResponseError(HttpRequestPreperationParams preperationParams, InputStream responseErrorStream) {
		try{
			if(responseErrorStream != null){
				byte[] response = readFully(responseErrorStream);
				String errorString = "";
				if(Utils.isAndroidVersionAtLeast(9)){
					errorString = new String(response, preperationParams.responseCharset);
				}
				else{
					errorString = new String(response, preperationParams.responseCharset.displayName(Locale.US));
				}
				preperationParams.responseError = errorString;
			}
		}
		catch(Throwable ex){
			Log.w(true, TAG, "Error while reading response error stream", ex);
		}
		finally{
			try{
				if(responseErrorStream != null){
					responseErrorStream.close();
				}
			}
			catch(Throwable e){
				Log.e(true, TAG, "Error while colsing response error stream", e);
			}
		}
	}

	/**
	 * @param preperationParams
	 * @return
	 */
	private String createResponseParamsString(HttpRequestPreperationParams preperationParams) {
		//TODO: make sure tht parameter definitions are correct (%s is appropriate where it is etc.)
		String responseParams = String.format(Locale.US, "\nResponse statusLine: %s, is content Gzipped: %s, Response content length: %s, Response ETag: %s, Response error: %s, response headers: %s",
				preperationParams.responseStatusLine,					//response code
				preperationParams.isContentGziped,						//is content gziped
				preperationParams.responseContentLength,				//response content length
				preperationParams.responseEtag,									//response etag
				preperationParams.responseError ,						//response error
				preperationParams.responseHeaders						//response headers
				);
		return responseParams;
	}

	/**
	 * @return
	 */
	private String createUserAgentString() {
		Context context = mContext;

		PackageInfo pi = null;
		try {
			pi = context.getPackageManager().getPackageInfo(context.getPackageName(), 0);
		} catch (NameNotFoundException e) {
			Log.e(true, TAG, "Package name doesn't exist");
		}

		// Get the first google account
		String userId = "unknown";
		String deviceInfo = "Device: " + android.os.Build.MANUFACTURER + " " + android.os.Build.MODEL + " " + android.os.Build.VERSION.RELEASE + " (api level " + android.os.Build.VERSION.SDK_INT + ")";
		//AccountManager am = AccountManager.get(context);
		//Account[] accounts = am.getAccountsByType(null);
		//if(accounts != null && accounts.length > 0){
		//	for (Account a: accounts) {
		//		if (a != null && a.name != null && a.name.contains("@gmail.com")) {
		//			// Cut everything after the @gmail.com..
		//			int endIndex = a.name.indexOf("@gmail.com") + 10;
		//			userId = a.name.substring(0, endIndex);
		//			break;
		//		}
		//	}
		//}

		String android_id = Secure.getString(this.mContext.getContentResolver(), Secure.ANDROID_ID);
		
		// Get install date
		long appInstallTime = AniwaysServiceUtils.getAppInstallTime(context, System.currentTimeMillis());
		SimpleDateFormat dateTimeFormat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss", Locale.US);
		dateTimeFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
		Calendar calendar = Calendar.getInstance();
		calendar.setTimeInMillis(appInstallTime);
		calendar.setTimeZone(TimeZone.getTimeZone("UTC"));
		String installDateString = dateTimeFormat.format(calendar.getTime());
		
		Location location = new Location();
		EasyJSONObject locationJson = location.get(this.mContext);
		
		String locationString = String.format(Locale.US, "latitude=%s;longitude=%s;speed=%s;countryCode=%s;countryName=%s;state=%s;county=%s;city=%s;neighborhood=%s;landmark=%s",
                locationJson == null ? Location.UNKNOWN : locationJson.getString(Location.LATITUDE, Location.UNKNOWN),
                locationJson == null ? Location.UNKNOWN : locationJson.getString(Location.LONGITUDE, Location.UNKNOWN),
                locationJson == null ? Location.UNKNOWN : locationJson.getString(Location.SPEED, Location.UNKNOWN),
                locationJson == null ? Location.UNKNOWN : locationJson.getString(Location.COUNTRY_CODE, Location.UNKNOWN),
                locationJson == null ? Location.UNKNOWN : locationJson.getString(Location.COUNTRY_NAME, Location.UNKNOWN),
                locationJson == null ? Location.UNKNOWN : locationJson.getString(Location.STATE, Location.UNKNOWN),
                locationJson == null ? Location.UNKNOWN : locationJson.getString(Location.COUNTY, Location.UNKNOWN),
                locationJson == null ? Location.UNKNOWN : locationJson.getString(Location.CITY, Location.UNKNOWN),
                locationJson == null ? Location.UNKNOWN : locationJson.getString(Location.NEIGHBORHOOD, Location.UNKNOWN),
                locationJson == null ? Location.UNKNOWN : locationJson.getString(Location.LANDMARK, Location.UNKNOWN));

		// TODO: add to the request the parameters:
		// sdk version, json file version, app name and version, app id
		//appName=Aniways;appVersion=1.5.2;deviceName=iPhone Simulator;osName=iPhone OS;osVersion=6.1;language=en_US;appId=1;keywordsVersion=1;screenRatio=1;sdkVersion=1.0
		String userAgent = String.format(Locale.US, "appName=%s;appVersion=%d;appVersionName=%s;configVersion=%s;configVersionName=%s;deviceName=%s;osName=%s;osVersion=%s;language=%s;appId=%s;keywordsVersion=%s;screenRatio=%s;sdkVersion=%s;installationId=%s;deviceId=%s;userId=%s;installDate=%s;%s;end=end",
				pi == null ? "Unknown" : pi.packageName,												//app name
						pi == null ? 0 : pi.versionCode,														//app version
								pi == null ? "Unknown" : pi.versionName,												//app version name
										AniwaysPrivateConfig.getInstance().version,										//config version
										AniwaysPrivateConfig.getInstance().versionName,									//config version name
										deviceInfo,																				//device name
										ANDROID,																				//os name
										String.valueOf(android.os.Build.VERSION.SDK_INT),										//os version
										Locale.getDefault().getDisplayLanguage() + "-" + Locale.getDefault().getLanguage(),		//language //TODO: get the languages that the keyboard supports
										AniwaysPrivateConfig.getInstance().appId,											//app id
										KeywordsHelper.getInstance().getKeywordsVersion(context),								//keywords version
										String.valueOf(context.getResources().getDisplayMetrics().densityDpi),					//screen ratio
										AniwaysServiceUtils.SDK_VERSION,														//sdk version
										Installation.id(context),																//installation id
										android_id == null ? "Unknown" : android_id,											//device id
										userId,																					//user id
										installDateString,																		//install date
										locationString																			//location info
												
				);
		return userAgent;
	}

	@SuppressLint("NewApi") private String getJsonString(InputStream is, Charset charset) throws IOException{
		byte[] response = readFully(is);
		if(Utils.isAndroidVersionAtLeast(9)){
			return new String(response, charset);
		}
		else{
			return new String(response, charset.displayName(Locale.US));
		}
	}

	private byte[] readFully(InputStream in) throws IOException {
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		byte[] buffer = new byte[1024];
		for (int count; (count = in.read(buffer)) != -1; ) {
			out.write(buffer, 0, count);
		}
		return out.toByteArray();
	}

	private void workAroundReverseDnsBugInHoneycombAndEarlier(HttpClient client) {
		// Android had a bug where HTTPS made reverse DNS lookups (fixed in Ice Cream Sandwich) 
		// http://code.google.com/p/android/issues/detail?id=13117
		SocketFactory socketFactory = new LayeredSocketFactory() {
			SSLSocketFactory delegate = SSLSocketFactory.getSocketFactory();
			@Override public Socket createSocket() throws IOException {
				return delegate.createSocket();
			}
			@Override public Socket connectSocket(Socket sock, String host, int port,
					InetAddress localAddress, int localPort, HttpParams params) throws IOException {
				return delegate.connectSocket(sock, host, port, localAddress, localPort, params);
			}
			@Override public boolean isSecure(Socket sock) throws IllegalArgumentException {
				return delegate.isSecure(sock);
			}
			@Override public Socket createSocket(Socket socket, String host, int port,
					boolean autoClose) throws IOException {
				injectHostname(socket, host);
				return delegate.createSocket(socket, host, port, autoClose);
			}
			private void injectHostname(Socket socket, String host) {
				try {
					Field field = InetAddress.class.getDeclaredField("hostName");
					field.setAccessible(true);
					field.set(socket.getInetAddress(), host);
				} catch (Exception ignored) {
				}
			}
		};
		client.getConnectionManager().getSchemeRegistry()
		.register(new Scheme("https", socketFactory, 443));
	}
}
