package com.aniways.service.helper;

import java.io.File;
import java.io.IOException;

import com.aniways.Log;
import com.aniways.Utils;
import com.aniways.data.AniwaysPhraseReplacementData;
import com.aniways.data.AniwaysStorageManager;
import com.aniways.service.utils.AniwaysServiceUtils;
import android.content.Context;
import android.content.SharedPreferences;
import android.content.SharedPreferences.Editor;

/**
 * Used to interact and change values related to the keywords file
 */
public class KeywordsHelper {
	
	// The class is designed by the singleton pattern.
	// This is the static instance of the class, that the user gets
	// when he will use the class.
	private static KeywordsHelper mInstance;
	private static final String TAG = "AniwaysKeywordsHelper";
	
	/**
	 * Class default constructor, private because of the singleton usage pattern.
	 */
	private KeywordsHelper() {
		
	}
	
	/**
	 * Get an instance of the KeywordsHelper class.
	 * @return The instance of the KeywordsHelper class.
	 */
	public static KeywordsHelper getInstance() {
		// if the instance doesn't exists, create it
		if (null == mInstance) {
			mInstance = new KeywordsHelper();
		}
		
		// return the instance
		return mInstance;
	}
	
	/**
	 * Updates the keywords version in the shared preferences.
	 * @param context Application context.
	 * @param version New version string.
	 */
	public synchronized void setKeywordsVersion(Context context, String version, String eTag) {
		Log.i(TAG, "Setting Keywords version to: " + version);
		
		// open the shared preferences
		SharedPreferences prefs = 
				context.getSharedPreferences(AniwaysServiceUtils.SHARED_PREFERENCES, Utils.getSharedPreferencesFlags());
		Editor edit = prefs.edit();
		// update the new version
		edit.putString(AniwaysServiceUtils.KEY_KEYWORDS_VERSION, version);
		edit.putString(AniwaysServiceUtils.KEY_KEYWORDS_ETAG, eTag);
		edit.commit();
	}

	/**
	 * Retrieve the keywords file version from the shared preferences.
	 * @param context Application context.
	 * @return The keywords file version as string.
	 */
	public synchronized String getKeywordsVersion(Context context) {
		// open the shared preferences.
		SharedPreferences prefs = 
				context.getSharedPreferences(AniwaysServiceUtils.SHARED_PREFERENCES, Utils.getSharedPreferencesFlags());

		// return the keywords file version ("0.0" if the value doesn't exists yet).
		return prefs.getString(AniwaysServiceUtils.KEY_KEYWORDS_VERSION, AniwaysPhraseReplacementData.EMPTY_PARSER_VERSION);
	}
	
	public synchronized String getKeywordsEtag(Context context) {
		// open the shared preferences.
		SharedPreferences prefs = 
				context.getSharedPreferences(AniwaysServiceUtils.SHARED_PREFERENCES, Utils.getSharedPreferencesFlags());

		// return the keywords file version ("0.0" if the value doesn't exists yet).
		return prefs.getString(AniwaysServiceUtils.KEY_KEYWORDS_ETAG, "");
	}
	
	/**
	 * Update the new downloaded keywords file to be the one that the sdk uses.
	 * @param context Application context.
	 * @return true is the update succeeded, false otherwise.
	 * @throws java.io.IOException
	 */
	public synchronized boolean upadateKeywordsFile(Context context, AniwaysStorageManager storageManager) throws IOException {
		// Copy the newly downloaded keywords file from the cache dir to the internal storage
		// where it will be used by the app.
		
		String source = storageManager.getKeywordsCacheDir().getAbsolutePath()  + "/" + AniwaysServiceUtils.KEYWORDS_FILE;
		String dest = storageManager.getKeywordsDir().getAbsolutePath() + "/" + AniwaysServiceUtils.KEYWORDS_FILE;
		
		File keywordsDirFile = new File(storageManager.getKeywordsDir().getAbsolutePath());
		if (!keywordsDirFile.exists()) {
			if(!keywordsDirFile.mkdirs()){
				throw new IOException("Could not create keywords dir: " + keywordsDirFile.getAbsolutePath());
			}
			Log.i(TAG, "Created keywords dir: " + keywordsDirFile.getAbsolutePath());
		}
		
		Log.i(TAG, "Copying new keywords file from " + source + " to: " + dest);
		
		return new File(source).renameTo(new File(dest));
		
	}
	
}
