package com.aniways.service;

import java.io.IOException;
import java.util.Map;
import java.util.concurrent.ExecutionException;

import android.content.Context;
import android.content.Intent;

import com.aniways.Log;
import com.aniways.Utils;
import com.aniways.VersionComparisonResult;
import com.aniways.analytics.AnalyticsReporter;
import com.aniways.analytics.GoogleAnalyticsReporter;
import com.aniways.data.AniwaysBackendSyncChecker;
import com.aniways.data.AniwaysConfiguration.Verbosity;
import com.aniways.data.AniwaysPrivateConfig;
import com.aniways.data.AniwaysStorageManager;
import com.aniways.service.helper.ConfigHelper;
import com.aniways.service.helper.KeywordsHelper;
import com.aniways.service.task.DownloadJSONFileTask;
import com.aniways.service.utils.AniwaysServiceUtils;

/**
 * IntentService class that implements equivalent Linux's cron job pattern, 
 * for the checking and downloading updates from the server, when it gets executed.
 */
public class AniwaysIntentService extends WakefulIntentService {
	private static final String TAG = "AniwaysIntentService";

	/**
	 * AniwaysIntentService class constructor.
	 */
	public AniwaysIntentService() {
		super(TAG);
	}

	/**
	 * Execute the keywords updates.
	 */
	@Override
	protected void doWakefulWork(Intent intent) {
		try{
			// Get when the last alarm executed successfully
			Context context = this.getApplicationContext();
			long lastSuccessfulAlarm = AniwaysServiceUtils.getLastSuccessfulUpdate(context);

			if (intent.getAction() != null && intent.getAction().equalsIgnoreCase(AniwaysServiceUtils.ACTION_SYNC_WITH_SERVER)) {
				Log.i(TAG, "Received request to sync with server from the app (not from the scheduled alarms)");
			}
			
			Log.i(TAG, "Checking for updates from server. Last successful check was: " + lastSuccessfulAlarm + ". Current time is: " + System.currentTimeMillis());
			
			boolean result = false;
			
			try{
				// Check for config from server
				result = checkForConfigFromServer(getApplicationContext());
				
				// Update config version if necessary
				final AniwaysPrivateConfig config = AniwaysPrivateConfig.getInstance();

				VersionComparisonResult configResult = null;
				String configVersionString = ConfigHelper.getInstance().getConfigVersion(getApplicationContext());
				try{
					configResult = Utils.compareVersionStrings(config.version, configVersionString);
				}
				catch(IllegalArgumentException e){
					Log.e(true, TAG, "Error parsing config version: " + configVersionString, e);
				}

				if (configResult != null && configResult.result == -1){
					Log.i(TAG, "detected lower version of config in current config: " + config.version + ". Scheduling parsing the new file");

					// Parse the config file
					AniwaysBackendSyncChecker.parseConfingAndKeywordsIfNecessary(context, true, false);
				}
				
			}
			catch (Exception e){
				Log.e(true, TAG, "Check for updates from server was not successful", e);
			}
			
			if(result){
				// Set separate finish time to config updates and update it here..
				Log.i(TAG, "Finished checking for config updates successfuly");
			}
			else{
				Log.w(false, TAG, "check for config updates from server was not successful");
			}
			
			
			result = checkForUpdates(getApplicationContext());

			if(result){
				long finishTime = System.currentTimeMillis();
				Log.i(TAG, "Finished checking for updates successfuly. Putting finish time in shared preferences: " + finishTime);
				// Put the last successful execution time as now in the shared preferences
				AniwaysServiceUtils.setLastSuccessfulUpdate(context, finishTime);
			}
			else{
				Log.w(false, TAG, "check for updates from server was not successful");
			}
		}
		catch (Exception e){
			Log.e(true, TAG, "Check for updates from server was not successful", e);
		}
	}

	/**
	 * Download the updated files from the server, delete the unnecessary files and notify
	 * when update is completed.
	 * @param context Application context.
	 * @param keywordsNewVersion Keywords file new version.
	 * @return whether successful or not.
	 * @throws java.io.IOException
	 */
	private boolean updateKeywords(Context context, String oldKeywordsVersion, String keywordsNewVersion, String newKeywordsEtag, AniwaysStorageManager storageManager) throws IOException {
		Log.i(TAG, "Update keywords. new keywords ver: " + keywordsNewVersion);

		boolean result = false;

		// Create keywords helper
		KeywordsHelper keywordsHelper = KeywordsHelper.getInstance();
		// Replace the old keywords file with the new one
		if (keywordsHelper.upadateKeywordsFile(context, storageManager)) {
			// Let the storage manager record where the latest version of keywords file sit
			storageManager.recordLocationOfNewKeywords();

			// Set the new version for the keywords files
			keywordsHelper.setKeywordsVersion(context, keywordsNewVersion, newKeywordsEtag);

			// Notify that the update was completed successfully
			AniwaysServiceUtils.setUpdateRequired(context, true);
			result = true;
			Log.i(TAG, "Finished updating keywords. New keywords version is: " + keywordsNewVersion);
			GoogleAnalyticsReporter.reportEvent(Verbosity.Info, "Statistics", "Backend Service: Finished Processing of Keywords Version", keywordsNewVersion, 0);
			AnalyticsReporter.reportFinishedProcessingKeywordsVersion(oldKeywordsVersion, keywordsNewVersion);
		}
		else{
			Log.w(false, TAG, "Update of keywords file failed");
		}
			
		return result;
	}

	/**
	 * Check for keywords file update, and if such update exists download it.
	 * @param context Application context.
	 * @return whether successful or not.
	 * @throws java.util.concurrent.ExecutionException
	 * @throws InterruptedException 
	 * @throws java.io.IOException
	 */
	private boolean checkForUpdates(Context context) throws InterruptedException, ExecutionException, IOException {
		boolean result = true;
		Log.v(TAG, "Starting Check for updates..");

		AniwaysStorageManager storageManager = AniwaysStorageManager.getInstance(context);

		// Download latest keywords json file from the server
		String oldKeywordsVersion = KeywordsHelper.getInstance().getKeywordsVersion(context);
		String oldKeywordsEtag = KeywordsHelper.getInstance().getKeywordsEtag(context);
		DownloadJSONFileTask downloadKeywordsFile =
				new DownloadJSONFileTask(context, AniwaysServiceUtils.getKeywordsUrl(),
						storageManager.getKeywordsCacheDir().getAbsolutePath(), 
						AniwaysServiceUtils.KEYWORDS_FILE,
						oldKeywordsVersion,
						oldKeywordsEtag);


		Log.i(TAG, "Scheduling download of Keywords file");
		Map<String, Object> downloadKeywordsFileResult = downloadKeywordsFile.execute().get();
		Log.i(TAG, "Finished download of keywords file - result is " + (Boolean)downloadKeywordsFileResult.get(DownloadJSONFileTask.STATUS_KEY) + ". is update required: " + (Boolean) downloadKeywordsFileResult.get(DownloadJSONFileTask.NEED_TO_UPDATE_KEY));
		// Check if download succeeded and if update is required
		if ((Boolean) downloadKeywordsFileResult.get(DownloadJSONFileTask.STATUS_KEY)) {
			if ((Boolean) downloadKeywordsFileResult.get(DownloadJSONFileTask.NEED_TO_UPDATE_KEY)) {
				String keywordsNewVersion = (String) downloadKeywordsFileResult
						.get(DownloadJSONFileTask.VERSION_KEY);
				String newKeywordsEtag = (String) downloadKeywordsFileResult
						.get(DownloadJSONFileTask.ETAG_KEY);

				// Update the keywords files.
				result &= updateKeywords(context, oldKeywordsVersion, keywordsNewVersion, newKeywordsEtag, storageManager);
			}
			else{
				Log.i(TAG, "No need to update keywords file");
			}
		}
		else{
			Log.e(false, TAG, "download keywords file error"); // Silent, cause there has got to be report of the actual problem below..
			result = false;
		}
		return result;
	}

	private boolean checkForConfigFromServer(Context context) throws InterruptedException, ExecutionException, IOException {
		boolean result = true;
		Log.v(TAG, "Starting Check for config..");
		
		String oldConfigVersion = ConfigHelper.getInstance().getConfigVersion(context);
		String oldConfigEtag = ConfigHelper.getInstance().getConfigEtag(context);

		// Download latest keywords json file from the server
		DownloadJSONFileTask downloadConfigFile =
				new DownloadJSONFileTask(context, AniwaysServiceUtils.getConfigUrl(),
						AniwaysStorageManager.getConfigCacheDirPath(context), 
						AniwaysStorageManager.getConfigFileName(),
						oldConfigVersion,
						oldConfigEtag);

		Log.i(TAG, "Scheduling download of config file");
		Map<String, Object> downloadConfigFileResult = downloadConfigFile.execute().get();
		Log.i(TAG, "Finished download of config file - result is " + (Boolean)downloadConfigFileResult.get(DownloadJSONFileTask.STATUS_KEY) + ". is update required: " + (Boolean) downloadConfigFileResult.get(DownloadJSONFileTask.NEED_TO_UPDATE_KEY));
		// Check if download succeeded and if update is required
		if ((Boolean) downloadConfigFileResult.get(DownloadJSONFileTask.STATUS_KEY)) {
			if ((Boolean) downloadConfigFileResult.get(DownloadJSONFileTask.NEED_TO_UPDATE_KEY)) {
				ConfigHelper helper = ConfigHelper.getInstance();
				AniwaysStorageManager storageManager = AniwaysStorageManager.getInstance(context);
				// Update the config file.
				if (helper.upadateConfigFile(context, storageManager)) {
					// Set the new version for the config file
					String configNewVersion = (String) downloadConfigFileResult.get(DownloadJSONFileTask.VERSION_KEY);
					String configNewEtag = (String) downloadConfigFileResult.get(DownloadJSONFileTask.ETAG_KEY);
					helper.setConfigVersion(context, configNewVersion, configNewEtag); 

					// Notify that the update was completed successfully
					AniwaysServiceUtils.setUpdateRequired(context, true);
					result = true;
					Log.i(TAG, "Finished updating config. New version is: " + configNewVersion);
					GoogleAnalyticsReporter.reportEvent(Verbosity.Info, "Statistics", "Backend Service: Finished Processing of config Version", configNewVersion, 0);
					AnalyticsReporter.reportFinishedProcessingConfigVersion(oldConfigVersion, configNewVersion);
				}
				else{
					Log.w(false, TAG, "Update of config file failed");
				}
				
			}
			else{
				Log.i(TAG, "No need to update config file");
			}
		}
		else{
			Log.e(false, TAG, "download config file error"); // Silent, cause there has got to be report of the actual problem below..
			result = false;
		}
		return result;
	}

}