package com.aniways.quick.action;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.Intent;
import android.graphics.Point;
import android.graphics.Rect;
import android.os.Handler;
import android.os.Looper;
import android.support.v7.widget.GridLayout;
import android.util.DisplayMetrics;
import android.view.Gravity;
import android.view.LayoutInflater;
import android.view.MotionEvent;
import android.view.View;
import android.view.View.MeasureSpec;
import android.view.View.OnClickListener;
import android.view.View.OnTouchListener;
import android.view.ViewGroup;
import android.view.ViewGroup.LayoutParams;
import android.view.ViewTreeObserver;
import android.view.animation.Animation;
import android.view.animation.AnimationUtils;
import android.widget.ImageView;
import android.widget.PopupWindow;
import android.widget.ProgressBar;
import android.widget.RelativeLayout;
import android.widget.TextView;
import android.widget.Toast;

import com.aniways.IconData;
import com.aniways.Log;
import com.aniways.R;
import com.aniways.Utils;
import com.aniways.analytics.NonThrowingRunnable;
import com.aniways.blur.BlurLinearLayout;
import com.aniways.blur.ContextualBubbleFrameLayout;
import com.aniways.data.AniwaysLockedIconHelper;
import com.aniways.data.AniwaysLockedIconHelper.OnItemContainingLockedIconClickListener;
import com.aniways.data.AniwaysPrivateConfig;
import com.aniways.data.JsonParser;
import com.aniways.settings.AniwaysSettingsActivity;
import com.aniways.ui.AniwaysUiUtil;
import com.aniways.ui.views.AniwaysSlidingLayer;
import com.aniways.volley.toolbox.IResponseListener;
import com.aniways.volley.toolbox.NetworkImageView;
import com.aniways.volley.toolbox.Volley;

/**
 * QuickAction dialog.
 *
 * @author Lorensius W. L. T <lorenz@londatiga.net>
 *
 * Contributors:
 * - Kevin Peck <kevinwpeck@gmail.com>
 */
public class QuickAction extends PopupWindows implements PopupWindow.OnDismissListener, ViewTreeObserver.OnGlobalLayoutListener {
    private TextView mCreditsBalance;
    private Animation mTrackAnim;
    private LayoutInflater inflater;
    private GridLayout mTrack;
    private OnActionItemClickListener mItemClickListener;
    private OnDismissListener mDismissListener;
    private static HashSet<QuickAction> sOpenedQuickActions = new HashSet<>();
    private ImageView showMore;
    private OnTouchListener mOnTouchListener;
    private List<ActionItem> mActionItemList = new ArrayList<>();
    private List<View> mActionItemViewsList = new ArrayList<>();

    private boolean mAnimateTrack;

    private int mChildPos;
    private int mAnimStyle;

    public static final int ANIM_GROW_FROM_LEFT = 1;
    public static final int ANIM_GROW_FROM_RIGHT = 2;
    public static final int ANIM_GROW_FROM_CENTER = 3;
    public static final int ANIM_AUTO = 4;

    private static final String TAG = "AniwaysQuickAction";

    private AniwaysLockedIconHelper mLockedIconHelper;
    private OnItemContainingLockedIconClickListener mOnItemContainingLockedIconClickListener;
    private View mAnchor;
    private Point mArrowCenterOffsetFromAnchorTopLeft, mArrowCenterOffsetFromAnchorBottomLeft;
    private TextView mIconText;
    private boolean isIconInfo;
    private ImageView showMoreIcons;
    private AniwaysSlidingLayer slidingLayer;
    /**
     * Constructor.
     *
     * @param context Context
     */
    public QuickAction(Context context, View anchor, JsonParser parser,
                       boolean showCreditsIfStoreIsEnabled, boolean isIconInfo, boolean delayAutoDismiss) {
        super(context, delayAutoDismiss);

        mAnchor = anchor;
        this.isIconInfo = isIconInfo;
        inflater 	= (LayoutInflater) context.getSystemService(Context.LAYOUT_INFLATER_SERVICE);

        //TODO: create tranlate animation by code, in order to control the x offset by dips
        mTrackAnim 	= AnimationUtils.loadAnimation(context, R.anim.aniways_rail);

        if (isIconInfo) {
            setRootViewId(R.layout.aniways_icon_info_popup);

        } else {
            setRootViewId(R.layout.aniways_contextual_suggestions_popup);

        }

        mOnItemContainingLockedIconClickListener = new OnItemContainingLockedIconClickListener() {
            @Override
            public void onItemClick(Object iconClickContext) {
                QuickActionIconClickContext context = (QuickActionIconClickContext) iconClickContext;
                callItemClickListener(context);
            }
        };

        if (!isIconInfo) {
            mLockedIconHelper = new AniwaysLockedIconHelper(parser,
                    mOnItemContainingLockedIconClickListener, null,
                    mCreditsBalance, mWindow, mAnchor,
                    showCreditsIfStoreIsEnabled, this.mContext, "SP");
            mAnimateTrack = true;
        }

        mAnimStyle = ANIM_AUTO;
        mChildPos = 0;

        super.setOnDismissListener(this);
        anchor.getViewTreeObserver().addOnGlobalLayoutListener(this);
    }

    public static double bezierPoint(double point1, double point2, double point3, double point4, double time) {
        double t1 = 1.0f - time;
        return point1*t1*t1*t1 + 3*point2*time*t1*t1 + 3*point3*time*time*t1 + point4*time*time*time;
    }

    /**
     * Get action item at an index
     *
     * @param index  Index of item (position from callback)
     *
     * @return  Action Item at the position
     */
    public ActionItem getActionItem(int index) {
        return mActionItemList.get(index);
    }

    /**
     * Set root view.
     *
     * @param id Layout resource id
     */
    public void setRootViewId(int id) {
        mRootView = (ViewGroup) inflater.inflate(id, null);
        if (isIconInfo) {
            mIconText = (TextView) mRootView
                    .findViewById(R.id.aniways_quickaction_tracks);
        } else {
            mTrack = (GridLayout) mRootView
                    .findViewById(R.id.aniways_quickaction_tracks);
            mCreditsBalance = (TextView) mRootView
                    .findViewById(R.id.aniways_quickaction_credits_balance);
            if(mCreditsBalance == null){
                mCreditsBalance = new TextView(this.mContext);
            }
        }

        BlurLinearLayout bubbleView = (BlurLinearLayout) mRootView.findViewById(R.id.aniways_contextual_popup_blured_linear_layout);
        bubbleView.setViewBehindPopup(mAnchor);

        //This was previously defined on show() method, moved here to prevent force close that occured
        //when tapping fastly on a view to show quickaction dialog.
        //Thanx to zammbi (github.com/zammbi)
        //mRootView.setLayoutParams(new LayoutParams(LayoutParams.WRAP_CONTENT, LayoutParams.WRAP_CONTENT));

        setContentView(mRootView);
    }

    public void addActionItems(List<ActionItem> actionItemList) {
        boolean isLastItemInRow = false;
        boolean isLastRow = false;
        boolean isSettingItem = false;
        int pullStripHeightInPixels = AniwaysUiUtil.convertDipsToPixels(16);
        int itemLeftRightMarginInPixels = AniwaysUiUtil.convertDipsToPixels(9);
        int itemTopBottomMarginInPixels = AniwaysUiUtil.convertDipsToPixels(9);

        clearActionItems();
        initSlidingLayer();

        if (actionItemList == null || actionItemList.isEmpty()){
            Log.i(TAG,"trying to add action items to track with null or empty list");
            return;
        }

        int total = actionItemList.size() + 1; // plus a setting button
        int maxNumItemsInColumn = calcColumnsNumber(AniwaysPrivateConfig.getInstance().iconInSuggestionPopupWidth + itemLeftRightMarginInPixels, itemLeftRightMarginInPixels);
        int columns = Math.min(total,maxNumItemsInColumn);
        int rows = calcRowNumber(total, columns);

        if (columns <= 0){ //Could not calculate columns - close the popup and return.
            this.clearActionItems();
            this.dismiss();
            return;
        }

        if (rows > 1){
            addPullupStripAsFirstRow();
        }

        adjustSlideLayerHeight(rows, columns, AniwaysPrivateConfig.getInstance().iconInSuggestionPopupHeight + itemTopBottomMarginInPixels, itemTopBottomMarginInPixels, pullStripHeightInPixels);

        mTrack.setColumnCount(columns);
        mTrack.setRowCount(rows);

        for (int i = 0, c = 0, r = 0; i< total; i++, c++){
            if (i == total-1){ //last item - insert te setting image to grid
                isSettingItem = true;
            }

            IconData icon = null;

            if (!isSettingItem){
                icon = actionItemList.get(i).getIconData();
            }
            if (icon == null && !isSettingItem){
                --c;
                continue;
            }

            if (c == columns - 1){
                isLastItemInRow = true;
            }

            if(c == columns)
            {
                c = 0;
                r++;
            }

            if (r == rows - 1){
                isLastRow = true;
            }

            ActionItem action = null;

            if (!isSettingItem){
                action = actionItemList.get(i);
            }

            addActionItemToGridContainer(c,r, action, itemLeftRightMarginInPixels, itemTopBottomMarginInPixels, isLastRow, isLastItemInRow, isSettingItem);

            isLastItemInRow = false;
            isLastRow = false;
        }
    }

    private void initSlidingLayer() {
        if (slidingLayer == null){
            slidingLayer = (AniwaysSlidingLayer) mRootView.findViewById(R.id.sliding_layer);
        }

        slidingLayer.setOnInteractListener(new AniwaysSlidingLayer.OnInteractListener() {
            @Override
            public void onOpen() {
                rotateShowMoreIndicator(R.anim.aniways_chevron_rotate_down,true);
            }

            @Override
            public void onClose() {
                rotateShowMoreIndicator(R.anim.aniways_chevron_rotate_up, false);
            }

            @Override
            public void onOpened() {
            }

            @Override
            public void onClosed() {
            }
        });
    }

    private void rotateShowMoreIndicator(int animationId, boolean setFillAfter) {
        if (showMore == null)
            return;

        Animation animation = AnimationUtils.loadAnimation(mContext, animationId);
        animation.setRepeatCount(1);
        showMore.startAnimation(animation);
        animation.setFillAfter(setFillAfter);
    }

    private void adjustSlideLayerHeight(int rows, int columns, int singleIconHeight, int heightMargin, int pullStripHeightInPixels) { //Todo: ask if we can assume that the icon is square
        int showMoreStripHeight = rows == 1 ? 0 : pullStripHeightInPixels;
        int availableHeight = calcAvailableHeight();
        int fitInRowsNumber = availableHeight / singleIconHeight;

        if (((fitInRowsNumber * singleIconHeight) + showMoreStripHeight) > availableHeight){
            fitInRowsNumber --;
        }

        int maxRows = Math.min(rows, fitInRowsNumber);

        slidingLayer.setOffsetWidth(singleIconHeight + heightMargin + showMoreStripHeight);
        slidingLayer.getLayoutParams().height = ((maxRows * singleIconHeight) + heightMargin + showMoreStripHeight);
    }

    private int calcAvailableHeight() {
        int[] location = new int[2];
        mAnchor.getLocationOnScreen(location);

        Rect rectangle = new Rect();
        mAnchor.getWindowVisibleDisplayFrame(rectangle);
        int statusBarHeight = rectangle.top;

        return location[1] - statusBarHeight;
    }

    private int calcRowNumber(int total, int columns) {
        if(columns == 0)
            return 0;

        int row = (total/columns);
        row+= (total % columns == 0) ? 0: 1;
        return row;
    }

    private void addPullupStripAsFirstRow() {
        showMore = (ImageView) mRootView.findViewById(R.id.show_more_icons);
        showMore.setVisibility(View.VISIBLE);
        mRootView.findViewById(R.id.show_more_icons_up_strip).setVisibility(View.VISIBLE);
        mRootView.findViewById(R.id.show_more_icons_down_strip).setVisibility(View.VISIBLE);
    }

    private void addActionItemToGridContainer(int column, int row, ActionItem action, int itemLeftRightMarginInPixels,
                                              int itemTopBottomMarginInPixels, boolean isLastRow, boolean isLastItemInRow, boolean isSettingItem) {
        if (isSettingItem){
            GridLayout.LayoutParams lp = getItemInGridLayoutParams(column, row, itemLeftRightMarginInPixels, itemTopBottomMarginInPixels, isLastRow, isLastItemInRow);
            addSettingsBtnToTrack(lp);
            return;
        }

        mActionItemList.add(action);

        final View iconContainer = inflater.inflate(R.layout.aniways_contextual_suggestions_popup_emoticon_item, null);

        mActionItemViewsList.add(iconContainer);
        addProgressBar(iconContainer);

        final ImageView lock = (ImageView) iconContainer.findViewById(R.id.aniways_ebp_emoticons_item_locked_icon);
        final NetworkImageView img 	= (NetworkImageView) iconContainer.findViewById(R.id.aniways_action_item_icon);

        LayoutParams lp =img.getLayoutParams();

        lp.width = AniwaysPrivateConfig.getInstance().iconInSuggestionPopupWidth;
        lp.height = AniwaysPrivateConfig.getInstance().iconInSuggestionPopupHeight;
        // Just to re-enforce it again, as it doesn't always stick..
        img.setMinimumWidth(AniwaysPrivateConfig.getInstance().iconInSuggestionPopupWidth);
        img.setMinimumHeight(AniwaysPrivateConfig.getInstance().iconInSuggestionPopupHeight);
        img.setScaleType(ImageView.ScaleType.FIT_CENTER);
        img.setLayoutParams(lp);

        GridLayout.LayoutParams containerLp = getItemInGridLayoutParams(column, row, itemLeftRightMarginInPixels, itemTopBottomMarginInPixels, isLastRow, isLastItemInRow);

        iconContainer.setLayoutParams(containerLp);

        registerImgerResponseListener(img, lock, iconContainer, action);
        addImageToRequestQueue(action, img);
        bindImageContainer(action, iconContainer, lock, img, mChildPos);

        mTrack.addView(iconContainer);

        mChildPos++;
    }

    private GridLayout.LayoutParams getItemInGridLayoutParams(int column, int row, int itemLeftRightMarginInPixels, int itemTopBottomMarginInPixels, boolean isLastRow, boolean isLastItemInRow) {
        GridLayout.LayoutParams containerLp = new GridLayout.LayoutParams();

        containerLp.height = LayoutParams.WRAP_CONTENT;
        containerLp.width = LayoutParams.WRAP_CONTENT;

        containerLp.leftMargin = itemLeftRightMarginInPixels;
        containerLp.topMargin = itemTopBottomMarginInPixels;

        if (isLastItemInRow){
            containerLp.rightMargin = itemLeftRightMarginInPixels;
        }

        if (isLastRow){
            containerLp.bottomMargin = itemTopBottomMarginInPixels;
        }

        containerLp.setGravity(Gravity.CENTER);
        containerLp.columnSpec = GridLayout.spec(column);
        containerLp.rowSpec = GridLayout.spec(row);
        return containerLp;
    }

    private void addImageToRequestQueue(ActionItem action, NetworkImageView img) {
        AniwaysPrivateConfig config = AniwaysPrivateConfig.getInstance();
        IconData icon = action.getIconData();
        img.setImageUrl(config.getIconUrl(icon, false, false), icon, Volley.getImageLoader(), config.getMaxWidthForCache(icon), config.getMaxHeightForCache(icon), true);
    }

    private void registerImgerResponseListener(final NetworkImageView img, final ImageView lockImage, final View imageContainer, final ActionItem action) {
        img.registerResponseListener(new IResponseListener() {

            @Override
            public void onSuccess() {
                try{
                    mLockedIconHelper.setIconAndLockVisibility(img, lockImage, action.getIconData());
                    ProgressBar spinner = (ProgressBar)imageContainer.findViewById(R.id.aniways_progress_bar_id);
                    spinner.setVisibility(View.GONE);
                }
                catch(Throwable t){
                    Log.e(true, TAG, "Exception in IResponseListener onSuccess", t);
                }
            }

            @Override
            public void onError() {
                boolean closePopup = AniwaysPrivateConfig.getInstance().noInternetPopupClose;
                Log.i(TAG, "Error loading image in quick action" + (closePopup ? ". Closing the popup" : ""));

                Handler h = new Handler(Looper.getMainLooper());
                h.postDelayed(new NonThrowingRunnable(TAG, "addActionItem.onError", ""){

                    @Override
                    public void innerRun() {
                        if(QuickAction.this.isShowing()){
                            Log.i(TAG, "Dissmissing window because of error loading image");
                            // Put a toast asking to check internet connection and then close the popup
                            String toastString = "Please make sure you are connected to the Internet";
                            try{
                                toastString = mContext.getResources().getString(R.string.aniways_check_internet_connection);
                            }
                            catch(Throwable ex){
                                Log.w(true, TAG, "Caught Exception while getting make sure connected to intenet toast string", ex);
                            }
                            Toast toast = Toast.makeText(mContext, toastString, Toast.LENGTH_LONG);
                            toast.setGravity(Gravity.CENTER, 0, 0);
                            toast.show();
                            QuickAction.this.dismiss();
                        }
                    }

                }, AniwaysPrivateConfig.getInstance().noInternetPopupCloseDelay);

            }
        });
    }

    private int calcColumnsNumber(int singleIconWidth, int margin) {
        DisplayMetrics dm = new DisplayMetrics();
        mWindowManager.getDefaultDisplay().getMetrics(dm);
        int screenWidth = dm.widthPixels;
        int columnNumber = (screenWidth - margin) / (singleIconWidth);

        return columnNumber; //Normalize - edged margins are not count when calculating column numbers.
    }

    /**
     * Add action item
     *
     * @param action  {@link ActionItem}
     */
    @SuppressLint("InflateParams")
    public void addActionItem(final ActionItem action) {

        mActionItemList.add(action);

        IconData icon 	= action.getIconData();

        final View container = inflater.inflate(R.layout.aniways_contextual_suggestions_popup_emoticon_item, null);
        mActionItemViewsList.add(container);

        addProgressBar(container);

        final ImageView lock = (ImageView) container.findViewById(R.id.aniways_ebp_emoticons_item_locked_icon);
        final NetworkImageView img 	= (NetworkImageView) container.findViewById(R.id.aniways_action_item_icon);

        // Set image height and width according to configuraiton
        LayoutParams lp = img.getLayoutParams();
        lp.width = AniwaysPrivateConfig.getInstance().iconInSuggestionPopupWidth;
        lp.height = AniwaysPrivateConfig.getInstance().iconInSuggestionPopupHeight;
        // Just to re-enforce it again, as it doesn't always stick..
        img.setMinimumWidth(AniwaysPrivateConfig.getInstance().iconInSuggestionPopupWidth);
        img.setMinimumHeight(AniwaysPrivateConfig.getInstance().iconInSuggestionPopupHeight);
        img.setScaleType(ImageView.ScaleType.FIT_CENTER);

        img.setLayoutParams(lp);

        if (icon != null) {
            registerImgerResponseListener(img,lock,container,action);
            addImageToRequestQueue(action,img);
        } else {
            img.setVisibility(View.GONE);
        }

        final int position = mChildPos;

        bindImageContainer(action, container, lock, img, position);

        mTrack.addView(container, mChildPos);

        mChildPos++;
    }

    private void bindImageContainer(final ActionItem action, View container, final ImageView lock, final NetworkImageView img, final int position) {
        container.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(final View clickedOnActionItem) {
                try{
                    mLockedIconHelper.onIconClicked(img, lock, action.getIconData(), new QuickActionIconClickContext(clickedOnActionItem, position, action.getActionId()));
                }
                catch(Throwable ex){
                    Log.e(true, TAG, "Caught Exception in onClick on action item", ex);
                }
            }
        });
    }

    public void clearActionItems()
    {
        mTrack.removeAllViews();
        mChildPos = 0;
        mActionItemViewsList.clear();
        mActionItemList.clear();
    }

    public void addSettingsBtnToTrack(GridLayout.LayoutParams layoutParams){
        if (mTrack == null)
            return; //mTrack must be inflated before adding views to it.

        layoutParams.setGravity(Gravity.CENTER);

        ImageView settingBtn = new ImageView(mContext);
        settingBtn.setImageResource(R.drawable.aniways_settings_btn);
        settingBtn.setLayoutParams(layoutParams);

        //Todo: just pass back the click event - no need for quickAction to be familiar with the settings activity

        settingBtn.setOnClickListener(new OnClickListener() {
            @Override
            public void onClick(View v) {
                Intent intent = new Intent(v.getContext(), AniwaysSettingsActivity.class);
                mContext.startActivity(intent);
            }
        });

        mTrack.addView(settingBtn);
    }

    private void addProgressBar(final View container) {
        View relativeLayout = container.findViewById(R.id.relative_container);
        int progressBarStyle = AniwaysPrivateConfig.getInstance().iconInSuggestionPopupWidth >= AniwaysUiUtil.convertDipsToPixels(76)? android.R.attr.progressBarStyleLarge : android.R.attr.progressBarStyleSmall;
        ProgressBar pb = new ProgressBar(mContext, null, progressBarStyle);
        pb.setId(R.id.aniways_progress_bar_id);
        RelativeLayout.LayoutParams params = new RelativeLayout.LayoutParams(RelativeLayout.LayoutParams.WRAP_CONTENT, RelativeLayout.LayoutParams.WRAP_CONTENT);
        params.addRule(RelativeLayout.CENTER_IN_PARENT, RelativeLayout.TRUE);
        pb.setLayoutParams(params );
        ((ViewGroup) relativeLayout).addView(pb);
    }

    public void addActionIconInfo(final ActionItem action) {
        mIconText.setText(action.getTitle());
    }

    public void setOnActionItemClickListener(OnActionItemClickListener listener) {
        mItemClickListener = listener;
    }

    /**
     * Show popup mWindow
     */
    public void show(Point arrowCenterOffsetFromAnchorTopLeft, Point arrowCenterOffsetFromAnchorBottomLeft, boolean fillParentWidth) {

        sOpenedQuickActions.add(this);

        preShow(fillParentWidth);

        setAnimationStyle();
        adjustLayout(arrowCenterOffsetFromAnchorTopLeft, arrowCenterOffsetFromAnchorBottomLeft);

        if (mAnimateTrack)
        {
            int offsetBase = this.mContext.getResources().getInteger(R.integer.aniwaysPopupAimationTime) - 50;
            int fadeInOffset = this.mContext.getResources().getInteger(R.integer.aniwaysContextualPopupFadeInAimationOffset);
            mTrackAnim.setStartOffset(offsetBase);
            if (!isIconInfo) {
                mTrack.startAnimation(mTrackAnim);

                int numChildren = mTrack.getChildCount();
                for (int i = 0; i < numChildren; i++) {
                    Animation fadeInAnimation = AnimationUtils.loadAnimation(
                            mContext, R.anim.aniways_contextual_fade_in);
                    fadeInAnimation.setStartOffset(offsetBase
                            + (i * fadeInOffset));
                    mTrack.getChildAt(i).startAnimation(fadeInAnimation);
                }
            }
        }
    }


    public void showAndCloseAfterDelay(Point centerPoint, Point centerPoint1, boolean fillParentWidth) {
        show(centerPoint, centerPoint1, fillParentWidth);
        mRootView.postDelayed(new Runnable() {
            @Override
            public void run() {
                dismiss();
            }
        }, AniwaysPrivateConfig.getInstance().popupDismissDelay);

    }

    @SuppressLint("NewApi")
    @Override
    public void onGlobalLayout()
    {
        if (mDismissed)
        {
            if (Utils.isAndroidVersionAtLeast(16))
            {
                mAnchor.getViewTreeObserver().removeOnGlobalLayoutListener(this);
            }
        }
        else
        {
            if (mArrowCenterOffsetFromAnchorTopLeft !=null && mArrowCenterOffsetFromAnchorBottomLeft != null){
                adjustLayout(mArrowCenterOffsetFromAnchorTopLeft, mArrowCenterOffsetFromAnchorBottomLeft);
            }
        }
    }

    private void adjustLayout(Point arrowCenterOffsetFromAnchorTopLeft, Point arrowCenterOffsetFromAnchorBottomLeft)
    {
        int[] location = new int[2];

        mArrowCenterOffsetFromAnchorTopLeft = arrowCenterOffsetFromAnchorTopLeft;
        mArrowCenterOffsetFromAnchorBottomLeft = arrowCenterOffsetFromAnchorBottomLeft;
        mAnchor.getLocationOnScreen(location);

        Rect anchorRect = new Rect(location[0], location[1], location[0] + mAnchor.getWidth(), location[1] + mAnchor.getHeight());

        mRootView.measure(MeasureSpec.UNSPECIFIED, MeasureSpec.UNSPECIFIED);

        int rootWidthWithMode 		= mRootView.getMeasuredWidth();
        int rootHeightWithMode 		= mRootView.getMeasuredHeight();

        int rootWidth = MeasureSpec.getSize(rootWidthWithMode);
        int rootHeight = MeasureSpec.getSize(rootHeightWithMode);

        DisplayMetrics dm = new DisplayMetrics();
        mWindowManager.getDefaultDisplay().getMetrics(dm);
        int screenWidth = dm.widthPixels;

        int xPos = 0;
        int xPosArrowCenter = anchorRect.left + mArrowCenterOffsetFromAnchorTopLeft.x;

        if( rootWidth > screenWidth){
            xPos = 0;
        }
        else{
            int halfWidth = rootWidth / 2;

            // if arrow is before the center of the popup window then attach popup to the left of the screen
            if(halfWidth > xPosArrowCenter){
                xPos = 0;
            }
            // if the arrow cannot be positioned in the center of the popup because then the popup would spill offscreen then attach the
            // popup to the right of the screen
            else if(xPosArrowCenter + halfWidth > screenWidth){
                xPos = screenWidth - rootWidth;
            }
            // position the popup so that the arrow would be at its center
            else{
                xPos = xPosArrowCenter - halfWidth;
            }
        }


        int xPosOffsetFromAnchor = xPos - anchorRect.left;
        if (!isIconInfo){
            xPosOffsetFromAnchor = 0 - anchorRect.left;
        }


        boolean onTop = true;

        Rect rectgle = new Rect();
        mAnchor.getWindowVisibleDisplayFrame(rectgle);
        int statusBarHeight = rectgle.top;

        int rootTop = anchorRect.top - rootHeight + mArrowCenterOffsetFromAnchorTopLeft.y - statusBarHeight;
        // display on bottom
        if (rootTop < 0){
            onTop	= false;
        }

        //TODO: What if cant fit in top and bottom??

        int arrowCenterInRoot = xPosArrowCenter - xPos;

        View bubbleLayout = mRootView.findViewById(R.id.aniways_contextual_popup_frame_layout);

        if (onTop)
        {
            if(bubbleLayout != null && bubbleLayout instanceof ContextualBubbleFrameLayout){
                ((ContextualBubbleFrameLayout)bubbleLayout).setArrowDown(arrowCenterInRoot);
            }
            // TODO: the convertDipsToPixels is because the EditText height is not reported correctly for some reason..
            int heightOffset = AniwaysUiUtil.convertDipsToPixels(2) - rootHeight - anchorRect.height();
            try{
                if (mWindow.isShowing())
                {
                    mWindow.update(mAnchor, xPosOffsetFromAnchor, heightOffset, mWindow.getWidth(), mWindow.getHeight());
                }
                else
                {
                    mWindow.showAsDropDown(mAnchor, xPosOffsetFromAnchor, heightOffset);
                }
            }
            catch(Throwable ex){
                Log.e(true, TAG, "Caught error while opening popup", ex);
                return;
            }
        }
        else{
            if(bubbleLayout != null && bubbleLayout instanceof ContextualBubbleFrameLayout){
                ((ContextualBubbleFrameLayout)bubbleLayout).setArrowUp(arrowCenterInRoot);
            }
            try{
                if (mWindow.isShowing())
                {
                    mWindow.update(mAnchor, xPosOffsetFromAnchor, -mArrowCenterOffsetFromAnchorBottomLeft.y, mWindow.getWidth(), mWindow.getHeight());
                }
                else
                {
                    mWindow.showAsDropDown(mAnchor, xPosOffsetFromAnchor, -mArrowCenterOffsetFromAnchorBottomLeft.y);
                }
            }
            catch(Throwable ex){
                Log.e(true, TAG, "Caught error while opening popup", ex);
                return;
            }
        }
    }

    private void setAnimationStyle()
    {
        //mWindow.setAnimationStyle(R.style.AniwaysAnimations_PopUpMenu_Bottom);
        mWindow.setAnimationStyle(0);
    }

    /**
     * Set listener for window dismissed. This listener will only be fired if the quicakction dialog is dismissed
     * by clicking outside the dialog or clicking on sticky item.
     */
    public void setOnDismissListener(OnDismissListener listener) {
        mDismissListener = listener;
    }

    @Override
    public void onDismiss()
    {
        mDismissed = true;

        Log.d(TAG, "removing quick action from hashset of opened quick actions");
        sOpenedQuickActions.remove(this);

        if (mDismissListener != null) {
            mDismissListener.onDismiss();
        }
    }

    private void callItemClickListener(QuickActionIconClickContext context) {
        if (mItemClickListener != null) {

            mItemClickListener.onItemClick(QuickAction.this, context.pos, context.actionId);
        }

        if (!getActionItem(context.pos).isSticky()) {
            // Workaround for transparent background bug,
            // thx to Roman Wozniak <roman.wozniak@gmail.com>
            context.clickedOnActionItem.post(new NonThrowingRunnable(TAG, "after calling onItemClick", "") {
                @Override
                public void innerRun() {
                    dismiss();
                }
            });
        }
    }

    /*
     * Closes all Aniways popups if any are showing.
     * Returns true if it actually closed something and false if not.
     */
    public static boolean dismissAllOpenQuickActions(){
        // Because view detached exception might be thrown
        boolean closedSomething = false;
        try{
            Log.d(TAG, "dismissing all quick actions");

            Log.d(TAG, "Closing opened quickactions");
            // Need this secondary since the dismiss will also remove from the collection on which we iterate
            HashSet<QuickAction> secondary = new HashSet<QuickAction>(sOpenedQuickActions);
            for(QuickAction qa: secondary){
                if (qa.isShowing()){
                    qa.dismiss();
                    closedSomething = true;
                }
                else{
                    Log.d(TAG, "An opened quickaction is not showing");
                }
            }
        }
        catch (Exception e){
            Log.w(true, TAG, "caught Exception while dismissing all opened quick actions", e);
        }
        return closedSomething;
    }

    public static void cancelAutoDismissOfAllOpenQuickActions()
    {
        for (QuickAction quickAction : sOpenedQuickActions)
        {
            quickAction.cancelDismissal();
        }
    }

    /**
     * Listener for item click
     *
     */
    public interface OnActionItemClickListener {
        public abstract void onItemClick(QuickAction source, int pos, int actionId);
    }

    /**
     * Listener for window dismiss
     *
     */
    public interface OnDismissListener {
        public abstract void onDismiss();
    }

    public void setTouchInterceptor(OnTouchListener onTouchListener) {
        mOnTouchListener = onTouchListener;
    }

    @Override
    protected boolean processPopupTouchEvent(View v, MotionEvent event)
    {
        boolean retConsumed = super.processPopupTouchEvent(v, event);

        if (mOnTouchListener != null)
        {
            retConsumed |= mOnTouchListener.onTouch(v, event);
        }

        return retConsumed;
    }
}