package com.aniways.data;

import android.support.annotation.NonNull;

import com.aniways.Log;
import com.aniways.service.utils.AniwaysAction;
import com.aniways.volley.Request;
import com.aniways.volley.Response;
import com.aniways.volley.VolleyError;
import com.aniways.volley.toolbox.JsonObjectRequest;
import com.aniways.volley.toolbox.Volley;

import org.json.JSONObject;

import java.util.LinkedList;
import java.util.List;

/**
 * Holds the shared functionality for all web/json-based asset info builders.
 */
public abstract class WebAssetInfoBuilderBase implements IAssetInfoBuilder {

    private final String builderName;
    private List<IAniwaysAssetInfo> assetInfoItems;
    private int numberOfRequestToPerform;
    private int numberOfRequestDone;
    private AniwaysAction<List<IAniwaysAssetInfo>> callback;

    /**
     * When implemented by derived classes, indicates the name of the builder; this is used for logging purposes,
     * as well as for comparing a providerName ith the builder's name.
     *
     * @param name The provider's name - used for logging and matching in the canBuild() implementation.
     */
    protected WebAssetInfoBuilderBase(String name) {
        this.builderName = name;
    }

    @Override
    public boolean canBuild(String providerName) {
        return builderName.equalsIgnoreCase(providerName);
    }

    @Override
    public void buildAssets(String[] resourcesIds, AniwaysAction<List<IAniwaysAssetInfo>> callback) {
        try {
            this.callback = callback;

            if (callback == null) { return; }

            if (resourcesIds == null || resourcesIds.length == 0) {
                callback.call(null);
                return;
            }

            this.assetInfoItems = new LinkedList<>();
            this.numberOfRequestToPerform = resourcesIds.length;
            this.numberOfRequestDone = 0;

            for (String resource : resourcesIds) {
                if (resource.startsWith(AniwaysPrivateConfig.getInstance().providerSearchTermPrefix)) {
                    performSearchRequest(resource, false);
                }
                else {
                    performSearchRequest(resource, true);
                }
            }
        }
        catch (Throwable ex) {
            Log.e(true, builderName, "Caught Exception while building", ex);
        }
    }

    @Override
    public void cancelPendingRequests() {
        Volley.cancelPendingRequests(builderName);
    }

    private void performSearchRequest(final String resource, final boolean searchedById) {

        final String requestUrl;

        try {
            if (searchedById) {
                requestUrl = getSearchUrlByResourceId(resource);
            }
            else {
                requestUrl = getSearchUrlBySearchTerm(resource);
            }
        }
        catch (Exception e) {
            Log.e(true, builderName, "Unable to construct search request url for resource:" + resource, e);
            signalRequestCompleted();
            return;
        }

        try {
            JsonObjectRequest jsonObjectRequest = new JsonObjectRequest(Request.Method.GET, requestUrl, null, new Response.Listener<JSONObject>() {

                @Override
                public void onResponse(JSONObject response) {
                    processResponse(response, searchedById);
                }
            }, new Response.ErrorListener() {

                @Override
                public void onErrorResponse(VolleyError error) {
                    Log.e(true, builderName, "Search failed for the following url:" + requestUrl, error);
                    signalRequestCompleted();
                }
            });

            Volley.addToRequestQueue(jsonObjectRequest, builderName);
        }
        catch (Exception e) {
            Log.e(true, builderName, "Exception thrown while trying to search Giphy api using the following url:" + requestUrl, e);
            callback.call(null);
        }
    }

    private synchronized void processResponse(JSONObject response, boolean searchedById) {
        if (response != null) {
            if (searchedById) {
                IAniwaysAssetInfo item = parseResourceIdSearchResponse(response);

                if (item != null) {
                    assetInfoItems.add(item);
                }
            }
            else {
                List<IAniwaysAssetInfo> items = parseTermSearchResponse(response);

                if (items != null && items.size() > 0) {
                    assetInfoItems.addAll(items);
                }
            }
        }

        signalRequestCompleted();
    }

    private void signalRequestCompleted() {
        numberOfRequestDone++;

        // No pending requests, invoke the callback.
        if (numberOfRequestDone == numberOfRequestToPerform && callback != null) {
            this.callback.call(assetInfoItems);
        }
    }

    protected abstract String getSearchUrlByResourceId(String resourceId);

    protected abstract String getSearchUrlBySearchTerm(String resource);

    protected abstract IAniwaysAssetInfo parseResourceIdSearchResponse(@NonNull JSONObject response);

    protected abstract List<IAniwaysAssetInfo> parseTermSearchResponse(@NonNull JSONObject response);
}