/**
 * 
 */
package com.aniways.data;

import com.aniways.Log;
import com.aniways.Utils;
import com.aniways.VersionComparisonResult;
import com.aniways.analytics.AnalyticsReporter;
import com.aniways.analytics.GoogleAnalyticsReporter;
import com.aniways.analytics.NonThrowingRunnable;
import com.aniways.data.AniwaysConfiguration.Verbosity;
import com.aniways.data.JsonParser.InputStreamProvider;
import com.aniways.service.utils.AniwaysServiceUtils;

import android.content.Context;
import android.content.SharedPreferences;
import android.content.SharedPreferences.Editor;
import android.os.Handler;
import android.os.Looper;
import java.util.ArrayList;
import java.util.List;

/**
 * @author Shai
 *
 */
public class AniwaysPhraseReplacementData {
	private static JsonParser sParser;
	private static boolean sInitCalled = false;
	private static final String TAG = "AniwaysPhraseReplacementData";
	public static final String EMPTY_PARSER_VERSION = "0.0";
	private static Context sContext;
	private static List<IOnNonEmptyParserSetListener> sOnNonEmptyParserSetListeners; 

	static void forceInit(final Context context)
	{
		if(sParser != null){
			Log.w(true, TAG, "Calling init of AniwaysWordReplacementData more than once");
		}

		sInitCalled = true;
		sContext = context;

		// Set a new empty parser, just until the shared perfs listener thread prompts a reall parsing.
		// This is done for performance - in order not to do the expensive parsing here..
		sOnNonEmptyParserSetListeners = new ArrayList<>();
		sParser = new JsonParser();
	}

	/**
	 * If the data parsing is successful, need to call setNewParser() later.
	 * This is split into 2 functions because it can be used in a different thread, and the setting
	 * of the parser needs to happen on the UI thread
	 */
	static JsonParser parseData(InputStreamProvider inputStreamProvider){
		if(!sInitCalled){
			Log.e(true, TAG, "ParseData called b4 init");
		}

		Log.i(TAG, "Parsing Keywords Json");

		try{
			JsonParser parser = new JsonParser();
			parser.parseDataFile(inputStreamProvider);
			return parser;
		}
		catch(Throwable ex){
			Log.e(true, TAG, "Error parsing json", ex);
			return JsonParser.emptyInstance;
		}
	}

	static synchronized void setNewParser(JsonParser parser, String expectedVersion){
		if(!sInitCalled){
			Log.e(true, TAG, "setNewParser called b4 init");
		}

		Log.i(TAG, "Setting new parser");

		// If we have an empty parser (for whatever reason)
		// then we continue to use the old parser that we have
		// and we set the version on it to be the expected version 
		// of the new parser, in order not to make the app try to parse over and over again
		// because of the same error..
		if(parser.isEmpty()){
			if(!Utils.isVersionLegal(expectedVersion)){
				Log.e(true, TAG, "Received illegel expected version, so not setting new parser: " + expectedVersion);
				
				if(sParser.isEmpty()){
					// If the parser we have now is empty then put it to 0.1 to not trigger more parses, unless getting something
					// with a version..
					sParser.setKeywordsVersion("0.1");
				}
				
				return;
			}
			
			Log.w(true, TAG, "Got an empty new parser, so sticking with the old one, with version: " + sParser.getKeywordsVersion() + ". Setting its version to the expected version of the new one in order not to try and parse over and over again: " + expectedVersion);
			parser = sParser;
			parser.setKeywordsVersion(expectedVersion);
		}
		
		sParser = parser;

		String oldKeywordsVersion = getParsedKeywordsVersion();
		setParsedKeywordsVersion(parser.getKeywordsVersion());

		VersionComparisonResult keywordsResult = null;
		try{
			keywordsResult = Utils.compareVersionStrings(parser.getKeywordsVersion(), oldKeywordsVersion);
		}
		catch(IllegalArgumentException e){
			Log.e(true, TAG, "Error parsing keywords version: " + oldKeywordsVersion, e);
		}

		if (keywordsResult != null){
			if(keywordsResult.result == 1){
				if(AnalyticsReporter.isInitialized()){
					GoogleAnalyticsReporter.reportEvent(Verbosity.Info, "Statistics", "Updated Keywords Version", "from: " + oldKeywordsVersion + ", to: " + parser.getKeywordsVersion(), 0);
					AnalyticsReporter.reportUpdatedKeywordsVersion(oldKeywordsVersion, parser.getKeywordsVersion());
				}
			}
			else if (keywordsResult.result == 0){
				Log.i(TAG, "Setting new keywords parser to version which has already been parsed when the app was running at a previous time: " + parser.getKeywordsVersion() );
			}
			else{
				Log.e(true, TAG, "Setting parser for lower version than was parsed before. Old version: " + oldKeywordsVersion + " . New version:" + parser.getKeywordsVersion());
			}
		}

		if(AnalyticsReporter.isInitialized()){
			GoogleAnalyticsReporter.reportCustomDimention(4, parser.getKeywordsVersion());
		}
		
		if(!sParser.isEmpty()){
			Log.d(TAG, "Setting non empty parser and calling all the listeners on the UI thread: " + sOnNonEmptyParserSetListeners.size());
			Handler h = new Handler(Looper.getMainLooper());
			
			int i = 0;
			for(final IOnNonEmptyParserSetListener listener : sOnNonEmptyParserSetListeners){
				i++;
				final int counter = i; 
				h.post(new NonThrowingRunnable(TAG, "Calling on set non empty parser listener", ""){

					@Override
					public void innerRun() {
						Log.d(TAG, "Start onNonEmptyParserSet listener number: " + counter);
						listener.onNonEmptyParserSet(sParser);
						Log.d(TAG, "End onNonEmptyParserSet listener number: " + counter);
					}
				});
			}
			
			sOnNonEmptyParserSetListeners.clear();
		}
		else{
			Log.w(false, TAG, "Setting an empty parser");
		}
	}
	
	public synchronized static void addOnNonEmptyParserSetListener(IOnNonEmptyParserSetListener listener){
		sOnNonEmptyParserSetListeners.add(listener);
	}

	// All users of this class need to get a parser object and make a 'transaction' with the same one, to ensure
	// data integrity between calls to the parser
	public static JsonParser getDataParser(){
		return sParser;
	}

	// TODO: put these in a different shared perfs file and put the keys and all in a different class cause they do not belong to the service, this is entirely in the app itself..
	private static synchronized String getParsedKeywordsVersion() {
		return getParsedKeywordsVersion(sContext);

	}

	public static synchronized String getParsedKeywordsVersion(Context context) {
		// open the shared preferences
		SharedPreferences prefs = context.getSharedPreferences(AniwaysServiceUtils.SHARED_PREFERENCES, Utils.getSharedPreferencesFlags());

		// return the values that stored there, in case that no value
		// is stored return false
		return prefs.getString(AniwaysServiceUtils.KEY_PARSED_KEYWORDS_VERSION, AniwaysPhraseReplacementData.EMPTY_PARSER_VERSION);

	}

	/**
	 * Update the APP_ID value in the shared preferences with the given value
	 */
	public static synchronized void setParsedKeywordsVersion(String version) {
		// open the shared preferences
		SharedPreferences prefs = 
				sContext.getSharedPreferences(AniwaysServiceUtils.SHARED_PREFERENCES, Utils.getSharedPreferencesFlags());
		Editor edit = prefs.edit();
		// write the new value
		edit.putString(AniwaysServiceUtils.KEY_PARSED_KEYWORDS_VERSION, version);
		edit.commit();
	}

	public static boolean isInit(){
		return sParser != null;
	}
	
	public interface IOnNonEmptyParserSetListener{
		void onNonEmptyParserSet(JsonParser newParser);
	}
}


