package com.aniways.contextual;

import android.content.Context;
import android.content.res.Resources;
import android.graphics.Rect;
import android.support.v4.view.ViewPager;
import android.util.DisplayMetrics;
import android.util.TypedValue;
import android.view.View;
import android.view.ViewTreeObserver;
import android.view.WindowManager;
import android.widget.ImageView;
import android.widget.TextView;

import com.aniways.Log;
import com.aniways.R;
import com.aniways.data.AniwaysLockedIconHelper;
import com.aniways.nineoldandroids.view.ViewHelper;
import com.aniways.quick.action.ContextualItemSelectedByUserEvent;
import com.aniways.quick.action.ContextualPopupCreationContext;
import com.aniways.quick.action.SelectedGridItemInfo;
import com.aniways.service.utils.AniwaysAction;
import com.aniways.ui.views.AniwaysPopupWindow;
import com.aniways.ui.views.ContextualAssetView;
import com.aniways.ui.views.ContextualSettingsView;

/**
 * Presents contextual content to the user.
 */
public class ContextualPopup extends AniwaysPopupWindow implements ViewTreeObserver.OnGlobalLayoutListener
{
    // Defines the number of pages the pager is expected to load up front.
    private static final int MaxPagesToLoad = 10;
    private static final String TAG = "ContextualPopup";

    // Holds the WindowManager associated with this instance.
    protected final WindowManager windowManager;

    private final ContextualPopupAdapter assetsAdapter;

    private final int cardAtHintHeight;
    private AniwaysLockedIconHelper mLockedIconHelper;

    /**
     * Presents the contextual content to the user in a popup fashion.
     *
     * @param context The visual context for this popup.
     */
    public ContextualPopup(Context context, ContextualPopupCreationContext creationContext)
    {
        super(context, creationContext, true, new PopupTimingOptions());

        View popupContent = layoutInflater.inflate(R.layout.aniways_contextual_popup, null);
        setContentView(popupContent);

        ViewPager pager = (ViewPager)popupContent.findViewById(R.id.aniways_contextual_popup);
        this.assetsAdapter = new ContextualPopupAdapter(context, layoutInflater, pager, creationContext);
        pager.setAdapter(this.assetsAdapter);
        pager.setOffscreenPageLimit(MaxPagesToLoad);
        pager.setOnPageChangeListener(this.assetsAdapter);
        pager.setPageTransformer(true, new StackedPagesPageTransformer(this.assetsAdapter.getCount(), context));

        this.windowManager = (WindowManager)context.getSystemService(Context.WINDOW_SERVICE);

        Resources res = context.getResources();
        this.cardAtHintHeight = (int)(res.getDimension(R.dimen.aniways_contextual_card_min_height) + res.getDimension(R.dimen.aniways_contextual_card_hint_height));

        initLockedIconHelper();
    }

    @Override
    public void show(View anchor)
    {
        super.show(anchor);
        this.assetsAdapter.showFirstPageHintOrWink();
    }

    private void initLockedIconHelper()
    {
        try
        {
            mLockedIconHelper = new AniwaysLockedIconHelper(
                    creationContext.parser,
                    new AniwaysLockedIconHelper.OnItemContainingLockedIconClickListener()
                    {
                        @Override
                        public void onItemClick(Object iconClickContext)
                        {
                        }
                    },
                    null,
                    new TextView(context),
                    this,
                    creationContext.anchor
                    , true, context, "SP");
        }
        catch (Exception e)
        {
            Log.e(true, TAG, "Exception thrown while trying to init AniwaysLockedIconHelper in Contextual popup", e);
        }
    }


    public void handleLockedIconSelectedEvent(ContextualItemSelectedByUserEvent itemSelectedByUserEvent, final AniwaysAction<Boolean> callback)
    {
        try
        {
            SelectedGridItemInfo selectedGridItemInfo = itemSelectedByUserEvent.selectedGridItemInfo;
            View gridItem = selectedGridItemInfo.getGridItem();
            ImageView iconImage = (ImageView)gridItem.findViewById(R.id.aniways_action_item_icon);
            ImageView lockedImage = (ImageView)gridItem.findViewById(R.id.aniways_ebp_emoticons_item_locked_icon);

            if (lockedImage == null)
            {
                callback.call(false);
                return;
            }

            mLockedIconHelper.onIconClicked(iconImage, lockedImage, itemSelectedByUserEvent.assetInfo.getIconData(), itemSelectedByUserEvent, new AniwaysLockedIconHelper.OnItemContainingLockedIconClickListener()
            {
                @Override
                public void onItemClick(Object iconClickContext)
                {
                    callback.call(true);
                }
            });
        }
        catch (Exception e)
        {
            Log.e(true, TAG, "Exception thrown while trying to call handleLockedIconSelectedEvent.onIconSelected in Contextual popup", e);
            callback.call(false);
        }
    }

    /**
     * Calculates the desired bounds (size, position) of the popup.
     *
     * @return A rectangle at the size and position that  in.
     */
    protected Rect calculateTargetBounds(View anchor)
    {
        // We are looking for a place above the anchor and up to the action bar (or status bar,
        // in case there's no action bar).

        int[] location = new int[2];
        anchor.getLocationOnScreen(location);

        Rect rectangle = new Rect();
        anchor.getWindowVisibleDisplayFrame(rectangle);
        int statusBarHeight = rectangle.top;

        DisplayMetrics dm = new DisplayMetrics();
        windowManager.getDefaultDisplay().getMetrics(dm);
        int screenWidth = dm.widthPixels;

        // Calculate ActionBar height
        int actionBarHeight = 0;
        TypedValue tv = new TypedValue();
        if (this.context.getTheme().resolveAttribute(android.R.attr.actionBarSize, tv, true))
        {
            actionBarHeight = TypedValue.complexToDimensionPixelSize(tv.data, this.context.getResources().getDisplayMetrics());
        }

        Rect retBounds = new Rect(0, statusBarHeight + actionBarHeight, screenWidth, location[1]);

        this.assetsAdapter.setToastOffset(retBounds.bottom - this.cardAtHintHeight - statusBarHeight);

        return retBounds;
    }

    /**
     * Positions the popup pager's pages one on top of the other, allowing only a swipe to the left,
     * so that there's a sense of revealing cards that are laid one on top of the other.
     */
    private class StackedPagesPageTransformer implements ViewPager.PageTransformer
    {
        private final int pages;
        private final float settingsCardWidth, settingsCardHookWidth, cardIconWidth, revealWidth;

        public StackedPagesPageTransformer(int pages, Context context)
        {
            this.pages = pages;

            this.settingsCardWidth = context.getResources().getDimension(R.dimen.aniways_contextual_card_setting_peek_width);
            this.settingsCardHookWidth = context.getResources().getDimension(R.dimen.aniways_contextual_card_setting_visible_width);
            this.cardIconWidth = context.getResources().getDimension(R.dimen.aniways_contextual_card_icon_holder_width);
            this.revealWidth = context.getResources().getDimension(R.dimen.aniways_contextual_card_reveal_width);
        }

        public void transformPage(View view, float position)
        {
            int pageWidth = view.getWidth();

            if (view instanceof ContextualAssetView)
            {
                ContextualAssetView assetView = ((ContextualAssetView)view);

                // Let the card view itself update its visuals according to its position.
                assetView.updateByPosition(position);
                updateAssetPagePosition(assetView, position, pageWidth);
            }
            else if (view instanceof ContextualSettingsView)
            {
                ContextualSettingsView settingsView = (ContextualSettingsView)view;

                // Let the card view itself update its visuals according to its position.
                settingsView.updateByPosition(position, pageWidth);
                updateSettingsPagePosition(settingsView, position, pageWidth);
            }
        }

        private void updateAssetPagePosition(ContextualAssetView assetView, float position, int pageWidth)
        {
            if (position >= 0)
            {
                // This page is partially or fully swiped to the right. Keep it aligned to the left edge of the screen.
                ViewHelper.setTranslationX(assetView, -pageWidth * position);
            }
            // The last assets card before the settings card.
            else if (assetView.index == this.pages - 2)
            {
                // Last assets page does not slide all the way to the left.
                float pinningPosition = (this.settingsCardHookWidth - this.revealWidth) / pageWidth;

                if (position <= -pinningPosition)
                {
                    // Pin the last assets page to seem like it's being pinned by the settings page.
                    ViewHelper.setTranslationX(assetView, -pageWidth * (pinningPosition + position));
                }
            }
        }

        private void updateSettingsPagePosition(ContextualSettingsView settingsView, float position, int pageWidth)
        {
            float relativeIconWidth = this.cardIconWidth / pageWidth;

            if (position >= 1.0)
            {
                // Keep the settings page peeking at the right edge of the screen.
                ViewHelper.setTranslationX(settingsView, pageWidth * (1 - position - relativeIconWidth));
            }
            else
            {
                float pinningPosition = 1 - ((this.settingsCardWidth - this.revealWidth) / pageWidth);

                if (position >= pinningPosition)
                {
                    float relativeRevealWidth = this.revealWidth / pageWidth;
                    float fullOverlap = relativeIconWidth - relativeRevealWidth;
                    float progress = 1 - ((position - pinningPosition) / (1 - pinningPosition));

                    // Make sure the settings card stays overlapping behind the right edge of the card above it.
                    ViewHelper.setTranslationX(settingsView, pageWidth * (-relativeIconWidth + (progress * fullOverlap)));
                }
                else
                {
                    float relativeCardWidth = this.settingsCardWidth / pageWidth;

                    // Pin the settings page so that it doesn't leave the right edge of the screen.
                    ViewHelper.setTranslationX(settingsView, pageWidth * (1 - relativeCardWidth - position));
                }
            }
        }
    }
}