package com.aniways.analytics.request;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.ProtocolException;
import java.net.URL;
import java.security.SecureRandom;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.List;
import java.util.Locale;
import java.util.Random;
import java.util.TimeZone;
import java.util.Map.Entry;
import java.util.zip.GZIPOutputStream;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSession;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;

import android.annotation.SuppressLint;
import android.content.Context;
import android.text.TextUtils;

import com.aniways.Log;
import com.aniways.Utils;
import com.aniways.analytics.info.Build;
import com.aniways.analytics.request.azure.AzureRequest;
import com.aniways.data.AniwaysPrivateConfig;
import com.aniways.data.AniwaysPrivateConfig.AnalyticsCloud;
import com.aniways.data.Installation;
import com.aniways.service.utils.AniwaysServiceUtils;

import org.apache.http.Header;
import org.apache.http.HttpResponse;
import org.apache.http.ParseException;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.params.HttpClientParams;
import org.apache.http.conn.params.ConnManagerParams;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.message.BasicHeader;
import org.apache.http.params.HttpConnectionParams;
import org.apache.http.protocol.HTTP;
import org.apache.http.util.EntityUtils;

public class BasicRequester {

	private static final String TAG = "AniwaysAnalyticsBasicRequester";
	private Context mContext;

	public BasicRequester(Context context){
		mContext = context;
	}

	public boolean sendToErrorHandler(String content) {
		HttpResponse response = null;
		String url = AniwaysPrivateConfig.getInstance().errorHandlingEndpoint;
		try {
			// Create client
			HttpClient httpClient = new DefaultHttpClient(); 

			// Set its properties
			ConnManagerParams.setTimeout(httpClient.getParams(), AniwaysServiceUtils.CONNECTION_MANAGER_TIMEOUT);
			HttpConnectionParams.setSoTimeout(httpClient.getParams(), AniwaysServiceUtils.READ_TIMEOUT); 
			HttpConnectionParams.setConnectionTimeout(httpClient.getParams(), AniwaysServiceUtils.CONNECTION_TIMEOUT); 
			HttpClientParams.setRedirecting(httpClient.getParams(), true);
			System.setProperty("http.keepAlive", "false");

			// Create the Post request
			HttpPost post = new HttpPost(url);
			ByteArrayEntity se = new ByteArrayEntity(content.getBytes("UTF-8")); 
			post.setHeader("Content-Type", "application/json; charset=utf-8");
			se.setContentEncoding(new BasicHeader(HTTP.CONTENT_TYPE, "application/json; charset=utf-8"));

			// Perform the request
			post.setEntity(se);
			response = httpClient.execute(post);
		}
		catch (MalformedURLException e) {
			Log.eToGaOnly(true, TAG, "Malformed URL: " + url, e);
		} 
		catch (ProtocolException e) {
			Log.eToGaOnly(true, TAG, "Protocol Exception for url: " + url, e);
		}
		catch (IOException e) {
			String networkInterface = Utils.isInternetAvailable(mContext);
			if (networkInterface != null){
				Log.wToGaOnly(true, TAG, "IO Exception on URL " + url + " Network interface: " + networkInterface, e);
			}
			else{
				Log.wToGaOnly(false,TAG, "IO Exception, but there is no internet connection, so that's expected. URL: " + url + " Retry number: ", e);
			}
		} catch (IllegalArgumentException e) {
			Log.eToGaOnly(true, TAG, "Illegal arg Exception on URL: " + url, e);
		}
		catch(Throwable e){
			Log.wToGaOnly(true, TAG, "Failed to send request on URL: " + url, e);
		}

		if (response == null) {
			// there's been an error
			Log.wToGaOnly(false, TAG, "Failed to make request to the server. Url: " + url, null);
			return false;
		} else if (response.getStatusLine().getStatusCode() != 200) {
			String responseString = "";
			try {
				// there's been a server error
				responseString += response.getStatusLine().getReasonPhrase() + "\n";
				Header[] headers = response.getAllHeaders();
				if(headers != null){
					for(Header header : headers){
						responseString += header.getName();
						responseString += " : ";
						responseString += header.getValue() + "\n";
					}
				}

				responseString += EntityUtils.toString(response.getEntity());

				Log.eToGaOnly(true, TAG, "Received a failed response from the server. Url: " + url + ".Response code: " + response.getStatusLine().getStatusCode() + ". Response: " + responseString, null);

			} catch (ParseException e) {
				Log.wToGaOnly(true, TAG, "Failed to parse the response from the server. Url: " + url + ". Response: " + responseString, e);
			} catch (IOException e) {
				String networkInterface = Utils.isInternetAvailable(mContext);
				if (networkInterface != null){
					Log.wToGaOnly(true, TAG, "Failed to read the response from the server. Url: " + url + ". Response: " + responseString, e);
				}
				else{
					Log.wToGaOnly(false,TAG, "Failed to read the response from the server, but there is no internet connection, so that's expected. URL: " + url + " Retry number: ", e);
				}
			}
			if(response.getStatusLine().getStatusCode() == 413){
				// This status code means that the request was too long, so no point keeping it in the db and trying again..
				return true;
			}
			return false;
		} else {
			return true;
		}
	}

	public class NullHostNameVerifier implements HostnameVerifier {

		public boolean verify(String hostname, SSLSession session) {
			Log.i("RestUtilImpl", "Approving certificate for " + hostname);
			return true;
		}
	}
	
	public boolean sendAnalytics(String json, AnalyticsCloud provider) {
		return sendAnalytics(json, provider, null);
	}
	
	// TODO: unite this method with the sendToErrorHandling above
	@SuppressLint("TrulyRandom") 
	private boolean sendAnalytics(String json, AnalyticsCloud provider, String date) {
		String url = null;
		
		if(provider == AnalyticsCloud.Amazon){
			url = AniwaysPrivateConfig.getInstance().analyticsS3Address; 
		}
		else if(provider == AnalyticsCloud.Azure){
			url = AniwaysPrivateConfig.getInstance().azureAnalyticsEndpint;
		}
		else if(provider == AnalyticsCloud.Aniways){
			url = AniwaysPrivateConfig.getInstance().baseAnalyticsUrl;
		}
		else{
			Log.e(true, TAG, "Bad analytics provider: " + provider);
			return false;
		}
		
		HttpURLConnection connection = null;

		try{
			
			byte[] byteArrayToSend = null;
			boolean isCompressed = false;
			if(AniwaysPrivateConfig.getInstance().compressAnalytics){
				try{
					byteArrayToSend = compress(json);
					isCompressed = true;
				}
				catch (IOException e) {
					Log.e(true, TAG, "Could not compress json: " + json, e);
				}
			}
			if(byteArrayToSend == null){
				byteArrayToSend = json.getBytes("UTF-8");
				isCompressed = false;
			}

			// Make the https work - create an always true host name verifier
			
			// Subcription - id 7070f51d-bd60-4990-b107-9d2fbc14fa16
			// Endpoint - https://aniways.blob.core.windows.net/aniways-analytics
			
			TrustManager trustAllCerts = new X509TrustManager() {
				public X509Certificate[] getAcceptedIssuers() {
					return new X509Certificate[] {};
				}

				public void checkClientTrusted(X509Certificate[] chain,
						String authType) throws CertificateException {
				}

				public void checkServerTrusted(X509Certificate[] chain,
						String authType) throws CertificateException {
				}
			};
			HttpsURLConnection.setDefaultHostnameVerifier(new NullHostNameVerifier());
			SSLContext context = SSLContext.getInstance("TLS");
			context.init(null, new X509TrustManager[]{(X509TrustManager) trustAllCerts}, new SecureRandom());
			HttpsURLConnection.setDefaultSSLSocketFactory(context.getSocketFactory());

			// Setup the file name
			SimpleDateFormat dateTimeFormat = new SimpleDateFormat("yyyyMMdd'T'HHmmss", Locale.US);
			dateTimeFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
			String dateTimeString = dateTimeFormat.format(Calendar.getInstance().getTime());

			SimpleDateFormat dateFormat = new SimpleDateFormat("yyyyMMdd", Locale.US);
			dateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
			String dateString = dateFormat.format(Calendar.getInstance().getTime());
			
			SimpleDateFormat hourFormat = new SimpleDateFormat("HH", Locale.US);
			hourFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
			String hourString = hourFormat.format(Calendar.getInstance().getTime());
			
			String appName = Build.getAppName(mContext);
			if(appName == null){
				appName = AniwaysPrivateConfig.getInstance().appId;
			}
			
			Random r = new Random();
			int partitionNumber = r.nextInt(AniwaysPrivateConfig.getInstance().numberOfAnalyticsPartitions);
			String partitionNumberString = Integer.toString(partitionNumber);
			int numberOfZerosToAdd = AniwaysPrivateConfig.getInstance().analyticsPartitionsStringLength - partitionNumberString.length();
			for(int i = 0; i < numberOfZerosToAdd; i++){
				partitionNumberString = '0' + partitionNumberString;
			}
			
			String fileName = "v2/android/" + appName.replace(" ", "_") + "/" + dateString + "/" + hourString + "/" + partitionNumberString + "/" + dateTimeString + "." + Installation.id(mContext).replace("-", "") + ".json";
			if(isCompressed){
				fileName += ".gz";
			}

			// Create the connection
			URL urlConnectionOpener = new URL(url + "/" + fileName);
			// Open the connection
			connection = (HttpURLConnection) urlConnectionOpener.openConnection();

			// Setup the connection
			connection.setReadTimeout(AniwaysServiceUtils.READ_TIMEOUT);
			connection.setConnectTimeout(AniwaysServiceUtils.CONNECTION_TIMEOUT);
			HttpURLConnection.setFollowRedirects(true);
			if (Utils.isAndroidVersionAtLeast(8)){
				// Fix bug prior to Froyo
				connection.setRequestProperty("Connection", "Keep-Alive");
			}
			else{
				System.setProperty("http.keepAlive", "false");
			}
			connection.setDoInput(true);
			connection.setDoOutput(true);
			connection.setUseCaches(false);
			if(provider == AnalyticsCloud.Aniways){
				connection.setRequestMethod("POST");
			}
			else{
				connection.setRequestMethod("PUT");
			}
			connection.setRequestProperty("Content-Type", "application/json; charset=utf-8");
			if(isCompressed){
				connection.setRequestProperty("Content-Encoding", "gzip");
				// TODO: find out if need to set content encoding is it is not zipped
			}
			connection.setRequestProperty("Content-Length", Integer.toString(byteArrayToSend.length));
			
			if(provider == AnalyticsCloud.Amazon){
				// TODO: add user agent
				connection.setRequestProperty("x-amz-storage-class", AniwaysPrivateConfig.getInstance().s3StorageClass);
				connection.setRequestProperty("x-amz-acl", AniwaysPrivateConfig.getInstance().s3Acl);
			}
			else if(provider == AnalyticsCloud.Azure){
				AzureRequest.addAzureHeaders(connection, byteArrayToSend.length, date);
			}
			else if(provider == AnalyticsCloud.Aniways){
				// TODO: add user agent
			}
			else{
				Log.e(true, TAG, "Bad analytics provider2: " + provider);
				return false;
			}
			
			// Get the output stream
			int bytesRead, bytesAvailable, bufferSize;
			byte[] buffer;
			int maxBufferSize = 1*1024*1024;
			DataOutputStream outputStream = new DataOutputStream(connection.getOutputStream());

			// Send the body to the output stream
			ByteArrayInputStream fileInputStream = new ByteArrayInputStream(byteArrayToSend);
			bytesAvailable = fileInputStream.available();
			bufferSize = Math.min(bytesAvailable, maxBufferSize);
			buffer = new byte[bufferSize];
			bytesRead = fileInputStream.read(buffer, 0, bufferSize);
			while(bytesRead > 0) {
				outputStream.write(buffer, 0, bufferSize);
				bytesAvailable = fileInputStream.available();
				bufferSize = Math.min(bytesAvailable, maxBufferSize);
				bytesRead = fileInputStream.read(buffer, 0, bufferSize);
			}

			// Finish the transfer
			outputStream.flush();
			outputStream.close();
		}
		catch (MalformedURLException e) {
			Log.eToGaOnly(true, TAG, "Malformed analytics URL: " + url, e);
		} 
		catch (ProtocolException e) {
			Log.eToGaOnly(true, TAG, "Protocol Exception for analytics url: " + url, e);
		}
		catch (IOException e) {
			String networkInterface = Utils.isInternetAvailable(mContext);
			if (networkInterface != null){
				Log.wToGaOnly(true, TAG, "IO Exception on analytics URL " + url + " Network interface: " + networkInterface, e);
			}
			else{
				Log.wToGaOnly(false,TAG, "IO Exception in analytics, but there is no internet connection, so that's expected. URL: " + url + " Retry number: ", e);
			}
		} catch (IllegalArgumentException e) {
			Log.eToGaOnly(true, TAG, "Illegal arg Exception on analytics URL: " + url, e);
		}
		catch(Throwable e){
			Log.wToGaOnly(true, TAG, "Failed to send request on analytics URL: " + url, e);
		}

		String responseString = "";
		try {
			if (connection == null) {
				// there's been an error
				Log.wToGaOnly(false, TAG, "Failed to make request to the analytics server. Url: " + url, null);
				return false;
			} 
			else if (connection.getResponseCode() != 200 && connection.getResponseCode() != 201 && connection.getResponseCode() != 204) {
				
				// If we get a 403 response from Azure then its probably an issue with the x-ms-date header which 
				// is more than 15 minutes difference than the server's. So, we take the date from the server
				// and transplant it into a new request and try again..
				if(connection.getResponseCode() == 403 && provider == AnalyticsCloud.Azure && date == null){
					String dateFromServer = connection.getHeaderField("Date");
					if(!TextUtils.isEmpty(dateFromServer)){
						dateFromServer.replace("[", "");
						dateFromServer.replace("]", "");
						Log.w(false, TAG, "Couldnt send file to Azure. Trying to send again with server time: " + dateFromServer);
						return sendAnalytics(json, provider, dateFromServer);
					}
				}
				
				// there's been a server error
				responseString += connection.getResponseMessage() + "\n";

				if(connection.getHeaderFields() != null && connection.getHeaderFields().entrySet() != null){
					StringBuilder responseHeadersSb = new StringBuilder();
					for (Entry<String, List<String>> header : connection.getHeaderFields().entrySet()) {
						responseHeadersSb.append("\n" + header.getKey() + ":" + header.getValue());
					}
					responseString += responseHeadersSb.toString();
				}
				
				InputStream responseStream = connection.getInputStream();
				if(responseStream != null){
					responseString += "\n" + convertStreamToString(responseStream);
					responseStream.close();
				}
				
				InputStream errorStream = connection.getErrorStream();
				if(errorStream != null){
					responseString += "\n" + convertStreamToString(errorStream);
					errorStream.close();
				}

				Log.eToGaOnly(true, TAG, "Received a failed response from the analytics server. Url: " + url + "Response code: " + connection.getResponseCode() + ". Response: " + responseString, null);
				return false;

			} else {
				return true;
			}
		} catch (ParseException e) {
			Log.wToGaOnly(true, TAG, "Failed to parse the response from the analytics server. Url: " + url + ". Response: " + responseString, e);
		} catch (IOException e) {
			String networkInterface = Utils.isInternetAvailable(mContext);
			if (networkInterface != null){
				Log.wToGaOnly(true, TAG, "Failed to read the response from the analytics server. Url: " + url + ". Response: " + responseString, e);
			}
			else{
				Log.wToGaOnly(false,TAG, "Failed to read the response from the analytics server, but there is no internet connection, so that's expected. URL: " + url + " Retry number: ", e);
			}
		}
		finally{
			if(connection != null){
				connection.disconnect();
			}
		}

		return false;
	}

	private String convertStreamToString(InputStream is) {
		BufferedReader reader = new BufferedReader(new InputStreamReader(is));
		StringBuilder sb = new StringBuilder();

		String line = null;
		try {
			while ((line = reader.readLine()) != null) {
				sb.append(line);
			}
		} catch (IOException e) {
			e.printStackTrace();
		} finally {
			try {
				is.close();
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
		return sb.toString();
	}

	private byte[] compress(String string) throws IOException {
		ByteArrayOutputStream os = new ByteArrayOutputStream(string.length());
		GZIPOutputStream gos = new GZIPOutputStream(os);
		gos.write(string.getBytes());
		gos.close();
		byte[] compressed = os.toByteArray();
		os.close();
		return compressed;
	}
}
