package com.aniways.analytics.info;

import java.io.IOException;
import java.util.List;
import com.aniways.Log;
import com.aniways.Utils;
import com.aniways.analytics.models.EasyJSONObject;
import com.aniways.service.utils.AniwaysServiceUtils;

import org.json.JSONObject;

import android.content.Context;
import android.content.SharedPreferences;
import android.content.SharedPreferences.Editor;
import android.location.Address;
import android.location.Criteria;
import android.location.Geocoder;
import android.location.LocationManager;

public class Location implements Info<JSONObject> {

	private static final String TAG = "AniwaysAnalyticsContextLocation";
	private static final String LAST_LOCATION_KEY_PREFIX = "com.aniways.LAST_LOCATION_";
	public static final String UNKNOWN = "unknown";
	public static final String COUNTRY_CODE = "countryCode";
	public static final String COUNTRY_NAME = "country";
	public static final String STATE = "state";
	public static final String COUNTY = "county";
	public static final String CITY = "city";
	public static final String NEIGHBORHOOD = "neighborhood";
	public static final String LANDMARK = "landmark";
	public static final String LATITUDE = "latitude";
	public static final String LONGITUDE = "longitude";
	public static final String SPEED = "speed";

	@Override
	public String getKey() {
		return "location";
	}

	@Override
	public EasyJSONObject get(Context context) {
        EasyJSONObject object;

        try {
            object = new EasyJSONObject();

            String longitude = null;
            String latitude = null;
            String speed = null;
            String countryName = null;
            String countryCode = null;
            String city = null;
            String neighborhood = null;
            String county = null;
            String landmark = null;
            String state = null;

            boolean coordinatesTaken = false;
            boolean extraPropertiesTaken = false;

            try {
                LocationManager locationManager = (LocationManager) context.getSystemService(Context.LOCATION_SERVICE);

                // http://stackoverflow.com/questions/5505429/switching-between-network-and-gps-provider
                // only read from the network to avoid turning on GPS
                Criteria crit = new Criteria();
                crit.setPowerRequirement(Criteria.POWER_LOW);
                crit.setAccuracy(Criteria.ACCURACY_COARSE);
                String provider = locationManager.getBestProvider(crit, true);

                if (provider != null) {
                    android.location.Location location = locationManager.getLastKnownLocation(provider);

                    if (location != null) {
                        latitude = Double.toString(location.getLatitude());
                        longitude = Double.toString(location.getLongitude());
                        speed = Float.toString(location.getSpeed());
                        coordinatesTaken = true;

                        try {
                            Geocoder geocoder = new Geocoder(context.getApplicationContext(), java.util.Locale.US);
                            List<Address> listAddresses = geocoder.getFromLocation(location.getLatitude(), location.getLongitude(), 1);
                            if (null != listAddresses && listAddresses.size() > 0) {
                                countryCode = listAddresses.get(0).getCountryCode();
                                countryName = listAddresses.get(0).getCountryName();
                                state = listAddresses.get(0).getAdminArea();
                                county = listAddresses.get(0).getSubAdminArea();
                                city = listAddresses.get(0).getLocality();
                                neighborhood = listAddresses.get(0).getSubLocality();
                                landmark = listAddresses.get(0).getFeatureName();
                                extraPropertiesTaken = true;
                            }
                        } catch (IOException e) {
                            // This exception is to be expected if there is no Internet connection
                            Log.w(false, TAG, "Caught Exception in get location", e);
                        }
                    }
                }
            } catch (Exception ex) {
                Log.eToGaOnly(true, TAG, "Caught Exception in Location info", ex);
            }

            if (!coordinatesTaken) {
                // Get from previous time that succeeded
                latitude = getProperty(context, LATITUDE);
                longitude = getProperty(context, LONGITUDE);
                speed = getProperty(context, SPEED);
            }

            if (!extraPropertiesTaken) {
                // Get from previous time that succeeded
                countryCode = getProperty(context, COUNTRY_CODE);
                countryName = getProperty(context, COUNTRY_NAME);
                state = getProperty(context, STATE);
                county = getProperty(context, COUNTY);
                city = getProperty(context, CITY);
                neighborhood = getProperty(context, NEIGHBORHOOD);
                landmark = getProperty(context, LANDMARK);
            }

            object.put(LATITUDE, (latitude == null ? UNKNOWN : latitude));
            object.put(LONGITUDE, (longitude == null ? UNKNOWN : longitude));
            object.put(SPEED, (speed == null ? UNKNOWN : speed));
            object.put(COUNTRY_CODE, (countryCode == null ? UNKNOWN : countryCode));
            object.put(COUNTRY_NAME, (countryName == null ? UNKNOWN : countryName));
            object.put(STATE, (state == null ? UNKNOWN : state));
            object.put(COUNTY, (county == null ? UNKNOWN : county));
            object.put(CITY, (city == null ? UNKNOWN : city));
            object.put(NEIGHBORHOOD, (neighborhood == null ? UNKNOWN : neighborhood));
            object.put(LANDMARK, (landmark == null ? UNKNOWN : landmark));

            if (coordinatesTaken) {
                setProperty(context, LATITUDE, (latitude == null ? UNKNOWN : latitude));
                setProperty(context, LONGITUDE, (longitude == null ? UNKNOWN : longitude));
                setProperty(context, SPEED, (speed == null ? UNKNOWN : speed));
            }
            if (extraPropertiesTaken) {
                setProperty(context, COUNTRY_CODE, (countryCode == null ? UNKNOWN : countryCode));
                setProperty(context, COUNTRY_NAME, (countryName == null ? UNKNOWN : countryName));
                setProperty(context, STATE, (state == null ? UNKNOWN : state));
                setProperty(context, COUNTY, (county == null ? UNKNOWN : county));
                setProperty(context, CITY, (city == null ? UNKNOWN : city));
                setProperty(context, NEIGHBORHOOD, (neighborhood == null ? UNKNOWN : neighborhood));
                setProperty(context, LANDMARK, (landmark == null ? UNKNOWN : landmark));
            }
        } catch (Throwable e) {
            // This exception is to be expected if there is no Internet connection
            Log.e(true, TAG, "Caught Exception while processing location data", e);
            object = null;
        }

		return object;
	}
	
	private synchronized static void setProperty(Context context, String name, String value){

		// open the shared preferences
		SharedPreferences prefs = 
				context.getSharedPreferences(AniwaysServiceUtils.SHARED_PREFERENCES, Utils.getSharedPreferencesFlags());
		Editor edit = prefs.edit();

		edit.putString(LAST_LOCATION_KEY_PREFIX + name, value);
		edit.commit();
	}

	private synchronized static String getProperty(Context context, String name) {
		// open the shared preferences.
		SharedPreferences prefs = 
				context.getSharedPreferences(AniwaysServiceUtils.SHARED_PREFERENCES, Utils.getSharedPreferencesFlags());

		return prefs.getString(LAST_LOCATION_KEY_PREFIX + name, UNKNOWN);
	}

}
