package com.aniways.analytics;

import com.aniways.IAniwaysImageSpan;
import com.aniways.IconData;
import com.aniways.analytics.models.EventProperties;
import com.aniways.analytics.utils.ContextualEventData;
import com.aniways.data.AnimatedGifPhraseAssetInfoBuilder;
import com.aniways.data.AniwaysConfiguration;
import com.aniways.data.AniwaysPrivateConfig;
import com.aniways.data.EmoticonsPhraseAssetBuilder;
import com.aniways.data.Phrase;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.List;
import java.util.TimeZone;

import static com.aniways.analytics.AnalyticsReporter.PhrasesEventAction;

public class ContextualEventReporter {
    private static final List<PhrasesEventAction> requiredMessageIdEvents = Arrays.asList(PhrasesEventAction.suggested, PhrasesEventAction.tapped
            , PhrasesEventAction.selected, PhrasesEventAction.sent, PhrasesEventAction.sentExternalContent);

    public static void reportContextualEvent(final ContextualEventData contextualEventData){
        if(AniwaysPrivateConfig.getInstance().analyticsEventsVerbosity.ordinal() > AniwaysConfiguration.Verbosity.Info.ordinal()){
            return;
        }

        final Calendar timestamp = Calendar.getInstance(TimeZone.getTimeZone("UTC"));

        EventProperties props = new EventProperties();
        props.put("category", "contextual");
        props.put("action", contextualEventData.action.getEventReportName());

        if (requiredMessageIdEvents.contains(contextualEventData.action)){
            props.put("message_id", contextualEventData.messageId);
        }
        //report all common contextual event properties
        if(contextualEventData.phrase != null){
            props.put("phrase_name", contextualEventData.phrase.getName());
            props.put("part_to_replace", contextualEventData.phrase.getPartToReplace());
            //props.put("phrase_language", );Todo: add this once language is supported\
        }

        if (isReportIndexInMessageRequired(contextualEventData)){
            props.put("index_in_message", contextualEventData.indexInMessage);
        }

        if (isReportAssetsRequired(contextualEventData)){
            addAssetsReportToProps(props, contextualEventData);
        }

        if (isReportPhraseLanguageRequired(contextualEventData)){
            //Todo: add this to props once v3 is supported
        }

        if(isUserInteractedWithContentEvent(contextualEventData)){
            addUserInteractionEventProperties(props,contextualEventData);
        }

        if (isExternalConnectSent(contextualEventData)){
            addExternalContentEventProperties(props,contextualEventData);
        }

        AniwaysAnalyticsReporter.track(props, timestamp);
    }

    private static void addExternalContentEventProperties(EventProperties props, ContextualEventData contextualEventData) {
        IconData[] icons = contextualEventData.icons;
        if (icons == null || icons.length <= 0) {
            return;
        }

        IconData interactedAsset = icons[0];

        props.put("asset_type", interactedAsset.assetType.toString());
        props.put("provider_name", interactedAsset.assetProvider.toString());
        props.put("content_id", EventReporterCommonMedhods.extractAssetId(interactedAsset));
        props.put("origin", contextualEventData.imageSpanMetadata.iconSelectionOrigin);

        if (interactedAsset.isEmoji()){
            props.put("unicode_representation", interactedAsset.getUnicodeRepresentation());
        }
    }

    private static boolean isExternalConnectSent(ContextualEventData contextualEventData) {
        return contextualEventData.action == PhrasesEventAction.sentExternalContent;
    }

    private static void addUserInteractionEventProperties(EventProperties props, ContextualEventData contextualEventData) {
        IconData[] icons = contextualEventData.icons;
        if (icons == null || icons.length <= 0) {
            return;
        }

        IconData interactedAsset = icons[0];

        props.put("asset_type", interactedAsset.assetType.toString());
        props.put("provider_name", interactedAsset.assetProvider.toString());
        props.put("content_id", EventReporterCommonMedhods.extractAssetId(interactedAsset));
        props.put("from_auto_popup", contextualEventData.imageSpanMetadata.iconSelectionOrigin == IAniwaysImageSpan.IconSelectionOrigin.ContextualAuto);
        props.put("from_reveal_more", contextualEventData.isFromRevealMore());
        props.put("asset_index", contextualEventData.position);
        props.put("asset_index_out_of", contextualEventData.outOf);

        //Todo: add places info once supported here
    }

    private static boolean isUserInteractedWithContentEvent(ContextualEventData contextualEventData) {
        PhrasesEventAction action = contextualEventData.action;

        return (action == PhrasesEventAction.selected
                || action == PhrasesEventAction.sent);
    }

    private static boolean isReportAssetsRequired(ContextualEventData contextualEventData) {
        PhrasesEventAction action = contextualEventData.action;

        return (action == PhrasesEventAction.suggested || action == PhrasesEventAction.tapped);
    }

    private static boolean isReportPhraseLanguageRequired(ContextualEventData contextualEventData) {
        return false;
    }

    private static void addAssetsReportToProps(EventProperties props, ContextualEventData contextualEventData) {
        Phrase phrase = contextualEventData.phrase;
        if (phrase == null){
            return;
        }

        List<String> availableAssetTypesForPhrase = new ArrayList<>();

        AnimatedGifPhraseAssetInfoBuilder animatedGifs = phrase.animatedGifsPhraseAssetBuilder;
        EmoticonsPhraseAssetBuilder iconsData = phrase.emoticonsPhraseAssetBuilder;

        //TODO: The names here should be as defined in the json
        if (animatedGifs != null){
            availableAssetTypesForPhrase.add("Animated Icons");
        }

        if (iconsData != null){
            //addIconIdsToAssetList(availableAssetTypesForPhrase,iconsData);
            availableAssetTypesForPhrase.add("Emoticons");
        }

        if (!availableAssetTypesForPhrase.isEmpty()){
            props.put("assets", availableAssetTypesForPhrase);
            //TODO: this is temporary, until we really know the assets_seen
            props.put("assets_seen", availableAssetTypesForPhrase);
        }
    }

    private static void addIconIdsToAssetList(List<String> assetsIds, EmoticonsPhraseAssetBuilder iconsData) {
        if (iconsData.icons == null || iconsData.icons.length <= 0){
            return;
        }

        for (IconData icon : iconsData.icons){
            assetsIds.add(String.valueOf(icon.id));
        }
    }

    private static boolean isReportIndexInMessageRequired(ContextualEventData contextualEventData) {
        PhrasesEventAction action = contextualEventData.action;
        return (action == PhrasesEventAction.suggested || action == PhrasesEventAction.sent || action == PhrasesEventAction.sentExternalContent);
    }
}
