package com.aniways.analytics;

import com.aniways.analytics.db.JsonPayloadSerializer;
import com.aniways.analytics.info.EventMetaDataCreator;
import com.aniways.analytics.models.Alias;
import com.aniways.analytics.models.BasePayload;
import com.aniways.analytics.models.Context;
import com.aniways.analytics.models.EventProperties;
import com.aniways.analytics.models.Identify;
import com.aniways.analytics.models.Log;
import com.aniways.analytics.models.Screen;
import com.aniways.analytics.models.Timing;
import com.aniways.analytics.models.Track;
import com.aniways.analytics.models.Traits;
import com.aniways.analytics.service.AniwaysAnalyticsService;
import com.aniways.analytics.stats.AnalyticsStatistics;
import com.aniways.data.AniwaysConfiguration.Verbosity;
import com.aniways.data.AniwaysPrivateConfig;

import java.util.Calendar;
import java.util.UUID;
import android.content.Intent;

import org.json.JSONException;

/**
 * Reports to the Aniways Server
 * @author Shai
 *
 */
public class AniwaysAnalyticsReporter {

    private static final String TAG = "AniwaysAnalyticsReporter";
    private static AnalyticsStatistics statistics;

	private static boolean initialized;

	private static String sessionId;

	private static DynamicPropertiesEvaluator dynamicPropertiesEvaluator;

	private static JsonPayloadSerializer serializer;

	private static android.content.Context sContext;

	/**
	 * Gets the client statistics
	 * @return+
	 */
	public static AnalyticsStatistics getStatistics() {
		return statistics;
	}

	static void initialize(final android.content.Context context) { 

		if (initialized) return;
		
		sContext = context;

		statistics = new AnalyticsStatistics();

		// create the serializer
		serializer = new JsonPayloadSerializer();

		// add a global dynamic context
		dynamicPropertiesEvaluator = new DynamicPropertiesEvaluator(){

			@Override
			public Context getEventMetaDataProperties() {
				Context result = new Context(EventMetaDataCreator.getInstance().create(sContext));
				return result;
			}

            @Override
            public Context getEventDataProperties(EventProperties properties, Calendar timestamp) {
                Context result = new Context();
                result.put("category",properties.get("category"));
                result.put("action",properties.get("action"));
                result.put("timestamp",timestamp);
                result.put("session_id", getSessionId());

                properties.remove("category");
                properties.remove("action");

                return result;
            }
        };
		
		initialized = true;
		
		startNewSession();
		
		//requestFlush();
	}
	
	//
	// Identify
	//
	
	/**
	 * Identifying a user ties all of their actions to an id, and associates
	 * user traits to that id.
	 * 
	 * @param userId
	 *            the user's id after they are logged in. It's the same id as
	 *            which you would recognize a signed-in user in your system.
	 * 
	 * @param traits
	 *            a dictionary with keys like subscriptionPlan or age. You only
	 *            need to record a trait once, no need to send it again.
	 * 
	 * @param timestamp
	 *            a {@link java.util.Calendar} representing when the identify took place.
	 *            If the identify just happened, leave it blank and we'll use
	 *            the server's time. If you are importing data from the past,
	 *            make sure you provide this argument.
	 * 
	 */
	static void identify(String userId, Traits traits, Calendar timestamp) {
		
		if (userId == null || userId.length() == 0) {
			throw new IllegalArgumentException("analytics-android #identify must be initialized with a valid user id.");
		}

		Context	context = dynamicPropertiesEvaluator.getEventMetaDataProperties();
		if (traits == null)
			traits = new Traits();

		Identify identify = new Identify(userId, traits, timestamp, context);

		enqueue(identify);

		statistics.updateIdentifies(1);
	}

	//
	// Track
	//

	/**
	 * Whenever a user triggers an event, you’ll want to track it.
	 * 
	 * Track will use an automatically generated userId unless one has been
	 * provided by identify(..).
	 * 
	 * @param event
	 *            describes what this user just did. It's a human readable
	 *            description like "Played a Song", "Printed a Report" or
	 *            "Updated Status".
	 * 
	 * @param properties
	 *            a dictionary with items that describe the event in more
	 *            detail. This argument is optional, but highly
	 *            recommended—you’ll find these properties extremely useful
	 *            later.
	 * 
	 * @param timestamp
	 *            a {@link java.util.Calendar} object representing when the track took
	 *            place. If the event just happened, leave it blank and we'll
	 *            use the server's time. If you are importing data from the
	 *            past, make sure you provide this argument. 
	 */
	static void track(EventProperties properties, Calendar timestamp) {

		Context	metaData = dynamicPropertiesEvaluator.getEventMetaDataProperties();
		if (properties == null)
			properties = new EventProperties();

        Context eventData = dynamicPropertiesEvaluator.getEventDataProperties(properties, timestamp);

		Track track = new Track(properties, timestamp, metaData,eventData);

		enqueue(track);

		statistics.updateTracks(1);
	}

	static void trackLog(EventProperties properties, Calendar timestamp) {

		Context	context = dynamicPropertiesEvaluator.getEventMetaDataProperties();
		if (properties == null)
			properties = new EventProperties();

		Log log = new Log(properties, timestamp, context);

		enqueue(log);

		statistics.updateTracks(1);
	}

	static void trackError(Verbosity severity, EventProperties properties, Calendar timestamp) {

		Context	context = dynamicPropertiesEvaluator.getEventMetaDataProperties();
		if (properties == null)
			properties = new EventProperties();

		com.aniways.analytics.models.Error error = new com.aniways.analytics.models.Error(severity.toString(), properties, timestamp, context);

		enqueue(error);

		statistics.updateTracks(1);
	}

	//
	// Screen
	//

	/**
	 * Whenever a user opens a new screen (or activity), track its screen view.
	 * Example:
	 * 
	 * 	Analytics.screen("Login Page");
	 * 
	 * You don't need to provide a userId to this method, because this library uses the most recent userId. 
	 * If identify(userId) or track(userId) was never called, a randomly generated session 
	 * id will be used. Otherwise, the most recent cached userId is used.  
	 * 
	 * @param screen
	 *            describes the screen name of the activity that the user just
	 *            opened. We don't recommend to name each screen dynamically. For 
	 *            example, if a screen shows a new article, you should call it
	 *            "News Screen" instead of the name of the news article.
	 * 
	 * @param properties
	 *            a dictionary with items that describe the screen in more
	 *            detail. This argument is optional, but highly
	 *            recommended—you’ll find these properties extremely useful
	 *            later.
	 * 
	 * @param timestamp
	 *            a {@link java.util.Calendar} object representing when the track took
	 *            place. If the event just happened, leave it blank and we'll
	 *            use the server's time. If you are importing data from the
	 *            past, make sure you provide this argument.
	 *  
	 */
	static void screen(String screen, EventProperties properties, Calendar timestamp) {

		if (screen == null || screen.length() == 0) {
			throw new IllegalArgumentException("analytics-android #screen must be initialized with a valid screen name.");
		}

		Context context = dynamicPropertiesEvaluator.getEventMetaDataProperties();
		if (properties == null)
			properties = new EventProperties();


		Screen screenAction = new Screen(screen, properties, timestamp, context);

		enqueue(screenAction);

		statistics.updateScreens(1);
	}

	//
	// Timing
	//

	static void timing(long interval, EventProperties properties, Calendar timestamp) {

		Context	context = dynamicPropertiesEvaluator.getEventMetaDataProperties();
		if (properties == null)
			properties = new EventProperties();

		Timing timing = new Timing(interval, properties, timestamp, context);

		enqueue(timing);

		statistics.updateTiming(1);
	}

	//
	// Alias
	//

	/**
	 * Aliases an anonymous user into an identified user.
	 * 
	 * @param from
	 *            the anonymous user's id before they are logged in.
	 * 
	 * @param to
	 *            the identified user's id after they're logged in.
	 * 
	 * 
	 * @param timestamp
	 *            a {@link java.util.Calendar} object representing when the track took
	 *            place. If the event just happened, leave it blank and we'll
	 *            use the server's time. If you are importing data from the
	 *            past, make sure you provide this argument.
	 *              
	 */
	static void alias(String from, String to, Calendar timestamp) {

		if (from == null || from.length() == 0) {
			throw new IllegalArgumentException("analytics-android #alias must be initialized with a valid from id.");
		}

		if (to == null || to.length() == 0) {
			throw new IllegalArgumentException("analytics-android #alias must be initialized with a valid to id.");
		}

		Context context = dynamicPropertiesEvaluator.getEventMetaDataProperties();

		Alias alias = new Alias(from, to, timestamp, context);

		enqueue(alias);

		statistics.updateAlias(1);
	}


	//
	// Internal
	//

	

	/**
	 * Requests that the Analytics serivce flush the events in its belly 
	 */
	//private static void requestFlush(){
	//	// Pass the request to the Analytics service
	//	Intent intent = new Intent(sContext, AniwaysAnalyticsService.class);
	//	intent.setAction(AniwaysAnalyticsService.ACTION_FLUSH);
	//
	//	sContext.startService(intent);
	//}

	/**
	 * Returns whether the client is initialized
	 * @return
	 */
	static boolean isInitialized() {
		return initialized;
	}

	static void startNewSession() {
		if(!AniwaysPrivateConfig.getInstance().isAnalyticsDisabled()){
			sessionId = UUID.randomUUID().toString();
		}
	}

	//
	// Internal
	//

	/**
	 * Serialize the payload (preferably on a different thread), and then send it to the service with an intent 
	 * @param payload
	 */
	private static void enqueue(BasePayload payload){
        try {
            if(AniwaysPrivateConfig.getInstance().printAnalyticsToLogs) {
                com.aniways.Log.i(TAG, payload.toString(4));
            }
        } catch (JSONException e) {
            e.printStackTrace();
        }

		// TODO: Do this on a different thread..
		String json = serializer.serialize(payload);
	
		// Pass the request to the Analytics service
		Intent intent = new Intent(sContext, AniwaysAnalyticsService.class);
		intent.setAction(AniwaysAnalyticsService.ACTION_ENQUEUE);
		intent.putExtra(AniwaysAnalyticsService.SERIALIZED_EVENT_KEY, json);

		sContext.startService(intent);
	}

	private static String getSessionId(){
		return sessionId;
	}
}
