package com.aniways.volley.toolbox;

import java.io.InputStream;

import android.content.Context;
import android.content.res.Resources.NotFoundException;
import android.graphics.Bitmap;
import android.support.v4.util.LruCache;

import com.aniways.AniwaysBitmapDecodeUtils;
import com.aniways.Log;
import com.aniways.Utils;
import com.aniways.data.AniwaysPrivateConfig;
import com.aniways.volley.toolbox.ImageLoader.ImageCache;

public class LruBitmapCache extends LruCache<String, Bitmap> implements ImageCache {
	private static final String ANIWAYS_ICON_NAME_PREFIX = "aniways_ic_id_";
	private static final String TAG = "AniwaysLRUBitmapCache";
	private Context mContext;

	public static int getDefaultLruCacheSize() {
		final int maxMemory = (int) (Runtime.getRuntime().maxMemory() / 1024);
		final int cacheSize = maxMemory / 8;

		return cacheSize;
	}


	public LruBitmapCache(Context context) {
		this(getDefaultLruCacheSize());
		mContext = context;
	}

	public LruBitmapCache(int sizeInKiloBytes) {
		super(sizeInKiloBytes);
	}

	@Override
	protected int sizeOf(String key, Bitmap value) {
		return value.getRowBytes() * value.getHeight() / 1024;
	}

	@Override
	public Bitmap getBitmap(String url, int maxWidth, int maxHeight) {
		Bitmap bitmap = get(url);
		//TODO: There is a problem here that we do not take into account the icon type (big, emoji, banner, etc.). These different types
		// might have different config values whether to take pre-installed or not..
		if(bitmap == null && AniwaysPrivateConfig.getInstance().usePreInstalledIcons){
			String imageName = url.substring(url.lastIndexOf('/') + 1);
			String packageName = mContext.getPackageName();
			if(packageName == null){
				Log.e(true, TAG, "package Name is null");
			}
			else{
				int id = mContext.getResources().getIdentifier(ANIWAYS_ICON_NAME_PREFIX + imageName.replace(".png",""), "raw", packageName);
				if(id > 0) {
					Log.v(TAG, "Fetching image from preInstalled icons");
					try {
						InputStream stream = mContext.getResources().openRawResource(id);
						if(Utils.isAndroidVersionAtLeast(20)){
							InputStream stream2 = mContext.getResources().openRawResource(id);

							bitmap = AniwaysBitmapDecodeUtils.decodeBitmapFromStream(stream, stream2, maxWidth, maxHeight, imageName, false);
						}
						else{
							bitmap = AniwaysBitmapDecodeUtils.decodeBitmapFromStream(stream, maxWidth, maxHeight, imageName, false);
						}
						if(bitmap == null){
							Log.w(false, TAG, "Bitmap is null: " + imageName);
							return null;
						}
						Log.v(TAG, "Decoded pre-installed bitmap: " + imageName);
						putBitmap(url, bitmap);
					} catch(NotFoundException ex){
						Log.w(true, TAG, "Icon file doesn't exist in preInstalled images: " + imageName);
						return null;
					} catch (OutOfMemoryError ex){
						System.gc();
						Log.e(true, TAG, "Could not create bitmap because not enough memory: " + imageName);
						// TODO: This would cause calling layer to get from network and cost us money and probably also
						//		 get the same Exception..
						return null;
					} catch(Throwable ex){
						Log.w(true, TAG, "Caught Exception while getting bitmap: " + ex);
						return null;
					}
				}
			}
		}
		return bitmap;
	}

	

	@Override
	public void putBitmap(String url, Object bitmap) {
		if(AniwaysPrivateConfig.getInstance().isLowMemoryDevice()){ return; }
		put(url, (Bitmap) bitmap);
	}
}
