package com.aniways.service.utils;

import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import com.aniways.Log;

/**
 * File utility methods.
 */
public class FileUtils
{
	private static final String TAG = "AniwaysFileUtils"; 

	/**
	 * Get a list of filenames in this folder.
	 * @param dirPath  Full path of directory 
	 * @param sort  Use 1 for {@link String#CASE_INSENSITIVE_ORDER}, 0 for no sort, -1 for reverse sort
	 * @return list of filenames (the names only, not the full path),
	 * or null if folder doesn't exist or isn't a directory,
	 * or if nothing matches fileNameFilterPattern
	 */
	public static List<String> getFileNames(String dirPath, int sort) {
		// the list of files in the directory
		List<String> filesList = new ArrayList<String>();
		File fileDir = new File(dirPath);

		// Check if the given path is to a directory and if its exists
		if (!fileDir.exists() || !fileDir.isDirectory()) {
			// invalid path return null
			Log.w(true, TAG, "Invalid dir path: " + fileDir);
			return null;
		}

		// list all of the files in the dir
		String[] files = fileDir.list();

		if (null == files) {
			// file listing failed return null
			Log.w(true, TAG, "Could not get listing of files in dir: " + fileDir);
			return null;
		}

		// convert the files array to a list
		// This is imutable by design!! - if for some reason the code tries to change this list then there is a bug!!
		// If downstream code wants to mutate this then needs to copy first!
		filesList = Arrays.asList(files);

		if (sort != 0) {
			// sort the array in alpahbetic order
			Collections.sort(filesList, String.CASE_INSENSITIVE_ORDER);

			if (sort < 0) {
				// reverse the array content
				Collections.reverse(filesList);
			}
		}

		// return the list of the files in the given directory
		return filesList;
	}

	/**
	 * Copy a file's contents.
	 * @param srcFilePath  Full path to source file
	 * @param destFilePath    Full path to destination file
	 * @param overwriteExisting if true, toFile will be deleted before the copy
	 * @return true if OK, false if couldn't copy.
	 */
	public static boolean copyFile(String srcFilePath, String destFilePath, boolean overwriteExisting) {
		Log.i(TAG, "Copying file from: " + srcFilePath + "to: " + destFilePath);
		try {
			// check if the source file path exists
			File srcFile = new File(srcFilePath);

			if (!srcFile.exists()) {
				// the source file doesnt exists which means we dont have where
				// to read the content from
				Log.e(true, TAG, "Source file doesn't exist");
				return false;
			}

			// check if the destination file exists
			File destFile = new File(destFilePath);

			if(destFile.exists()) {
				if(overwriteExisting){
					Log.v(TAG, "Source file exists, deleting");
					// if exists override the file.
					destFile.delete();
				}
				else{
					Log.w(true, TAG, "Source file exists, and not deleting");
				}
			}

			// copy to source file content to the destination file.
			return copyFile(srcFile, destFile);

		} catch (SecurityException e) {
			// got an exception the operation failed.
			Log.e(true, TAG, "Could not copy file: ", e);
			return false;
		}
	}

	/**
	 * Copy a file's contents.
	 * @param srcFile  Full path to source file; should not be open.
	 * @param destFile    Full path to destination file; should not be open.
	 * @return true if OK, false if couldn't copy.
	 */
	@SuppressWarnings("resource")
	public static boolean copyFile(File srcFile, File destFile) {
		// source file input channel
		FileChannel in = null;
		// destination file output channel
		FileChannel out = null;

		Log.i(TAG, "Copying file from: " + srcFile.getAbsolutePath() + "to: " + destFile.getAbsolutePath());

		try {
			// get the files channels
			in = new FileInputStream(srcFile).getChannel();
			out = new FileOutputStream(destFile).getChannel();

			// write the source file content to the destination file
			long size = in.size();
			MappedByteBuffer buf = in.map(FileChannel.MapMode.READ_ONLY, 0, size);

			out.write(buf);

			// close the channels
			if (in != null) {
				in.close();
			}

			if (out != null) {
				out.close();
			}

			// operation was successful
			return true;

		} catch (IOException e) {
			Log.e(true, TAG, "Could not copy file: ", e);
			// channels cleanup
			try {
				if (in != null) {
					in.close();
				}
			} catch (IOException e2) {
				Log.e(true, TAG, "Could not close in channel: ", e2);
			}

			try {
				if (out != null) {
					out.close();
				}
			} catch (IOException e2) {
				Log.e(true, TAG, "Could not close out channel: ", e2);
			}

			// got exception, operation failed.
			return false;
		}
	}

	/**
	 * Saves a string data into a file.
	 * @param destFile Destination file full path to save the data to.
	 * @param data String that contains the data to save in the file.
	 * @return true if operation succeeded false otherwise.
	 */
	public static boolean saveDataToFile(File destFile, String data) {
		Log.v(TAG, "saving data to file: " + destFile.getAbsolutePath() + ". Data is: " + data);
		BufferedOutputStream outputStreamWriter = null;
		
		// check if the destination file exists
		try {			
			if (!destFile.exists()) {
				Log.v(TAG, "Creating file");
				// file does not exists, create a new one
				destFile.createNewFile();
			}

			// write the string content into the file
			outputStreamWriter = new BufferedOutputStream(new FileOutputStream(destFile));
			outputStreamWriter.write(data.getBytes("UTF-8"));
			outputStreamWriter.flush();
			outputStreamWriter.close();
		} catch (FileNotFoundException e) {
			// got exception, operation failed
			Log.e(true, TAG, "Could not save data to file cause it was not found: ", e);
			return false;
		} catch (IOException e) {
			// clean up
			Log.e(true, TAG, "Could not save data to file.", e);
			if (null != outputStreamWriter) {
				try {
					outputStreamWriter.close();
				} catch (IOException e1) {
					Log.e(true, TAG, "Could not close output stream", e1);
				}
			}
			// got exception, operation failed
			return false;
		}

		// operation was successful
		return true;
	}

	/**
	 * Reads a file and returns his content as string.
	 * @param file Full path to the file.
	 * @return String contains the file content, null if the function failed.
	 */
	public static String getFileContent(File file) {
		// streams that used to read the file content
		FileInputStream fis = null;
		InputStreamReader isr = null;
		BufferedReader bufferedReader = null;

		try {
			// read the file content, line after line
			fis = new FileInputStream(file);
			isr = new InputStreamReader(fis);
			bufferedReader = new BufferedReader(isr);

			StringBuilder fileContent = new StringBuilder();
			String line;
			while ((line = bufferedReader.readLine()) != null) {
				fileContent.append(line);
			}

			// cleanup
			fis.close();
			isr.close();
			bufferedReader.close();

			// return the file content as string
			return fileContent.toString();

		} catch (IOException e) {
			Log.e(true, TAG, "Could not get file content: " + file.getAbsolutePath(), e);
			// cleanup
			if (null != fis) {
				try {
					fis.close();
				} catch (IOException e1) {
					Log.e(true, TAG, "Could not close input stream", e1);
				}
			}

			if (null != isr) {
				try {
					isr.close();
				} catch (IOException e1) {
					Log.e(true, TAG, "Could not close input stream reader", e1);
				}
			}

			if (null != bufferedReader) {
				try {
					bufferedReader.close();
				} catch (IOException e1) {
					Log.e(true, TAG, "Could not close buffered reader", e1);
				}
			}
			// operation failed, return null
			return null;
		}		
	}
}