package com.aniways.service.helper;

import java.io.File;
import java.io.IOException;

import com.aniways.Log;
import com.aniways.Utils;
import com.aniways.data.AniwaysPhraseReplacementData;
import com.aniways.data.AniwaysStorageManager;
import com.aniways.service.utils.AniwaysServiceUtils;
import android.content.Context;
import android.content.SharedPreferences;
import android.content.SharedPreferences.Editor;

/**
 * Used to interact and change values related to the keywords file
 */
public class ConfigHelper {
	
	// The class is designed by the singleton pattern.
	// This is the static instance of the class, that the user gets
	// when he will use the class.
	private static ConfigHelper mInstance;
	private static final String TAG = "AniwaysConfigHelper";
	
	/**
	 * Class default constructor, private because of the singleton usage pattern.
	 */
	private ConfigHelper() {
		
	}
	
	/**
	 * Get an instance of the KeywordsHelper class.
	 * @return The instance of the KeywordsHelper class.
	 */
	public static ConfigHelper getInstance() {
		// if the instance doesn't exists, create it
		if (null == mInstance) {
			mInstance = new ConfigHelper();
		}
		
		// return the instance
		return mInstance;
	}
	
	/**
	 * Updates the keywords version in the shared preferences.
	 * @param context Application context.
	 * @param version New version string.
	 */
	public synchronized void setConfigVersion(Context context, String version, String eTag) {
		Log.i(TAG, "Setting config version to: " + version);
		
		// open the shared preferences
		SharedPreferences prefs = 
				context.getSharedPreferences(AniwaysServiceUtils.SHARED_PREFERENCES, Utils.getSharedPreferencesFlags());
		Editor edit = prefs.edit();
		// update the new version
		edit.putString(AniwaysServiceUtils.KEY_CONFIG_VERSION, version);
		edit.putString(AniwaysServiceUtils.KEY_CONFIG_ETAG, eTag);
		edit.commit();
	}

	/**
	 * Retrieve the keywords file version from the shared preferences.
	 * @param context Application context.
	 * @return The keywords file version as string.
	 */
	public synchronized String getConfigVersion(Context context) {
		// open the shared preferences.
		SharedPreferences prefs = 
				context.getSharedPreferences(AniwaysServiceUtils.SHARED_PREFERENCES, Utils.getSharedPreferencesFlags());

		return prefs.getString(AniwaysServiceUtils.KEY_CONFIG_VERSION, AniwaysPhraseReplacementData.EMPTY_PARSER_VERSION);
	}
	
	public synchronized String getConfigEtag(Context context) {
		// open the shared preferences.
		SharedPreferences prefs = 
				context.getSharedPreferences(AniwaysServiceUtils.SHARED_PREFERENCES, Utils.getSharedPreferencesFlags());

		return prefs.getString(AniwaysServiceUtils.KEY_CONFIG_ETAG, "");
	}
	
	/**
	 * Update the new downloaded keywords file to be the one that the sdk uses.
	 * @param context Application context.
	 * @return true is the update succeeded, false otherwise.
	 * @throws java.io.IOException
	 */
	public synchronized boolean upadateConfigFile(Context context, AniwaysStorageManager storageManager) throws IOException {
		// Copy the newly downloaded config file from the cache dir to the internal storage
		// where it will be used by the app.
		
		String source = AniwaysStorageManager.getConfigCacheDirPath(context) + "/" + AniwaysStorageManager.getConfigFileName();
		String dest = AniwaysStorageManager.getConfigFilePath(context);
		
		File configDirFile = new File(AniwaysStorageManager.getConfigDirPath(context));
		if (!configDirFile.exists()) {
			if(!configDirFile.mkdirs()){
				throw new IOException("Could not create config dir: " + configDirFile.getAbsolutePath());
			}
			Log.i(TAG, "Created config dir: " + configDirFile.getAbsolutePath());
		}
		
		Log.i(TAG, "Copying new config file from " + source + " to: " + dest);
		
		return new File(source).renameTo(new File(dest));
		
	}
	
}
