package com.aniways.service;

import com.aniways.data.AniwaysPrivateConfig;
import com.aniways.data.AniwaysStatics;
import com.aniways.service.utils.AniwaysServiceUtils;

import android.app.AlarmManager;
import android.app.IntentService;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
//import android.os.PowerManager;
//import android.os.PowerManager.WakeLock;

import com.aniways.Log;

/**
 * IntentService that responsible for keeping the device awake executing its work.
 */
abstract public class WakefulIntentService extends IntentService {
	// class tag for debug purposes
	private static final String TAG = "AniwaysWakefulIntentService";
	private static final int SECOND = 1000;
	private static final int MINUTE = 60 * SECOND;
	private static final int HOUR = 60 * MINUTE;
	private static final int DAY = 24 * HOUR;

	// Wake lock object keep the device awake when executing work
	//private static volatile WakeLock lockStatic = null;

	/**
	 * WakefulIntentService class constructor.
	 */
	public WakefulIntentService(String name) {
		super(name);

		// make the process restart itself when the intent is redelivered,
		// if it dies before the onHandleIntent event.
		// TODO: this requires API level 5!!
		setIntentRedelivery(true);
	}

	@Override
	public int onStartCommand(Intent intent, int flags, int startId) {
		try{
			// acquire WakeLock to keep the device awake 
			//WakeLock wakeLock = getLock(this.getApplicationContext());
			//if (!wakeLock.isHeld() || (flags & START_FLAG_REDELIVERY) != 0) {
			//	wakeLock.acquire();
			//}
		}
		catch(Throwable ex){
			Log.e(true, TAG, "Caught Exception in onStartCommand", ex);
		}


		super.onStartCommand(intent, flags, startId);

		return START_REDELIVER_INTENT;
	}

	@Override
	public void onCreate(){
		super.onCreate();

		try{
			// Init Aniways
			AniwaysStatics.init(this.getApplicationContext(), this, true);
		}
		catch(Throwable e) {
			// TODO: Maybe let the Exception crash the process..
			Log.e(true, TAG, "Error creating service", e);
		}
	}

	@Override
	final protected void onHandleIntent(Intent intent) {
		// execute the actual work
		try {
			doWakefulWork(intent);
		}
		catch(Throwable e) {
			Log.e(true, TAG, "Error handling intent", e);
		}
		finally {
			// release the WakeLock after handling the intent so the device could fall asleep again
			//WakeLock wakeLock = getLock(this.getApplicationContext());

			//if (wakeLock.isHeld()) {
			//	wakeLock.release();
			//}
		}
	}

	/**
	 * The actual method that needs to be implemented for work execution.
	 * Eventually This function will be called when the alarm goes off, and the required work
	 * will be executed by this function.
	 * @param intent The intent supplied by the sendWakefulWork method.
	 */
	protected abstract void doWakefulWork(Intent intent);

	/**
	 * Acquire a WakeLock to keep the device awake.
	 * @param context Application context.
	 * @return WakeLock object for keeping the device awake.
	 */
	//synchronized private static WakeLock getLock(Context context) {
	//	// if the wake lock doesn't exists create a new one.
	//	if (null == lockStatic) {
	//		PowerManager powerManager = 
	//				(PowerManager) context.getSystemService(Context.POWER_SERVICE);
    //
	//		lockStatic = powerManager.newWakeLock(PowerManager.PARTIAL_WAKE_LOCK, TAG);
	//		lockStatic.setReferenceCounted(true);
	//	}
    //
	//	// return the WakeLock object.
	//	return lockStatic;
	//}

	/**
	 * Start the service to execute work.
	 * @param context Application context.
	 * @param Intnet to start the appropriate service.
	 */
	public static void sendWakefulWork(Context context, Intent intent) {
		//getLock(context.getApplicationContext()).acquire();

		try{
			context.startService(intent);
		}

		finally{

			// Release the WakeLock after handling the intent so the device could fall asleep again
			//WakeLock wakeLock = getLock(context.getApplicationContext());

			//if (wakeLock.isHeld()) {
			//	wakeLock.release();
			//}
		}

	}

	/**
	 * Start the service to execute work.
	 * @param context Application context.
	 * @param classService The class of the appropriate service to start.
	 * @param appId 
	 */
	public static void sendWakefulWork(Context context, Class<?> classService, boolean fromAlarm) {
		Log.i(TAG, "Starting request. From alarm=" + fromAlarm);
		Intent intent = new Intent(context, classService);
		if(!fromAlarm){
			intent.setAction(AniwaysServiceUtils.ACTION_SYNC_WITH_SERVER);
		}
		sendWakefulWork(context, intent);
	}

	/**
	 * Schedule alarms for service work execution.
	 * @param listener Alarm listener used to schedule the alarms.
	 * @param context Application context.
	 * @param force Should force to schedule the alarms.
	 */
	public static void scheduleAlarms(AlarmListener listener, Context context) {
		// Get when the last alarm executed the work 
		//SharedPreferences prefs = 
		//		context.getSharedPreferences(AniwaysServiceUtils.SHARED_PREFERENCES, Utils.getSharedPreferencesFlags());
		//long lastAlarm = prefs.getLong(AniwaysServiceUtils.KEY_LAST_ALARM, 0);

		//Log.v(TAG, "Alarm scheduling requested. Last alarm was: " + lastAlarm + ". Current time is: " + System.currentTimeMillis());

		// Check if needs to schedule the alarm.
		//if (lastAlarm == 0 || force || 
		//		(System.currentTimeMillis() > lastAlarm && 
		//				System.currentTimeMillis() - lastAlarm > listener.getMaxAge())) {

		// Schedule the alarm to execute work
		AlarmManager alarmManager = 
				(AlarmManager) context.getSystemService(Context.ALARM_SERVICE);
		Intent intent = new Intent(context, AlarmReceiver.class);
		// The second parameter is the request code which identifies the alarm. Since we always use
		// the same one then new alarms will override the old ones..
		PendingIntent pendingIntent = PendingIntent.getBroadcast(context, 0, intent, 0);
		
		Intent heartBeatIntent = new Intent(context, AlarmReceiver.class);
		heartBeatIntent.setAction(AlarmReceiver.ANIWAYS_HEARTBEAT_ACTION);
		// The second parameter is the request code which identifies the alarm. Since we always use
		// the same one then new alarms will override the old ones..
		PendingIntent heartBeatPendingIntent = PendingIntent.getBroadcast(context, 0, heartBeatIntent, 0);

		Log.i(TAG, "Scheduling alarms");
		long now = System.currentTimeMillis();
		long appInstallTime = AniwaysServiceUtils.getAppInstallTime(context, now);
		// If this is the first time, perform the action now (and set the first alarm to a day from now). 
		// Otherwise, we calculate the time until the next full day from the
		// install time and then we perform the action..
		long timeUntilNextFullDay = AlarmManager.INTERVAL_DAY;
		if(appInstallTime == now){
			AlarmReceiver.sendAppInstalledEvent(context, false);
		}
		else{
			long timeFromInstall = now - appInstallTime;
			long timeFromInstallAfterFullDays = timeFromInstall % DAY;
			timeUntilNextFullDay = DAY - timeFromInstallAfterFullDays;
		}
		
		// Start the sync service right now if needs to run in order not to create an alarm for 7 seconds from now.
		// Instead, let the first alarm be <interval time> from now..
		AniwaysAlarmListener alarmListener = new AniwaysAlarmListener();
		alarmListener.sendWakefulWork(context, false);
		listener.scheduleAlarms(alarmManager, pendingIntent, AniwaysPrivateConfig.getInstance().syncAlarmScheduleInterval, AniwaysPrivateConfig.getInstance().syncAlarmScheduleInterval, context, "check for updates from server");
		
		// Schedule the next heartbeat alarm, and the ones to follow it..
		listener.scheduleAlarms(alarmManager, heartBeatPendingIntent, AlarmManager.INTERVAL_DAY, timeUntilNextFullDay, context, "send app install time event");
		
		//}
		//else{
		//	Log.i(TAG, "Not scheduling alarm. Time from last alarm is too soon and force is false ");
		//}
	}

	///**
	// * Cancel the alarm related to the AlarmReceiver class, in order to prevent it from executing future work.
	// * @param context Application context.
	// */
	//public static void cancelAlarms(Context context) {
	//	Log.i(TAG, "Cancel Alarms called");
	//	// Get the pending intent related to the AlarmReceiver class
	//	AlarmManager alarmManager = 
	//			(AlarmManager) context.getSystemService(Context.ALARM_SERVICE);
	//	Intent intent = new Intent(context, AlarmReceiver.class);
	//	PendingIntent pendingIntent = PendingIntent.getBroadcast(context, 0, intent, 0);
	//
	//	// Cancel the alarm
	//	alarmManager.cancel(pendingIntent);
	//}

}