package com.aniways.emoticons.button;

import android.content.Context;
import android.support.v4.view.AniwaysDirectionalViewPager;
import android.view.KeyEvent;
import android.view.LayoutInflater;
import android.view.MotionEvent;
import android.view.View;
import android.widget.ImageView;

import com.aniways.AniwaysEditText;
import com.aniways.AssetType;
import com.aniways.IconData;
import com.aniways.Log;
import com.aniways.R;
import com.aniways.analytics.NonThrowingRunnable;
import com.aniways.data.AniwaysLockedIconHelper;
import com.aniways.emoticons.button.AniwaysEmoticonsButtonMaker.LastPagesLocation;
import com.aniways.viewpagerindicator.TabPageIndicator;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

public class EmoticonAssetsPagerAdapter extends EmoticonsOnDemandPagerAdapterBase
{
    private static final String TAG = "AniwaysEmoticonAssetsPagerAdapter";

    private static final long DELETE_DELAY_MS = 100;

    private AssetType[] mAssetTypesInOrder;
    private AniwaysLockedIconHelper mLockedIconHelper;
    private LastPagesLocation mLastPagesLocation;
    private List<IconData> mIconsForGenius = null;
    private Context mContext;
    private AniwaysEditText mEditText;
    private Runnable mDeleteRunnable;

    public EmoticonAssetsPagerAdapter(AniwaysLockedIconHelper lockedIconHelper, LastPagesLocation lastPagesLocation, Context context, List<IconData> iconsForGenius, AniwaysEditText editText)
    {
        super(TAG, AdapterType.Asset, 0, 0, 0, null, -1);

        try
        {
            this.mLockedIconHelper = lockedIconHelper;
            this.mLastPagesLocation = lastPagesLocation;
            this.mContext = context;
            this.mAssetTypesInOrder = mLockedIconHelper.getDataParser().getAssetTypesInOrder();
            if (mAssetTypesInOrder == null || mAssetTypesInOrder.length == 0)
            {
                Log.e(true, TAG, "No asset types in parser");
            }
            this.mIconsForGenius = iconsForGenius == null ? new ArrayList<IconData>() : iconsForGenius;
            this.mEditText = editText;
        }
        catch (Throwable ex)
        {
            Log.e(true, TAG, "Caught Exception in constructor", ex);
        }
    }

    @Override
    public int getCount()
    {
        if (mAssetTypesInOrder == null)
        {
            return 0;
        }
        return mAssetTypesInOrder.length;
    }

    public AssetType getAssetType(int position)
    {
        try
        {
            return mAssetTypesInOrder[position];
        }
        catch (Throwable ex)
        {
            Log.e(true, TAG, "Caught Exception in getAssetName", ex);
            return AssetType.Unknown;
        }
    }

    @Override
    protected View instantiateItem(View collection)
    {
        try
        {
            LayoutInflater inflater = (LayoutInflater)collection.getContext().getSystemService(Context.LAYOUT_INFLATER_SERVICE);
            return inflater.inflate(R.layout.aniways_emoticons_button_popup_icon_categories_pager, null);
        }
        catch (Throwable ex)
        {
            Log.e(true, TAG, "Caught Exception in instantiateItem", ex);
            return null;
        }
    }

    @Override
    protected void instantiateItemInternal(InstantiatedViewHolder holder)
    {

        View layout = holder.view;

        final AssetType assetType = this.getAssetType(holder.position);

        AniwaysDirectionalViewPager pager = (AniwaysDirectionalViewPager)layout.findViewById(R.id.aniways_ebp_icon_categories_pager);
        pager.setOffscreenPageLimit(1); //TODO: 1 is the minimum. 1. Make this value configurable. See if can somehow change this minimum
        holder.pager = pager;

        final List<IconData> filteredGeniusIcons = new ArrayList<>();

        for (IconData icon : mIconsForGenius)
        {
            if (icon.assetType == assetType)
            {
                filteredGeniusIcons.add(icon);
            }
        }


        final EmoticonCategoriesPagerAdapter adapter = new EmoticonCategoriesPagerAdapter(
                mLockedIconHelper,
                mLastPagesLocation,
                assetType,
                mContext,
                filteredGeniusIcons,
                this);
        pager.setAdapter(adapter);
        pager.setOrientation(AniwaysDirectionalViewPager.HORIZONTAL);

        TabPageIndicator indicator = (TabPageIndicator)layout.findViewById(R.id.aniways_ebp_icon_categories_pager_indicator);
        indicator.setViewPager(pager);

        indicator.setOnPageChangeListener(new AniwaysDirectionalViewPager.OnPageChangeListener()
        {

            @Override
            public void onPageScrollStateChanged(int arg0)
            {
                // TODO Auto-generated method stub
            }

            @Override
            public void onPageScrolled(int arg0, float arg1, int arg2)
            {
                // TODO Auto-generated method stub
            }

            @Override
            public void onPageSelected(int categoryPage)
            {
                try
                {
                    mLastPagesLocation.setCategoryPage(categoryPage);
                    Log.i(TAG, "Selected category num: " + categoryPage);
                }
                catch (Throwable ex)
                {
                    Log.e(true, TAG, "Caught Exception in onPageSelected", ex);
                }
            }

        });

        if (mLastPagesLocation.getSelectedAssetType() == mAssetTypesInOrder[holder.position])
        {
            indicator.setCurrentItem(mLastPagesLocation.getCategoryPage());

            // If user was already at a certain location in the button (which is not the recent), and its the same keywords version (so there are the same icons)
            // then return to that location..
            // If the user was in the recent icons then it is not relevant to return there because it will go to the recent anyway, and
            // the number of pages is also determined by the genius icons so it is not guaranteed that the same page will be displayed if
            // trying to return to it..
            if (!mLastPagesLocation.isCurrentCategoryTheRecentsCategory() && adapter.getCount() > mLastPagesLocation.getCategoryPage())
            {
                // It is needed to fire here, because setting the current view to the saved one will not
                // trigger it automatically
                mLastPagesLocation.reportScreenViewEvent();
                indicator.setCurrentItem(mLastPagesLocation.getCategoryPage());
            }
            else
            {
                // If recent icons are empty, and there are no genius icons then start with the next page
                // This line must be called !!after!! the adapter and indicator are set
                if (adapter.getCount() > 1 && AniwaysRecentIconsManager.isEmpty(assetType) && filteredGeniusIcons.size() == 0)
                {
                    indicator.setCurrentItem(1);
                }
                else if (adapter.getCount() > 0 && adapter.getCategoryName(0).equalsIgnoreCase(EmoticonCategoriesPagerAdapter.RECENT))
                {
                    // Start with the recent, even if user finished in a different page before
                    // It is needed to fire here, because setting the current view to 0 will not trigger it..
                    mLastPagesLocation.reportScreenViewEvent();
                    indicator.setCurrentItem(0);
                }
            }
        }
        else
        {
            // If recent icons are empty, and there are no genius icons then start with the next page
            // This line must be called !!after!! the adapter and indicator are set
            if (adapter.getCount() > 1 && AniwaysRecentIconsManager.isEmpty(assetType) && filteredGeniusIcons.size() == 0)
            {
                indicator.setCurrentItem(1);
            }
            else if (adapter.getCount() > 0 && adapter.getCategoryName(0).equalsIgnoreCase(EmoticonCategoriesPagerAdapter.RECENT))
            {
                // Start with the recent, even if user finished in a different page before
                // It is needed to fire here, because setting the current view to 0 will not trigger it..
                mLastPagesLocation.reportScreenViewEvent();
                indicator.setCurrentItem(0);
            }
        }

        // Prepare the 'delete' button
        if (mEditText != null)
        {
            ImageView backSpace = (ImageView)layout.findViewById(R.id.aniways_ebp_backspace_button);
            setupBackspace(backSpace);
        }

        holder.instantiated = true;
    }

    @Override
    protected void onHolderCreated(InstantiatedViewHolder holder)
    {
        // Intentionally empty, in order not to create other tabs since it degrades performance
    }

    @Override
    public CharSequence getPageTitle(int position)
    {
        return mAssetTypesInOrder[position].toString().toUpperCase(Locale.getDefault());
    }


    private void setupBackspace(final ImageView backSpace)
    {

        // Will continue deleting as long as the 'delete' button is pressed
        mDeleteRunnable = new NonThrowingRunnable(TAG, "while deleting in long press", "")
        {
            @Override
            public void innerRun()
            {
                if (backSpace.isPressed())
                {
                    KeyEvent event = new KeyEvent(0, 0, 0, KeyEvent.KEYCODE_DEL, 0, 0, 0, 0, KeyEvent.KEYCODE_ENDCALL);
                    mEditText.dispatchKeyEvent(event);

                    backSpace.postDelayed(mDeleteRunnable, DELETE_DELAY_MS);
                }
            }
        };


        // After a long click we want to continue deleting while the button is pressed..
        backSpace.setOnLongClickListener(new View.OnLongClickListener()
        {

            @Override
            public boolean onLongClick(View v)
            {
                try
                {
                    backSpace.postDelayed(mDeleteRunnable, DELETE_DELAY_MS);
                }
                catch (Throwable ex)
                {
                    Log.e(true, TAG, "Caught Exception in long click of backspace", ex);
                }

                return false;
            }

        });

        // Let the first down event also delete a character, so we don't have to wait until the long click is triggered
        backSpace.setOnTouchListener(new View.OnTouchListener()
        {

            @Override
            public boolean onTouch(View v, MotionEvent event)
            {

                try
                {
                    if (event.getAction() == MotionEvent.ACTION_DOWN)
                    {
                        KeyEvent newEvent = new KeyEvent(event.getDownTime(), event.getEventTime(), event.getAction(),
                                                         KeyEvent.KEYCODE_DEL, 0);

                        mEditText.dispatchKeyEvent(newEvent);
                    }
                }
                catch (Throwable ex)
                {
                    Log.e(true, TAG, "Caught Exception in backspace onTouch", ex);
                }

                return false;
            }

        });
    }
}