package com.aniways.data;

import android.support.annotation.NonNull;
import android.text.TextUtils;

import com.aniways.Log;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.LinkedList;
import java.util.List;

class GiphyAssetInfoBuilder extends WebAssetInfoBuilderBase implements IAssetInfoBuilder {

    private static final String TAG = "GiphyAssetInfoBuilder";
    private String apiKey;
    private String apiSearchTermUrlFormat;
    private String apiSearchByIdUrlFormat;
    private int limit;
    private int offset;
    private final static Phrase phrase = new Phrase("gif", null);
    private String rating;

    public GiphyAssetInfoBuilder() {
        super(AnimatedGifsProviders.GIPHY.getStringRepresentation());

        this.apiKey = AniwaysClientConfig.getInstance().getResourceString("aniways_giphy_api_key");
        this.apiSearchTermUrlFormat = AniwaysClientConfig.getInstance()
                                                         .getResourceString("aniways_giphy_api_search_term_url_format");
        this.apiSearchByIdUrlFormat = AniwaysClientConfig.getInstance()
                                                         .getResourceString("aniways_giphy_api_search_by_id_url_format");
        this.limit = AniwaysPrivateConfig.getInstance().animatedGifsRequestSize;
        this.rating = AniwaysPrivateConfig.getInstance().animatedGifsRating;
        this.offset = 0; //currently no paging is available
    }

    @Override
    protected String getSearchUrlByResourceId(String resourceId) {
        return String.format(apiSearchByIdUrlFormat, resourceId, apiKey);
    }

    @Override
    protected String getSearchUrlBySearchTerm(String resource){
        String searchTerm = resource.split(":")[1];
        searchTerm = searchTerm.replace(' ', '+');
        String requestUrl = String.format(apiSearchTermUrlFormat, searchTerm, apiKey, limit, offset);
        if (!TextUtils.isEmpty(rating)) {
            requestUrl += "&rating=" + rating;
        }

        return requestUrl;
    }

    @Override
    protected IAniwaysAssetInfo parseResourceIdSearchResponse(@NonNull JSONObject response) {
        try {
            JSONObject itemData = response.getJSONObject("data");

            return parseSingleItem(itemData);
        }
        catch (JSONException e) {
            Log.e(true,TAG,"Unable to parse Giphy api response:" + response.toString(),e);
            return null;
        }
    }

    @Override
    protected List<IAniwaysAssetInfo> parseTermSearchResponse(@NonNull JSONObject response) {
        try {
            JSONArray array = response.getJSONArray("data");

            List<IAniwaysAssetInfo> retItems = new LinkedList<>();

            for (int i = 0; i < array.length(); i++) {
                JSONObject itemData = array.getJSONObject(i);

                retItems.add(parseSingleItem(itemData));
            }

            return retItems;
        }
        catch (JSONException e) {
            Log.e(true, TAG, "Unable to parse Giphy api response:" + response.toString(), e);
            return null;
        }
    }

    private IAniwaysAssetInfo parseSingleItem(JSONObject itemData) throws JSONException {
        JSONObject images = itemData.getJSONObject("images");
        JSONObject highQualityObj = images.getJSONObject(AniwaysPrivateConfig.getInstance().giphyHighQualityImageObjectName);
        JSONObject lowQualityObj = images.getJSONObject(AniwaysPrivateConfig.getInstance().giphyLowQualityImageObjectName);
        JSONObject stillObj = images.getJSONObject(AniwaysPrivateConfig.getInstance().giphyHighQualityStillImageObjectName);

        String id = itemData.getString("id");
        String stillUrl = stillObj.getString("url");
        String highQualityAnimationUrl = highQualityObj.getString("url");
        String lowQualityAnimationUrl = lowQualityObj.getString("url");
        AniwaysAnimatedAssetInfo result = new AniwaysAnimatedAssetInfo(new GiphyIconData(id, stillUrl, lowQualityAnimationUrl, highQualityAnimationUrl, phrase));
        result.setId(id);

        result.setHighQualityAnimatedGifUrl(highQualityAnimationUrl);
        result.setLowQualityAnimatedGifUrl(lowQualityAnimationUrl);

        result.setHighHeight(highQualityObj.getInt("height"));
        result.setLowHeight(lowQualityObj.getInt("height"));

        result.setHighWidth(highQualityObj.getInt("width"));
        result.setLowWidth(lowQualityObj.getInt("width"));

        result.setStillUrl(stillUrl);
        result.setStillHeight(stillObj.getInt("height"));
        result.setStillWidth(stillObj.getInt("width"));

        result.setJson(itemData);

        return result;
    }
}