package com.aniways.analytics.request.azure;

import java.net.HttpURLConnection;
import java.security.InvalidKeyException;

import android.text.TextUtils;

import com.aniways.data.AniwaysPrivateConfig;
import com.aniways.data.Installation;

public final class AzureRequest {

    /**
     * Stores the user agent to send over the wire to identify the client.
     */
    private static String userAgent;

    public static void addAzureHeaders(HttpURLConnection connection, long contentLength, String date) throws InvalidKeyException {
    	
    	// We already have content length, type and encoding set.
    	
        // Note : by default sends Accept behavior as text/html, image/gif, image/jpeg, *; q=.2, */*; q=.2
        connection.setRequestProperty(Constants.HeaderConstants.ACCEPT, Constants.HeaderConstants.XML_TYPE);
        connection.setRequestProperty(Constants.HeaderConstants.ACCEPT_CHARSET, Constants.UTF8_CHARSET);

        // Note: by default sends Accept-Encoding as gzip
        connection.setRequestProperty(Constants.HeaderConstants.ACCEPT_ENCODING, Constants.EMPTY_STRING);

        // Note : by default sends Content-type behavior as application/x-www-form-urlencoded for posts.
        //connection.setRequestProperty(Constants.HeaderConstants.CONTENT_TYPE, Constants.EMPTY_STRING);
        
        // The type of the blob
        connection.setRequestProperty("x-ms-blob-type", "BlockBlob");
        // Used for correlating all requests to spesific clients, for monitoring purposes
        String installId = Installation.id();
        connection.setRequestProperty("x-ms-client-request-id", (installId == null ? "unknown" : installId));
        connection.setRequestProperty(Constants.HeaderConstants.STORAGE_VERSION_HEADER, Constants.HeaderConstants.TARGET_STORAGE_VERSION);
        connection.setRequestProperty(Constants.HeaderConstants.USER_AGENT, getUserAgent());
        
        Credentials creds = new Credentials(AniwaysPrivateConfig.getInstance().azureAnalyticsStorageAccountName, AniwaysPrivateConfig.getInstance().azureAnalyticsKey);
        
        signRequestForBlobAndQueue(connection, creds, contentLength, date);
    }


    /**
     * Gets the user agent to send over the wire to identify the client.
     * 
     * @return the user agent to send over the wire to identify the client.
     */
    private static String getUserAgent() {
        if (userAgent == null) {
            String userAgentComment = String.format(Utility.LOCALE_US, "(Android %s; %s; %s)",
                    android.os.Build.VERSION.RELEASE, android.os.Build.BRAND, android.os.Build.MODEL);
            userAgent = String.format("%s/%s %s", Constants.HeaderConstants.USER_AGENT_PREFIX,
                    Constants.HeaderConstants.USER_AGENT_VERSION, userAgentComment);
        }

        return userAgent;
    }



    /**
     * Signs the request appropriately to make it an authenticated request for Blob and Queue.
     * 
     * @param request
     *            a HttpURLConnection for the operation.
     * @param credentials
     *            the credentials to use for signing.
     * @param contentLength
     *            the length of the content written to the output stream, -1 if unknown.
     * @param opContext
     *            an object used to track the execution of the operation
     * @throws java.security.InvalidKeyException
     *             if the credentials key is invalid.
     */
    private static void signRequestForBlobAndQueue(final HttpURLConnection request, final Credentials credentials,
            final Long contentLength, String date) throws InvalidKeyException {
        request.setRequestProperty(Constants.HeaderConstants.DATE, (TextUtils.isEmpty(date) ? Utility.getGMTTime() : date));
        final Canonicalizer canonicalizer = BlobQueueFullCanonicalizer.insatnce;

        final String stringToSign = canonicalizer.canonicalize(request, credentials.accountName, contentLength);

        final String computedBase64Signature = StorageKey.computeMacSha256(credentials.key, stringToSign);

        request.setRequestProperty(Constants.HeaderConstants.AUTHORIZATION,
                String.format("%s %s:%s", "SharedKey", credentials.accountName, computedBase64Signature));
    }
}
