package com.aniways.analytics.db;

import com.aniways.Log;
import com.aniways.analytics.Constants;
import com.aniways.analytics.models.BasePayload;
import com.aniways.data.AniwaysPrivateConfig;
import com.aniways.data.Installation;

import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.concurrent.atomic.AtomicLong;

import android.annotation.SuppressLint;
import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.SQLException;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteException;
import android.database.sqlite.SQLiteOpenHelper;
import android.database.sqlite.SQLiteStatement;
import android.util.Pair;

public class PayloadDatabase extends SQLiteOpenHelper {

	//
	// Singleton
	//

	private static final String TAG = "AniwaysPayloadDatabase";
	private static PayloadDatabase instance;
	
	public static PayloadDatabase getInstance(Context context) {
		if (instance == null) {
			instance = new PayloadDatabase(context);
		}

		return instance;
	}

	//
	// Instance 
	//

	/**
	 * Caches the count of the database without requiring SQL count to be
	 * called every time. This will allow us to quickly determine whether
	 * our database is full and we shouldn't add anymore
	 */
	private AtomicLong count;
	private boolean initialCount;
	
	private AtomicLong errorCount;
	private boolean initialErrorsCount;

	private JsonPayloadSerializer serializer = new JsonPayloadSerializer();

	private PayloadDatabase(Context context) {
		super(context, getDbName(context), null, Constants.Database.VERSION);
		Log.d(TAG, "Created DB with name: " + getDbName(context));

		this.count = new AtomicLong();
		this.errorCount = new AtomicLong();
	}

	private static String getDbName(Context context){
		String databaseName = context.getPackageName();
		if(databaseName == null){
			databaseName = Installation.id(context);
		}

		return databaseName + ".AniwaysAnalyticsService";
	}

	@Override
	public void onCreate(SQLiteDatabase db) {
		try{
			createPayloadTable(db);
			
			createErrorsDb(db);
			
		}
		catch(Throwable ex){
			Log.eToGaOnly(true, TAG, "Caught Exception in onCreate", ex);
		}
	}

	private void createPayloadTable(SQLiteDatabase db) {
		String sql = String.format("CREATE TABLE IF NOT EXISTS %s (%s %s, %s %s, %s %s);",
				Constants.Database.PayloadTable.NAME,

				Constants.Database.PayloadTable.Fields.Id.NAME,
				Constants.Database.PayloadTable.Fields.Id.TYPE,

				Constants.Database.PayloadTable.Fields.Payload.NAME,
				Constants.Database.PayloadTable.Fields.Payload.TYPE,
				
				Constants.Database.PayloadTable.Fields.PayloadHash.NAME,
				Constants.Database.PayloadTable.Fields.PayloadHash.TYPE);
		try {
			db.execSQL(sql);
		} catch (SQLException e) {
			Log.eToGaOnly(true, TAG, "Failed to create Segment.io SQL lite database: " + sql, e);
		}
	}

	@Override
	public void onOpen(SQLiteDatabase db) {
		try{
			super.onOpen(db);
		}
		catch(Throwable ex){
			Log.eToGaOnly(true, TAG, "Caught Exception in onOpen", ex);
		}
	}

	/**
	 * Counts the size of the current database and sets the cached counter
	 * 
	 * This shouldn't be called onOpen() or onCreate() because it will cause
	 * a recursive database get.
	 */
	private void ensureCount() {
		if (!initialCount) {
			count.set(countRows(Constants.Database.PayloadTable.NAME));
			initialCount = true;
		}
	}
	
	private void ensureErrorsCount() {
		if (!initialErrorsCount) {
			errorCount.set(countRows(Constants.Database.ErrorsPayloadTable.NAME));
			initialErrorsCount = true;
		}
	}

	@Override
	public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
		Log.i(TAG, "Upgrading DB to version: " + newVersion + " . From version: " + oldVersion);
		db.execSQL("DROP TABLE IF EXISTS " + Constants.Database.PayloadTable.NAME);
		db.execSQL("DROP TABLE IF EXISTS " + Constants.Database.ErrorsPayloadTable.NAME);
		onCreate(db);
	}

	/**
	 * Adds a payload to the database
	 * @param payload 
	 */
	boolean addPayload(String payload) {

		ensureCount();

		long rowCount = getRowCount();
		if (rowCount >= AniwaysPrivateConfig.getInstance().analyticsMaxEventsInDb) {
			Log.wToGaOnly(true, TAG, "Cant add payload, the database is larger than max queue size.", null);
			return false;
		}

		boolean success = false;

		synchronized (this) {

			SQLiteDatabase db = null;

			try {

				db = getWritableDatabase();
				ContentValues contentValues = new ContentValues();

				contentValues.put(
						Constants.Database.PayloadTable.Fields.Payload.NAME, 
						payload);
				
				int payloadHash = payload.hashCode();

				contentValues.put(
						Constants.Database.PayloadTable.Fields.PayloadHash.NAME,
						payloadHash);

				long result = db.insertWithOnConflict(Constants.Database.PayloadTable.NAME, null,
						contentValues, SQLiteDatabase.CONFLICT_IGNORE);

				if (result == -1) {
					String sql = "SELECT COUNT(*) FROM " + Constants.Database.PayloadTable.NAME + " WHERE ? = ?";
					String[] selectionArgs = {Constants.Database.PayloadTable.Fields.PayloadHash.NAME, String.valueOf(payloadHash)};
					Cursor cursor = db.rawQuery(sql , selectionArgs );
					if(cursor.getCount() == 0) {
						Log.eToGaOnly(true, TAG, "Database insert failed. Result: " + result, null);
					}
					cursor.close();
				} else {
					success = true;
					// increase the row count
					count.addAndGet(1);
				}

			} catch (SQLiteException e) {

				Log.eToGaOnly(true, TAG, "Failed to open or write to analytics payload db: ", e);

			} finally {
				if (db != null) db.close();
			}

			return success;
		}
	}
	
	boolean addErrorPayload(String payload) {

		ensureErrorsCount();

		long rowCount = getErrorsRowCount();
		if (rowCount >= AniwaysPrivateConfig.getInstance().analyticsMaxEventsInDb) {
			Log.wToGaOnly(true, TAG, "Cant add payload, the database is larger than max queue size.", null);
			return false;
		}

		boolean success = false;

		synchronized (this) {

			SQLiteDatabase db = null;

			try {

				db = getWritableDatabase();
				ContentValues contentValues = new ContentValues();

				contentValues.put(Constants.Database.ErrorsPayloadTable.Fields.Payload.NAME, payload);

				long result = db.insert(Constants.Database.ErrorsPayloadTable.NAME, null, contentValues);

				if (result == -1) {
					Log.eToGaOnly(true, TAG, "Errors Database insert failed. Result: " + result, null);
				} else {
					success = true;
					// increase the row count
					errorCount.addAndGet(1);
				}

			} catch (SQLiteException e) {

				Log.eToGaOnly(true, TAG, "Failed to open or write to errors payload db: ", e);

			} finally {
				if (db != null) db.close();
			}

			return success;
		}
	}
	
	private void createErrorsDb(SQLiteDatabase db) {
		String sql;
		sql = String.format("CREATE TABLE IF NOT EXISTS %s (%s %s, %s %s);",
				Constants.Database.ErrorsPayloadTable.NAME,

				Constants.Database.ErrorsPayloadTable.Fields.Id.NAME,
				Constants.Database.ErrorsPayloadTable.Fields.Id.TYPE,

				Constants.Database.ErrorsPayloadTable.Fields.Payload.NAME,
				Constants.Database.ErrorsPayloadTable.Fields.Payload.TYPE);
		try {
			db.execSQL(sql);
			Log.i(TAG, "Created errors table");
		} catch (SQLException e) {
			Log.eToGaOnly(true, TAG, "Failed to create SQL lite errors database: " + sql, e);
		}
	}

	/**
	 * Fetches the total amount of rows in the database
	 * @return
	 */
	private long countRows(String tableName) {

		String sql = String.format("SELECT COUNT(*) FROM %s", tableName);

		long numberRows = 0;	

		SQLiteDatabase db = null;

		synchronized(this) {

			try {
				db = getWritableDatabase();
				SQLiteStatement statement = db.compileStatement(sql);
				numberRows = statement.simpleQueryForLong();

			} catch (SQLiteException e) {
				Log.eToGaOnly(true, TAG, "Failed to ensure row count in the table: " + tableName, e);
			} finally {
				if (db != null) db.close();
			}
		}

		return numberRows;
	}
	

	/**
	 * Fetches the total amount of rows in the database without
	 * an actual database query, using a cached counter.
	 * @return
	 */
	public long getRowCount() {
		if (!initialCount) ensureCount();
		return count.get();
	}
	
	public long getErrorsRowCount() {
		if (!initialErrorsCount) ensureErrorsCount();
		return errorCount.get();
	}

	/**
	 * Get the next (limit) events from the database
	 * @param limit
	 * @return
	 */
	List<Pair<Long, BasePayload>> getEvents(int limit) {

		List<Pair<Long, BasePayload>> result = 
				new LinkedList<Pair<Long, BasePayload>>();

		SQLiteDatabase db = null;
		Cursor cursor = null;

		synchronized (this) {

			try {

				db = getWritableDatabase();

				String table = Constants.Database.PayloadTable.NAME;
				String[] columns = Constants.Database.PayloadTable.FIELD_NAMES;
				String selection = null;
				String selectionArgs[] = null;
				String groupBy = null;
				String having = null;
				String orderBy = Constants.Database.PayloadTable.Fields.Id.NAME + " ASC";
				String limitBy = "" + limit;

				cursor = db.query(table, columns, selection, selectionArgs, 
						groupBy, having, orderBy, limitBy);
				
				long maxMem = Runtime.getRuntime().maxMemory();
				long freeMem = Runtime.getRuntime().freeMemory();
				int maxJsonLength = (int) ((((maxMem - (10 * 1024 * 1024)) / 2) / 6));
				Log.d(TAG, "MaxMem: " + (maxMem / (1024 * 1024)) + " FreeMem: " + (freeMem / (1024 * 1024)) + " maxLength: " + maxJsonLength);
				int jsonLength = 0;
				int counter = 0;
				while (cursor.moveToNext()) {
					long id = cursor.getLong(0);
					String json = cursor.getString(1);
					jsonLength += json.length();
					counter++;
					
					BasePayload payload = serializer.deseralize(json);

					if (payload != null) 
						result.add(new Pair<Long, BasePayload>(id, payload));
					
					if(jsonLength > maxJsonLength){
						Log.d(TAG, "Reached max mem after " + counter + " events");
						AniwaysPrivateConfig.getInstance().analyticsMaxEventsBeforeFlush = counter / 2;
						break;
					}
				}

			} catch (SQLiteException e) {

				Log.eToGaOnly(true, TAG, "Failed to open or read from the Segment.io payload db: ", e);

			} finally {
				if (cursor != null) cursor.close();
				if (db != null) db.close();
			}
		}

		return result;
	}
	
	List<Pair<Long, String>> getErrorEvents(int limit) {

		List<Pair<Long, String>> result = new LinkedList<Pair<Long, String>>();

		SQLiteDatabase db = null;
		Cursor cursor = null;

		synchronized (this) {

			try {

				db = getWritableDatabase();

				String table = Constants.Database.ErrorsPayloadTable.NAME;
				String[] columns = Constants.Database.ErrorsPayloadTable.FIELD_NAMES;
				String selection = null;
				String selectionArgs[] = null;
				String groupBy = null;
				String having = null;
				String orderBy = Constants.Database.ErrorsPayloadTable.Fields.Id.NAME + " ASC";
				String limitBy = "" + limit;

				cursor = db.query(table, columns, selection, selectionArgs, 
						groupBy, having, orderBy, limitBy);

				while (cursor.moveToNext()) {
					long id = cursor.getLong(0);
					String xml = cursor.getString(1);

					//BasePayload payload = serializer.deseralize(json);

					if (xml != null) 
						result.add(new Pair<Long, String>(id, xml));
				}

			} catch (SQLiteException e) {

				Log.eToGaOnly(true, TAG, "Failed to open or read from the errors payload db: ", e);

			} finally {
				if (cursor != null) cursor.close();
				if (db != null) db.close();
			}
		}

		return result;
	}

	/**
	 * Remove these events from the database
	 * @param minId
	 * @param maxId
	 */
	@SuppressLint("DefaultLocale")
	int removeEvents(long minId, long maxId) {

		ensureCount();

		SQLiteDatabase db = null;

		String ID_FIELD_NAME = Constants.Database.PayloadTable.Fields.Id.NAME;

		String filter = String.format(Locale.US, "%s >= %d AND %s <= %d", ID_FIELD_NAME, minId, ID_FIELD_NAME, maxId);
		Log.d(TAG, filter);
		int deleted = -1;

		synchronized (this) {

			try {
				db = getWritableDatabase();
				deleted = db.delete(Constants.Database.PayloadTable.NAME, filter, null);

				// decrement the row counter
				count.addAndGet(-deleted);

			} catch (SQLiteException e) {

				Log.eToGaOnly(true, TAG, "Failed to remove items from the Segment.io payload db: ", e);

			} finally {
				if (db != null) db.close();
			}
		}

		return deleted;
	}
	
	int removeErrorEvents(long minId, long maxId) {

		ensureErrorsCount();

		SQLiteDatabase db = null;

		String ID_FIELD_NAME = Constants.Database.ErrorsPayloadTable.Fields.Id.NAME;

		String filter = String.format(Locale.US, "%s >= %d AND %s <= %d", ID_FIELD_NAME, minId, ID_FIELD_NAME, maxId);

		int deleted = -1;

		synchronized (this) {

			try {
				db = getWritableDatabase();
				deleted = db.delete(Constants.Database.ErrorsPayloadTable.NAME, filter, null);

				// decrement the row counter
				errorCount.addAndGet(-deleted);

			} catch (SQLiteException e) {

				Log.eToGaOnly(true, TAG, "Failed to remove items from the errors payload db: ", e);

			} finally {
				if (db != null) db.close();
			}
		}

		return deleted;
	}

}