package com.aniways;

import java.util.Iterator;

import org.json.JSONException;
import org.json.JSONObject;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.app.Application;
import android.app.Application.ActivityLifecycleCallbacks;
import android.app.Service;
import android.content.Context;
import android.os.Bundle;

import com.aniways.analytics.Reporter;
import com.aniways.analytics.info.InfoManager;
import com.aniways.bugsnag.MetaData;
import com.aniways.bugsnag.android.Bugsnag;
import com.aniways.data.AniwaysPrivateConfig;
import com.aniways.service.utils.AniwaysServiceUtils;

public class BugsnagReporter implements Reporter {
	protected static final String TAG = "AniwaysBygsnagReporter";
	private Context mContext;

	public BugsnagReporter(Context context) {
		Bugsnag.register(context, AniwaysPrivateConfig.getInstance().errorHandlingAPIKey);
		Bugsnag.setAppVersion(AniwaysServiceUtils.SDK_VERSION);
		registerApplicationLifecycle(context);
		mContext = context.getApplicationContext();
	}

	@SuppressLint("NewApi")
	private void registerApplicationLifecycle(Context applicationContext) {

		if(!Utils.isAndroidVersionAtLeast(14)){
			return;
		}

		ActivityLifecycleCallbacks lifeCycle = new ActivityLifecycleCallbacks() {

			@Override
			public void onActivityStopped(Activity activity) {}

			@Override
			public void onActivityStarted(Activity activity) {}

			@Override
			public void onActivitySaveInstanceState(Activity activity, Bundle outState) {}

			@Override
			public void onActivityResumed(Activity activity) {
				Bugsnag.onActivityResume(activity);
			}

			@Override
			public void onActivityPaused(Activity activity) {
				Bugsnag.onActivityPause(activity);
			}

			@Override
			public void onActivityDestroyed(Activity activity) {
				Bugsnag.onActivityDestroy(activity);
			}

			@Override
			public void onActivityCreated(Activity activity, Bundle savedInstanceState) {
				Bugsnag.onActivityCreate(activity);
			}
		};

		if(applicationContext instanceof Application){
			((Application)applicationContext).registerActivityLifecycleCallbacks(lifeCycle);
		} else if(applicationContext instanceof Activity){
			((Activity)applicationContext).getApplication().registerActivityLifecycleCallbacks(lifeCycle);
		} else if (applicationContext instanceof Service){
			((Service)applicationContext).getApplication().registerActivityLifecycleCallbacks(lifeCycle);
		}

	}

	@Override
	public void notify(String tag, String message, Throwable error) {
		MetaData additionalData = new MetaData();
		additionalData.addToTab("TAG", tag);
		additionalData.addToTab("Message", message);
		JSONObject info = new InfoManager().build(mContext);
		spreadInfo(info, additionalData);
		Bugsnag.notify(error, additionalData);
	}

	private void spreadInfo(JSONObject info, MetaData additionalData) {
		JSONObject flattenJSON = flatten(info, null, "");
		Iterator<String> keys = flattenJSON.keys();
		while (keys.hasNext()){
			String next = (String) keys.next();
			try {
				additionalData.addToTab(next, flattenJSON.get(next));
			} catch (JSONException e) {
				Log.eToGaOnly(true, TAG, "Error adding info json", e);
			}
		}
	}
	
	private static JSONObject flatten(JSONObject object, JSONObject flattened, String parentKey){
	    if(flattened == null){
	        flattened = new JSONObject();
	    }
	    Iterator<?> keys = object.keys();
	    while(keys.hasNext()){
	        String key = (String)keys.next();
	        String flattenKey;
			if(parentKey.equals("")){
	        	flattenKey = key;
	        }else{
	        	flattenKey = parentKey + "-" + key;
	        }
	        try {
	            if(object.get(key) instanceof JSONObject){
	            	flatten(object.getJSONObject(key), flattened, flattenKey);
	            } else {
	                flattened.put(flattenKey, object.get(key));
	            }
	        } catch(JSONException e){
	            Log.eToGaOnly(true, TAG, "Error flattening info json", e);
	        }
	    }
	    return flattened;
	}
}
