package com.aniways;

import android.app.Activity;
import android.content.Context;
import android.graphics.Point;
import android.graphics.Rect;
import android.graphics.drawable.BitmapDrawable;
import android.text.Editable;
import android.text.Spannable;
import android.util.DisplayMetrics;
import android.util.TypedValue;
import android.view.Display;
import android.view.Gravity;
import android.view.LayoutInflater;
import android.view.View;
import android.view.View.OnClickListener;
import android.view.Window;
import android.view.WindowManager;
import android.widget.Button;
import android.widget.ImageView;
import android.widget.PopupWindow;
import android.widget.RelativeLayout.LayoutParams;

import com.aniways.AniwaysTutorialView.onTutorialWindowSingleTapConfirmed;
import com.aniways.analytics.AnalyticsReporter;
import com.aniways.analytics.AnalyticsReporter.EditTextTutorialAction;
import com.aniways.analytics.AnalyticsReporter.PhrasesEventAction;
import com.aniways.analytics.GoogleAnalyticsReporter;
import com.aniways.contextual.ContextualPopup;
import com.aniways.data.AniwaysConfiguration.Verbosity;
import com.aniways.data.AniwaysPhraseReplacementData;
import com.aniways.data.AniwaysPrivateConfig;
import com.aniways.data.AniwaysStoreManager;
import com.aniways.data.AppData;
import com.aniways.data.JsonParser;
import com.aniways.data.Phrase;
import com.aniways.data.SettingsKeys;
import com.aniways.emoticons.button.AniwaysEmoticonsButtonMaker;
import com.aniways.quick.action.QuickAction;

import java.util.ArrayList;

public class AniwaysEditTextTextWatcher implements IAniwaysTextWatcher
{
    private AniwaysEditText mMessageEditor;
    // The tutorial popup window
    private AniwaysTutorialPopupWindow mTutorialWindow;
    // The tutorial popup window view
    private AniwaysTutorialView mTutorialView;
    //
    private Editable mEditable;
    //
    private Rect mMarkedWordRect;
    //
    private AniwaysSuggestionSpan mAniwaysSuggestionSpanForTutorial;

    private IAniwaysImageSpan mImageSpanToRemove;
    private CharSequence mTextBelowImageSpan;
    private IAniwaysWordMarkerSpan mMarkerSpanToRemove;
    private CharSequence mTextBelowMarkerSpan;
    private int mMarkerSpanToRemoveStart;
    private CharSequence textBeforeTextChange;
    private AniwaysSuggestionSpan mSuggestionSpanForTheReplacedImage;
    private int mNewTextStart;
    private int mNewTextEnd;
    private String mNewText;
    private int mPlaceToWatchOutForReAddingDeletedTextBelowMarkerSpan;
    private int mTextToRemoveStart;
    private CharSequence mTextToRemove;
    private String mPreviousTextReported;
    private boolean mDuplicateChangeReported;
    private boolean mLeftPopupOpenDueToSpace;

    private final static String TAG = "AniwaysEditTextTextWatcher";

    public AniwaysEditTextTextWatcher(AniwaysEditText messageEditor)
    {
        this.mMessageEditor = messageEditor;
        try
        {
            // Inflate the tutorial layout
            LayoutInflater inflater =
                    (LayoutInflater) messageEditor.getContext().getSystemService(Context.LAYOUT_INFLATER_SERVICE);
            mTutorialView = (AniwaysTutorialView) inflater.inflate(R.layout.aniways_contextual_suggestions_tutorial, null);

            // Set the close tutorial button functionality
            Button closeTutorialButton = (Button) mTutorialView.findViewById(R.id.aniways_tutorial_close_button);

            if (mTutorialView.isNimbuzz())
            {
                Log.d(TAG, "Nimbuzz detected");
                closeTutorialButton.setVisibility(View.GONE);
            }
            else
            {
                // Disable the nimbuzz message
                Log.d(TAG, "Showing regular tutorial");
                View disableText = mTutorialView.findViewById(R.id.aniways_tutorail_settings_text);
                disableText.setVisibility(View.GONE);

                // Set up the tutorial close button
                closeTutorialButton.setOnClickListener(new OnClickListener()
                {

                    @Override
                    public void onClick(View v)
                    {
                        try
                        {
                            cancelTutorial();
                        }
                        catch (Throwable ex)
                        {
                            Log.e(true, TAG, "Caught Exception in onClick", ex);
                        }
                    }

                });
            }

            // Set up click on all other areas, including the Tutorial highlighted area
            mTutorialView.setOnTutorialWindowSingleTapConfirmed(new onTutorialWindowSingleTapConfirmed()
            {

                @Override
                public boolean tutorialWindowSingleTapConfirmed(boolean inHighlightedArea)
                {
                    try
                    {
                        if (mTutorialWindow != null && mTutorialWindow.isShowing())
                        {
                            if (inHighlightedArea)
                            {
                                mTutorialWindow.dismiss();
                                if (mAniwaysSuggestionSpanForTutorial != null)
                                {
                                    mMessageEditor.displaySuggestions(mAniwaysSuggestionSpanForTutorial, AniwaysPhraseReplacementData.getDataParser(), IAniwaysImageSpan.IconSelectionOrigin.ContextualManual);
                                    GoogleAnalyticsReporter.reportEvent(Verbosity.Info, "Statistics", "Tutorial Phrase Clicked", "tutorial display number: " + AppData.getInt(SettingsKeys.EDIT_TEXT_TUTORIAL_SHOWN_TIMES) + ", phrase: " + mAniwaysSuggestionSpanForTutorial.phrase.getPhraseSubPhraseString(), 0);
                                    AnalyticsReporter.reportEditTextTutorialEvent(EditTextTutorialAction.phraseClicked, mMessageEditor.getMessageId(), mAniwaysSuggestionSpanForTutorial.phrase, AppData.getInt(SettingsKeys.EDIT_TEXT_TUTORIAL_SHOWN_TIMES));
                                    return true;
                                }
                            }
                            else if (AniwaysPrivateConfig.getInstance().closeTutorialOnClickAnywhere)
                            {
                                // Close the tutorial popup window
                                cancelTutorial();
                                return true;
                            }
                        }
                        return false;
                    }
                    catch (Throwable ex)
                    {
                        Log.e(true, TAG, "Caught Exception in highlightedEllipseSingleTapConfirmed", ex);
                        return false;
                    }
                }
            });
        }
        catch (Throwable ex)
        {
            Log.e(true, TAG, "Caught an Exception while creating edit text text watcher");
        }
    }

    @Override
    public void afterTextChanged(Editable editable)
    {
        if (mDuplicateChangeReported)
        {
            Log.v(TAG, "Duplicate text change reported. Ignoring.");
            return;
        }

        try
        {
            AniwaysPrivateConfig config = AniwaysPrivateConfig.getInstance();
            if (!config.contextualIconSuggestionsEnabled && config.debugPerformance)
            {
                return;
            }

            Log.d(TAG, "in after text changed");
            if (mImageSpanToRemove != null)
            {
                Log.d(TAG, "afterTextChanged: " + (editable.getSpanStart(mImageSpanToRemove) >= 0 ? "true" : "false"));
            }

            mEditable = editable;
            mAniwaysSuggestionSpanForTutorial = null;
            long startTime = System.currentTimeMillis();

            // With text additions, check for characters that aren't supposed to change the state of an open popup.
            // Only allow leaving the popup open if we haven't already encountered a space character.
            if (mNewText != null && !mLeftPopupOpenDueToSpace)
            {
                // We allow any combination of special characters (leaveAutoPopupOpenCharacters) to keep the popup open,
                // up to the point we encounter the first space. That space character keeps the popup open,
                // but indicates that the user has now "moved on" from the current term being edited.

                // Put in regex mode: "[phrase](specialChars)*\s" leaves the popup open. Any character after that
                // will close the popup.

                switch (mNewText.length())
                {
                    case 1:
                        // Special case - Note that we've encountered a space and that the user has now "moved on".
                        if (mNewText.charAt(0) == ' ')
                        {
                            Log.v(TAG, "Found a first space after a phrase, leaving the popup as is.");
                            mLeftPopupOpenDueToSpace = true;
                            return;
                        }
                        else if (config.leaveAutoPopupOpenCharacters.contains(mNewText.subSequence(0, 1)))
                        {
                            Log.v(TAG, "Found a special character, leaving the popup as is.");
                            mLeftPopupOpenDueToSpace = false;
                            return;
                        }
                        break;
                    case 2:
                        if (Character.isSurrogatePair(mNewText.charAt(0), mNewText.charAt(1)) &&
                            config.leaveAutoPopupOpenCharacters.contains(mNewText.subSequence(0, 2)))
                        {
                            // Found a surrogate pair that is defined as "leave open" in the config.
                            Log.v(TAG, "Found a special surrogate pair, leaving the popup as is.");
                            mLeftPopupOpenDueToSpace = false;
                            return;
                        }
                        break;
                }
            }

            // Reaching here means it's ok to close the popup and everything is to be treated normally.
            mLeftPopupOpenDueToSpace = false;

            IAniwaysTextWatcher watcher = mMessageEditor.removeTheAniwaysTextWatcher();
            assertWatcherIsThis(watcher);

            // Bring back text beneath deleted image span if needed, bring back part of
            // the text beneath deleted marker spans if needed, etc
            // Do this only if it is the same text object from the b4 text changed callback
            // (otherwise it means that the callback is called because the setText() was called
            // with a new string and then we do not want to add text to it in any case..
            if (editable == textBeforeTextChange)
            {
                Log.d(TAG, "Start take care of span deletions");
                takeCareOfAniwaysSpanDeletions(editable);
                Log.d(TAG, "End take care of span deletions");
            }

            boolean shouldDismissExistingContextualPopup = true;

            if (mEditable.length() > 0)
            {
                // TODO: for performance: don't do this on the entire text
                // TODO: If there is an image and now it is adjoined to a word then it will no longer be clickable
                //       because the suggestion marker will not be added. Fix it
                // TODO: wherever there is selection start and end - make sure start < end

                long startTime2 = System.currentTimeMillis();
                JsonParser parser = AniwaysPhraseReplacementData.getDataParser();
                Log.d(TAG, "Start add markers");
                AniwaysMarkerInserter.addSuggestionMarkersToText(mEditable, this.mMessageEditor, parser, mMessageEditor.getContext());

                Log.d(TAG, "End add markers");
                AnalyticsReporter.ReportTiming(Verbosity.Statistical, startTime2, "Performance", "Phrase Highlighting", String.valueOf(mEditable.length()), TAG, "num chars");

                int selStart = mMessageEditor.getSelectionStart();
                int selEnd = mMessageEditor.getSelectionEnd();
                AniwaysSuggestionSpan[] aniwaysSuggestionSpanCandidatesForTutorial = mEditable.getSpans(selStart, selEnd, AniwaysSuggestionSpan.class);

                // remove all the suggestion spans that are covered by image spans, since we do not need to autoreplace, show popup, or show tutorial for them.
                // these spans are either spans for pictures inserted by user to replace text, or from a paste action, or from smileys button
                if (aniwaysSuggestionSpanCandidatesForTutorial != null && aniwaysSuggestionSpanCandidatesForTutorial.length > 0)
                {
                    // Find a suggestion span that is not covered by an image span and that there is a marker span under it
                    for (AniwaysSuggestionSpan candidateSpan : aniwaysSuggestionSpanCandidatesForTutorial)
                    {
                        boolean foundCoveringImageSpan = false;
                        boolean foundCoveredMarkerSpan = false;

                        int spanStart = mEditable.getSpanStart(candidateSpan);
                        int spanEnd = mEditable.getSpanEnd(candidateSpan);
                        IAniwaysImageSpan[] imageSpans = mEditable.getSpans(spanStart, spanEnd, IAniwaysImageSpan.class);
                        if (imageSpans != null && imageSpans.length > 0)
                        {
                            for (IAniwaysImageSpan imageSpan : imageSpans)
                            {
                                int st = mEditable.getSpanStart(imageSpan);
                                int e = mEditable.getSpanEnd(imageSpan);
                                if (st != spanEnd && e != spanStart)
                                {
                                    foundCoveringImageSpan = true;
                                    break;
                                }
                            }
                        }
                        if (foundCoveringImageSpan)
                        {
                            // There is an image span covering it, look at the next one
                            continue;
                        }

                        // There is no image span covering it, check that there is a marker span underneath
                        IAniwaysWordMarkerSpan[] markerSpans = mEditable.getSpans(spanStart, spanEnd, IAniwaysWordMarkerSpan.class);
                        if (markerSpans != null && markerSpans.length > 0)
                        {
                            for (IAniwaysWordMarkerSpan ms : markerSpans)
                            {
                                int st = mEditable.getSpanStart(ms);
                                int e = mEditable.getSpanEnd(ms);
                                if (st == spanStart && e == spanEnd)
                                {
                                    foundCoveredMarkerSpan = true;
                                    break;
                                }
                            }
                        }

                        if (!foundCoveredMarkerSpan)
                        {
                            Log.e(true, TAG, "Could not find a marker span underneath a suggestion span without an image above it");
                            // This is an illegal state, but we continue to search for another span that might be in a legal state..
                            continue;
                        }

                        // Found it!
                        mAniwaysSuggestionSpanForTutorial = candidateSpan;
                        break;
                    }
                }

                if (mAniwaysSuggestionSpanForTutorial != null)
                {
                    boolean allowTutorial = false;

                    // Display suggestions automatically, or replace phrase - according to config.
                    switch (config.suggestionMode)
                    {
                        case Manual:
                            allowTutorial = true;
                            break;
                        case AutoDisplaySuggestions:
                            shouldDismissExistingContextualPopup = false;
                            // TODO: think of better way to choose the suggestion span (if indeed more than 1 is legal here)
                            mMessageEditor.displaySuggestions(mAniwaysSuggestionSpanForTutorial, parser, IAniwaysImageSpan.IconSelectionOrigin.ContextualAuto);
                            break;
                        case AutoReplacePhrases:
                            shouldDismissExistingContextualPopup = false;
                            // TODO: think of better way to choose the suggestion span (if indeed more than 1 is legal here)
                            mMessageEditor.replaceSuggestionWithIcon(mAniwaysSuggestionSpanForTutorial, parser, IAniwaysImageSpan.IconSelectionOrigin.Autoreplace);
                            break;
                    }

                    // Display tutorial if needed.
                    // Do not display if the emoticons button is showing, or if we are deleting something (it gives a bad experience)
                    if (allowTutorial &&
                        config.useEditTextTutorial &&
                        !AniwaysEmoticonsButtonMaker.isEmoticonsButtonShowing() &&
                        mImageSpanToRemove == null &&
                        AppData.getInt(SettingsKeys.MANUAL_TAPS_COUNT) < AniwaysPrivateConfig.smartUseMarkerTapCount)
                    {
                        int timesShowedTutorial = AppData.getInt(SettingsKeys.EDIT_TEXT_TUTORIAL_SHOWN_TIMES);
                        if (timesShowedTutorial < config.maxTimesToShowTutorial)
                        {
                            String phrase = mAniwaysSuggestionSpanForTutorial.phrase.getPartToReplace().trim();
                            if (!Utils.isStringEmpty(phrase))
                            {
                                ArrayList<String> wordsThatDisplayedTutorial = AppData.getStringList(SettingsKeys.WORDS_THAT_DISPLAYED_TUTORIAL);
                                if (!wordsThatDisplayedTutorial.contains(phrase))
                                {
                                    if (timesShowedTutorial == 0 ||
                                        (wordsThatDisplayedTutorial.size() / timesShowedTutorial >= config.numberOfWordsRequiredToShowTutorialAgain && AppData.getInt(SettingsKeys.MANUAL_TAPS_COUNT) <= timesShowedTutorial /* because clicking on the tutorial doesn't count.. */))
                                    {
                                        mMarkedWordRect = calculateMarkedWordRect();
                                        // Do not display tutorial if the marked phrase contains more than one word,
                                        // and spread in more than one line.
                                        if (mMarkedWordRect != null && mMarkedWordRect.left < mMarkedWordRect.right)
                                        {
                                            //QuickAction.dismissAllOpenQuickActions();
                                            ContextualPopup.dismissAllOpenPopups();
                                            showTutorialPopupWindow();
                                            // Increase the number of times that the tutorial showed.
                                            int timeTutorialShown = AppData.incrementInt(SettingsKeys.EDIT_TEXT_TUTORIAL_SHOWN_TIMES);
                                            AnalyticsReporter.reportEditTextTutorialEvent(EditTextTutorialAction.opened, mMessageEditor.getMessageId(), mAniwaysSuggestionSpanForTutorial == null ? null : mAniwaysSuggestionSpanForTutorial.phrase, timeTutorialShown);

                                        }
                                        else if (mMarkedWordRect == null)
                                        {
                                            // This happens after recent layout change - so get point of position in text returns null..
                                            Log.w(true, TAG, "mMarkedWordRect is null, so not signaling to show tutorial");
                                        }
                                    }

                                    // TODO: move these to after the tutorial was actually displayed..
                                    AppData.putStringList(SettingsKeys.WORDS_THAT_DISPLAYED_TUTORIAL, wordsThatDisplayedTutorial);
                                }
                            }
                        }
                    }
                }
            }

            if (shouldDismissExistingContextualPopup)
            {
                //QuickAction.dismissAllOpenQuickActions();
                ContextualPopup.dismissAllOpenPopups();
            }

            watcher = mMessageEditor.addTheAniwaysTextWatcher();
            assertWatcherIsThis(watcher);

            Log.d(TAG, "ATC end");
            AnalyticsReporter.ReportTiming(Verbosity.Verbose, startTime, "Performance", "After Text Change Processing", String.valueOf(mEditable.length()), TAG, "num chars");
        }
        catch (Throwable ex)
        {
            Log.e(true, TAG, "Caught Exception in Aftet text changed. Message is: " + editable, ex);
        }
    }

    private void takeCareOfAniwaysSpanDeletions(Editable s)
    {
        try
        {

            Log.d(TAG, "Text below marker span: " + mTextBelowMarkerSpan + ". Below image span: " + mTextBelowImageSpan);

            // Cover emojis on new text
            if (this.mNewTextEnd <= s.length())
            {
                Editable newText = (Editable) s.subSequence(mNewTextStart, mNewTextEnd);
                if (newText != null && newText.toString().equalsIgnoreCase(mNewText))
                {
                    // Fix image and suggestion spans that contain more than just emoji unicode
                    Log.d(TAG, "Start make image spans fit emojis");
                    boolean changed = makeImageSpansFitEmojiAgain(s, mNewTextStart, mNewTextEnd, AniwaysPhraseReplacementData.getDataParser());
                    Log.d(TAG, "End make image spans fit emojis. Changed something: " + changed);

                    // Cover uncovered emoji
                    Log.d(TAG, "Start handle uncovered emojis");
                    boolean coveredSomething = AniwaysIconConverter.handleUncoveredEmoji(s, mNewTextStart, mNewTextEnd, AniwaysPhraseReplacementData.getDataParser(), this.mMessageEditor.getContext(), mMessageEditor, null, false, false, true);
                    Log.d(TAG, "End handle uncovered emojis. Covered something: " + coveredSomething);

                    Log.d(TAG, "Start remove isolated surrogates");
                    // remove isolated surrogates (should never happen, but Swiftkey has this issue.
                    // Needs to happen here, cause after this point, the position of the new text is not guaranteed
                    //TODO: do not need this calc anymore
                    int mNewTextEnd = mNewTextStart + newText.length();
                    int skip;
                    for (int i = mNewTextStart; i < mNewTextEnd; i += skip)
                    {
                        char c = s.charAt(i);
                        boolean remove = false;
                        skip = 1;
                        if (Character.isHighSurrogate(c))
                        {
                            if (mNewTextStart + 1 == s.length() || !Character.isLowSurrogate(s.charAt(i + 1)))
                            {
                                remove = true;
                            }
                            else
                            {
                                //we found a surrogate pair ahead, so skeep
                                skip += 1;
                            }
                        }
                        else if (Character.isLowSurrogate(c))
                        {
                            if (mNewTextStart == 0 || !Character.isHighSurrogate(s.charAt(i - 1)))
                            {
                                remove = true;
                            }
                        }
                        if (remove)
                        {
                            s.delete(mNewTextStart, mNewTextStart + 1);
                            mNewTextEnd--;
                            Log.w(false, TAG, "Removed lone surrogate");
                        }
                    }
                    Log.d(TAG, "End remove isolated surrogates");
                }
            }

            // Remove the image span that some text was deleted beneath it, if needed
            if (mImageSpanToRemove != null)
            {

                int start = mEditable.getSpanStart(this.mImageSpanToRemove);
                if (start >= 0)
                {
                    Log.d(TAG, "ATC deletion of icon needing deletion detected");
                    int end = mEditable.getSpanEnd(this.mImageSpanToRemove);
                    mEditable.removeSpan(mImageSpanToRemove);
                    mEditable.removeSpan(mSuggestionSpanForTheReplacedImage);
                    mEditable.delete(start, end);
                    mMessageEditor.setSelection(start);
                    Log.d(TAG, "Completely deleted an icon, and the text below it. Start pos: " + start + ". End: " + end);
                }
                // If needed, restore the text below the removed icon..
                //if(config.revertDeletedIconToText){
                //	mEditable.insert(start, mTextBelowImageSpan);
                //	mMessageEditor.setSelection(start + this.mTextBelowImageSpan.length());
                //}
            }
            // Add back the text below the marker span if it was deleted (or some of it), if needed
            else if (mMarkerSpanToRemove != null)
            {
                Log.d(TAG, "ATC deletion of marker span detected");

                // This means the span was already removed with the text below it
                // (cause its an exclusive-exclusive span which means that it is
                // removed when all the text is removed..)
                // TODO: This is not entirely correct because, in theory, between the time
                // mMarkerSpanToRemoveStart was calculated in b4 text change and this callback is called
                // other after text changed callbacks could have changed the text and this location.
                // The correct thing to do is to add a zero length span in the mMarkerSpanToRemoveStart location in
                // onTextChanged and then get its location here and delete it..
                int start = this.mMarkerSpanToRemoveStart;
                int length = mTextBelowMarkerSpan.length() - 1;
                String textToAdd = mTextBelowMarkerSpan.subSequence(0, length).toString();
                Log.d(TAG, "Adding text that was below deleted marker span: " + textToAdd + ". Pos: " + start);
                mEditable.insert(start, textToAdd);
                mMessageEditor.setSelection(start + length);
            }
            // Remove text that needs removing (i.e. in some keyboards, if a space after the marker span is deleted,
            // then it removes the char and then the entire text under the marker span and then it adds the text. After
            // it deletes the entire text, we add back some of it, so now we need to remove what we added, as the keyboard
            // adds it back. We could theoretically remove what the keyboard remove, but then some keyboards don't like it
            // and we have other problems, so it is better to just remove what we added).
            else if (mTextToRemove != null)
            {
                int start = mTextToRemoveStart;
                int selStart = mMessageEditor.getSelectionStart();
                Log.d(TAG, "Deleting text to remove: " + mTextToRemove + ". At position: " + start);
                mEditable.delete(start, start + mTextToRemove.length());
                mMessageEditor.setSelection(selStart - mTextToRemove.length());
            }
        }
        catch (Throwable ex)
        {
            Log.e(true, TAG, "Caught Exception while taking care of span deletion.", ex);
        }
    }

    // TODO: does this work well on 4 char emoji unicodes
    // This method should NEVER!! change the text, we rely on it later!!!!
    private static boolean makeImageSpansFitEmojiAgain(Spannable text, int start, int end, JsonParser parser)
    {
        boolean result = false;
        int skip;

        start = Math.max(0, start - 3);
        end = Math.min(end + 3, text.length());
        IAniwaysImageSpan[] spans = text.getSpans(start, end, IAniwaysImageSpan.class);
        if (spans != null && spans.length > 0)
        {
            for (IAniwaysImageSpan span : spans)
            {
                int startSpan = text.getSpanStart(span);
                int origEndSpan = text.getSpanEnd(span);
                int endSpan = Math.min(text.length(), origEndSpan + 1);
                int startEmoji = -1;
                int endEmoji = -1;

                for (int i = startSpan; i < endSpan; i += skip)
                {
                    // This supports the old iOS encoding
                    //char c = text.charAt(i);
                    //if (isSoftBankEmoji(c)) {
                    //    icon = getSoftbankEmojiResource(c);
                    //    skip = icon == 0 ? 0 : 1;
                    //}

                    IconData icon = null;

                    int unicode = Character.codePointAt(text, i);
                    skip = Character.charCount(unicode);

                    if (unicode > 0xff)
                    {
                        icon = parser.getEmoji(unicode);
                    }

                    boolean secondUnicodeExists = false;
                    int followUnicode = -1;
                    if (icon == null && i + skip < endSpan)
                    {
                        followUnicode = Character.codePointAt(text, i + skip);
                        icon = parser.getEmoji(new int[]{unicode, followUnicode});
                        if (icon != null)
                        {
                            secondUnicodeExists = true;
                            skip += Character.charCount(followUnicode);
                        }
                    }

                    // Add the emoji varient selector if needed
                    if (icon != null && i + skip < endSpan)
                    {
                        int nextUnicode = Character.codePointAt(text, i + skip);
                        if (nextUnicode == EmojiWithVarientSelector.EMOJI_VARIENT_SELECTOR)
                        {
                            IconData oldIcon = icon;
                            if (secondUnicodeExists)
                            {
                                icon = parser.getEmojiWithVarientSelector(new int[]{unicode, followUnicode});
                            }
                            else
                            {
                                icon = parser.getEmojiWithVarientSelector(unicode);
                            }
                            if (icon == null)
                            {
                                icon = new EmojiWithVarientSelector(oldIcon);
                                parser.addEmojiWithVarientSelector(icon);
                            }
                            skip += Character.charCount(nextUnicode);
                        }
                    }

                    if (icon != null)
                    {
                        startEmoji = i;
                        endEmoji = i + skip;
                        break;
                    }
                }


                // If there is no emoji unicode below span then remove it
                // If the span covers more than the emoji, make it cover the emoji only
                if (startEmoji != startSpan || endEmoji != origEndSpan)
                {
                    result = true;

                    text.removeSpan(span);

                    // get suggestion span below image and remove as well
                    AniwaysSuggestionSpan ss = getSuggestionSpanBelowImage(text, startSpan, origEndSpan);
                    if (ss != null)
                    {
                        text.removeSpan(ss);
                    }

                    if (startEmoji < 0 || endEmoji < 0 || startEmoji < startSpan || endEmoji > endSpan)
                    {
                        continue;
                    }

                    if (startEmoji > startSpan || endEmoji < origEndSpan)
                    {
                        if (ss != null)
                        {
                            text.setSpan(ss, startEmoji, endEmoji, Spannable.SPAN_EXCLUSIVE_EXCLUSIVE);
                        }
                        text.setSpan(span, startEmoji, endEmoji, Spannable.SPAN_EXCLUSIVE_EXCLUSIVE);
                        Log.d(TAG, "Reduced emoji from: " + startSpan + "-" + endSpan + " to: " + startEmoji + "-" + endEmoji);
                    }
                    else if (startEmoji >= startSpan && origEndSpan < endSpan && endEmoji > origEndSpan && endEmoji <= endSpan)
                    {
                        if (ss != null)
                        {
                            text.setSpan(ss, startEmoji, endEmoji, Spannable.SPAN_EXCLUSIVE_EXCLUSIVE);
                        }
                        text.setSpan(span, startEmoji, endEmoji, Spannable.SPAN_EXCLUSIVE_EXCLUSIVE);
                        Log.d(TAG, "Expanded emoji from: " + startSpan + "-" + endSpan + " to: " + startEmoji + "-" + endEmoji);
                    }
                }
            }
        }
        return result;
    }

    private static AniwaysSuggestionSpan getSuggestionSpanBelowImage(Spannable text, int startSpan, int endSpan)
    {
        AniwaysSuggestionSpan[] suggestionSpans = text.getSpans(startSpan, endSpan, AniwaysSuggestionSpan.class);
        if (suggestionSpans != null && suggestionSpans.length > 0)
        {
            for (AniwaysSuggestionSpan suggestionSpan : suggestionSpans)
            {
                int suggestionSpanStart = text.getSpanStart(suggestionSpan);
                int end2 = text.getSpanEnd(suggestionSpan);
                if (suggestionSpanStart == startSpan && end2 == endSpan)
                {
                    return suggestionSpan;
                }
            }
        }
        return null;
    }

    @Override
    public void beforeTextChanged(CharSequence text, int start, int count, int after)
    {
        try
        {
            AniwaysPrivateConfig config = AniwaysPrivateConfig.getInstance();

            if (!config.contextualIconSuggestionsEnabled && config.debugPerformance)
            {
                return;
            }

            QuickAction.cancelAutoDismissOfAllOpenQuickActions();
            ContextualPopup.cancelAutoDismissOfAllOpenQuickActions();

            long startTime = System.currentTimeMillis();

            mTextToRemoveStart = -1;
            mTextToRemove = null;

            // Look out for the keyboard trying to add back text we already added..
            // (we added that text minus the last char)
            if (start == mPlaceToWatchOutForReAddingDeletedTextBelowMarkerSpan &&
                mTextBelowMarkerSpan != null &&
                count == 0 &&
                after == mTextBelowMarkerSpan.length())
            {
                // We remove the text we added b4 because removing the text which the keyboard is adding
                // causes unwanted behaviors in some keyboards (they forcefully add it back)
                // The text we added b4 is located just before where we add text now. Its length is
                // one char less than the text that was below the marker span.
                // We temporarily do not cut the last char for comparison later, and then we cut it :)
                mTextToRemove = mTextBelowMarkerSpan;
                mTextToRemoveStart = start - (mTextToRemove.length() - 1);
            }

            mImageSpanToRemove = null;
            mSuggestionSpanForTheReplacedImage = null;
            mTextBelowImageSpan = null;
            mMarkerSpanToRemove = null;
            mTextBelowMarkerSpan = null;
            mMarkerSpanToRemoveStart = -1;
            textBeforeTextChange = text;
            mPlaceToWatchOutForReAddingDeletedTextBelowMarkerSpan = -1;

            Log.d(TAG, "B4 text changed - start: " + start + " count: " + count + " after: " + after);

            // Check if there is a deletion
            if (after < count)
            {
                int imageSpanToRemoveStart = -1;

                Spannable spannable = (Spannable) text;
                // check if there is an image span at the end of the block that is about to be removed
                // and that this image span is not starting at that position
                // (which means we are deleting just before the image span)
                IAniwaysImageSpan[] imageSpans = spannable.getSpans(start + count, start + count, IAniwaysImageSpan.class);
                int imageSpanEnd = 0;
                if (imageSpans != null && imageSpans.length > 0)
                {
                    for (IAniwaysImageSpan span : imageSpans)
                    {
                        imageSpanToRemoveStart = spannable.getSpanStart(span);
                        if (imageSpanToRemoveStart != start + count)
                        {
                            mImageSpanToRemove = span;
                            imageSpanEnd = spannable.getSpanEnd(span);
                            mTextBelowImageSpan = text.subSequence(imageSpanToRemoveStart, imageSpanEnd).toString(); //Need the toString() to remove spans
                            break;
                        }
                        imageSpanToRemoveStart = -1;
                    }
                }

                if (mImageSpanToRemove != null)
                {
                    AniwaysSuggestionSpan[] suggestionSpans = spannable.getSpans(imageSpanToRemoveStart, imageSpanEnd, AniwaysSuggestionSpan.class);
                    if (suggestionSpans != null && suggestionSpans.length > 0)
                    {
                        for (AniwaysSuggestionSpan suggestionSpan : suggestionSpans)
                        {
                            int spanStart = spannable.getSpanStart(suggestionSpan);
                            int spanEnd = spannable.getSpanEnd(suggestionSpan);
                            if (spanStart == imageSpanToRemoveStart && spanEnd == imageSpanEnd)
                            {
                                mSuggestionSpanForTheReplacedImage = suggestionSpan;
                                break;
                            }
                        }
                    }
                }
                // check if there is a word marker span that is completely being removed, and only it is being removed
                else if (after == 0)
                {
                    IAniwaysWordMarkerSpan[] markerSpans = spannable.getSpans(start + count, start + count, IAniwaysWordMarkerSpan.class);

                    if (markerSpans != null && markerSpans.length > 0)
                    {
                        int markerSpanEnd;
                        for (IAniwaysWordMarkerSpan span : markerSpans)
                        {
                            mMarkerSpanToRemoveStart = spannable.getSpanStart(span);
                            markerSpanEnd = spannable.getSpanEnd(span);
                            if (mMarkerSpanToRemoveStart == start && markerSpanEnd == start + count)
                            {
                                mMarkerSpanToRemove = span;
                                mTextBelowMarkerSpan = text.subSequence(mMarkerSpanToRemoveStart, markerSpanEnd).toString(); //Need the toString() to remove spans
                                mPlaceToWatchOutForReAddingDeletedTextBelowMarkerSpan = markerSpanEnd - 1;
                                break;
                            }
                            mMarkerSpanToRemoveStart = -1;
                        }
                    }
                }

                // Fire icon deleted event
                // TODO: it is more correct to fire it only after the text has changed - need to have all the data available there..
                if (mImageSpanToRemove != null)
                {
                    try
                    {
                        String removedIconPath = this.mImageSpanToRemove.getIcon().getFileName();
                        Phrase p = null;
                        String phraseString = null;
                        if (mSuggestionSpanForTheReplacedImage == null)
                        {
                            if(this.mImageSpanToRemove.getIcon().iconSelectionTag == null) {
                                Log.w(false, TAG, "There was no suggestion span for a replaced image span: " + removedIconPath);
                            }
                        }
                        else
                        {
                            p = mSuggestionSpanForTheReplacedImage.phrase;
                            if (p == null)
                            {
                                Log.e(true, TAG, "There is a deleting, but phrase is null: " + removedIconPath);
                            }
                            else
                            {
                                phraseString = Phrase.getPhraseSubPhraseString(p.getName(), p.getPartToReplace());
                            }
                        }
                        AnalyticsReporter.reportPhrasesEvent(
                                PhrasesEventAction.backspaceDelete,
                                this.mMessageEditor.getMessageId(),
                                p,
                                null,
                                AniwaysStoreManager.isIconUnlocked(this.mImageSpanToRemove.getIcon()),
                                this.mImageSpanToRemove.getIcon(),
                                false, null, 0, null, 0, 0, 0, 0, 0, 0, 0, 0, 0, null, null, mImageSpanToRemove.getImageSpanMetadata());
                        GoogleAnalyticsReporter.reportEvent(Verbosity.Info, "Deleted Icon", removedIconPath, phraseString, 0);
                        AnalyticsReporter.ReportTiming(Verbosity.Verbose, startTime, "Performance", "B4 Text Change Processing", String.valueOf(text.length()), TAG, "num chars");
                    }
                    catch (Throwable ex)
                    {
                        Log.e(true, TAG, "Caught exception while firing icon deleted event", ex);
                    }
                }
            }

            Log.d(TAG, "B4 text changed - end");
        }
        catch (Throwable ex)
        {
            Log.e(true, TAG, "Caught Exception in beforeTextChanged", ex);
        }
    }

    @Override
    public void onTextChanged(CharSequence text, int start, int before, int count)
    {
        String stringContent = text.toString();
        if (mPreviousTextReported != null && mPreviousTextReported.contentEquals(stringContent))
        {
            mDuplicateChangeReported = true;
            return;
        }
        else
        {
            mDuplicateChangeReported = false;
            mPreviousTextReported = stringContent;
        }

        AniwaysPrivateConfig config = AniwaysPrivateConfig.getInstance();

        if (!config.contextualIconSuggestionsEnabled && config.debugPerformance)
        {
            return;
        }

        Log.d(TAG, "onTextChanged.  text: " + text + ". start: " + start + ". before: " + before + ". count: " + count);
        if (mImageSpanToRemove != null)
        {
            Log.d(TAG, "onTextChanged: " + (((Spannable) text).getSpanStart(mImageSpanToRemove) >= 0 ? "true" : "false"));
        }

        // Check if the text the keyboard is trying to add is indeed the same as the one that it previously removed
        // (and was highlighted for replacement by Aniways).
        // If so, then delete the text we added b4, when the span and the text under it were removed.
        // We delete this and not the text the keyboard is adding now, because deleting the text the keyboard is adding now
        // causes issues with some keyboards (they just put it again)
        if (mTextToRemove != null)
        {
            if (text.subSequence(start, start + count).toString().equalsIgnoreCase(mTextToRemove.toString()))
            {
                // The actual text to remove is one char less than the one being added (they keyboard
                // would probably later delete one char as well)
                mTextToRemove = mTextToRemove.subSequence(0, mTextToRemove.length() - 1);
            }
            else
            {
                mTextToRemoveStart = -1;
                mTextToRemove = null;
            }
        }

        this.mNewTextStart = start;
        this.mNewTextEnd = start + count;
        this.mNewText = text.subSequence(start, start + count).toString();//toString() to remove spans

        Log.d(TAG, "onTextChanged - End");
    }

    private void showTutorialPopupWindow()
    {
        // Calculate the size of the tutorial popup window so it will cover the entire screen without the
        // notification bar.
        Display display =
                ((WindowManager) mMessageEditor.getContext().getSystemService(Context.WINDOW_SERVICE))
                        .getDefaultDisplay();

        Rect screenRect = new Rect();
        Window window = ((Activity) mMessageEditor.getContext()).getWindow();
        window.getDecorView().getWindowVisibleDisplayFrame(screenRect);

        int statusBarHeight = screenRect.top;

        // Calculate where to put the marked word transparent ellipse
        mMarkedWordRect = calculateMarkedWordRect();

        if (mMarkedWordRect == null)
        {
            Log.w(true, TAG, "mMarkedWordRect is null, so not showing tutorial");
            return;
        }

        mTutorialView.setHighlightEllipseValues(mMarkedWordRect);

        // Calculate where to put the tap instructions image
        ImageView tapInstructionsImage = (ImageView) mTutorialView.findViewById(R.id.aniways_tutorial_instructions_image);

        DisplayMetrics displaymetrics = new DisplayMetrics();
        ((Activity) mMessageEditor.getContext()).getWindowManager().getDefaultDisplay().getMetrics(displaymetrics);

        int[] imageAndMargin = getTapInstructionsImageAndMargin(
                displaymetrics.widthPixels,
                displaymetrics.heightPixels - statusBarHeight,
                mMarkedWordRect);

        tapInstructionsImage.setImageResource(imageAndMargin[0]);

        LayoutParams tapInstructionsLayoutParams = (LayoutParams) tapInstructionsImage.getLayoutParams();
        tapInstructionsLayoutParams.leftMargin = imageAndMargin[1];
        tapInstructionsLayoutParams.topMargin = imageAndMargin[2];
        tapInstructionsImage.setLayoutParams(tapInstructionsLayoutParams);

        // Create the tutorial popup window with the calculated size.
        if (mTutorialWindow == null)
        {
            mTutorialWindow = new AniwaysTutorialPopupWindow(mTutorialView,
                                                             display.getWidth(),
                                                             display.getHeight(),
                                                             true);
        }
        else
        {
            mTutorialWindow.setWidth(display.getWidth());
            mTutorialWindow.setHeight(display.getHeight());
        }

        mTutorialWindow.setInputMethodMode(PopupWindow.INPUT_METHOD_NOT_NEEDED);

        mTutorialWindow.showAtLocation(mMessageEditor, Gravity.CENTER, 0, 0);
    }

    private int[] getTapInstructionsImageAndMargin(int screenWidth, int screenHeight, Rect markedWordRect)
    {
        if ((screenWidth / 2) >= markedWordRect.centerX())
        {
            if ((screenHeight / 2) >= markedWordRect.centerY())
            {
                // bottom left
                // centerX, bottom
                return new int[]{R.drawable.aniways_tutorial_bottom_left,
                        markedWordRect.centerX(),
                        markedWordRect.bottom};
            }
            else
            {
                // upper left
                // centerX, top - imageHeight
                BitmapDrawable bd = (BitmapDrawable) mMessageEditor.getContext()
                        .getResources().getDrawable(R.drawable.aniways_tutorial_top_left);
                return new int[]{R.drawable.aniways_tutorial_top_left,
                        markedWordRect.centerX(),
                        markedWordRect.top - bd.getBitmap().getHeight()};
            }
        }
        else
        {
            if ((screenHeight / 2) >= markedWordRect.centerY())
            {
                // bottom right
                // centerX - imageWidth, bottom
                BitmapDrawable bd = (BitmapDrawable) mMessageEditor.getContext()
                        .getResources().getDrawable(R.drawable.aniways_tutorial_bottom_right);
                return new int[]{R.drawable.aniways_tutorial_bottom_right,
                        markedWordRect.centerX() - bd.getBitmap().getWidth(),
                        markedWordRect.bottom};
            }
            else
            {
                // upper right
                // centerX - imageWidth, top - imageHeight
                BitmapDrawable bd = (BitmapDrawable) mMessageEditor.getContext()
                        .getResources().getDrawable(R.drawable.aniways_tutorial_top_right);
                return new int[]{R.drawable.aniways_tutorial_top_right,
                        markedWordRect.centerX() - bd.getBitmap().getWidth(),
                        markedWordRect.top - bd.getBitmap().getHeight()};
            }
        }
    }

    /**
     * @return !!Be careful, may return null!!
     */
    private Rect calculateMarkedWordRect()
    {
        // Calculate the status bar height
        Rect screenRect = new Rect();
        Window window = ((Activity) mMessageEditor.getContext()).getWindow();
        window.getDecorView().getWindowVisibleDisplayFrame(screenRect);

        int statusBarHeight = screenRect.top;

        // Get the start and end position (in chars) of the suggestion span (the highlighted words)
        int start = mEditable.getSpanStart(mAniwaysSuggestionSpanForTutorial);
        int end = mEditable.getSpanEnd(mAniwaysSuggestionSpanForTutorial);

        // Get the points (in pixels) of the highlighted rectangle -
        // relative to the EditText control's upper left and lower left corners

        // These 2 points are relative to the upper left corner of the EditText
        Point upperLeftCorner = mMessageEditor.getPointOfPositionInText(start, true);
        Point lowerRightCorner = mMessageEditor.getPointOfPositionInText(end, false);

        if (upperLeftCorner == null || lowerRightCorner == null)
        {
            // May happen following recent layout change
            return null;
        }

        // Translate the points to be relative to the screen
        int[] location = new int[2];
        mMessageEditor.getLocationOnScreen(location);

        Log.v(TAG, String.valueOf(location[0] + upperLeftCorner.x) + ", " +
                String.valueOf(location[1] + upperLeftCorner.y) + ", " +
                String.valueOf(location[0] + lowerRightCorner.x) + ", " +
                String.valueOf(location[1] + lowerRightCorner.y));

        // The rectangle represent the marked word
        int ellipseMargin = (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, 7,
                                                            mMessageEditor.getContext().getResources().getDisplayMetrics());

        return new Rect(location[0] + upperLeftCorner.x - ellipseMargin,
                        location[1] + upperLeftCorner.y - statusBarHeight - ellipseMargin,
                        location[0] + lowerRightCorner.x + ellipseMargin,
                        location[1] + upperLeftCorner.y + ellipseMargin);

    }

    private void assertWatcherIsThis(IAniwaysTextWatcher watcher)
    {
        if (!this.equals(watcher))
        {
            Log.e(true, TAG, "Removed or added a text watcher which is not this");
        }
    }

    private void cancelTutorial()
    {
        // Close the tutorial popup window
        mTutorialWindow.dismiss();
        Phrase phrase = null;
        String phraseString = "";
        if (mAniwaysSuggestionSpanForTutorial != null && mAniwaysSuggestionSpanForTutorial.phrase != null)
        {
            phrase = mAniwaysSuggestionSpanForTutorial.phrase;
            phraseString = phrase.getPhraseSubPhraseString();
        }

        GoogleAnalyticsReporter.reportEvent(Verbosity.Info, "Statistics", "Tutorial Canceled", "tutorial display number: " + AppData.getInt(SettingsKeys.EDIT_TEXT_TUTORIAL_SHOWN_TIMES) + ", phrase: " + phraseString, 0);
        AnalyticsReporter.reportEditTextTutorialEvent(EditTextTutorialAction.cancelled, mMessageEditor.getMessageId(), phrase, AppData.getInt(SettingsKeys.EDIT_TEXT_TUTORIAL_SHOWN_TIMES));
    }
}