package com.aniways;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.UUID;

import android.annotation.SuppressLint;
import android.content.ComponentCallbacks2;
import android.content.Context;
import android.content.res.Configuration;
import android.graphics.Point;
import android.support.annotation.NonNull;
import android.text.Editable;
import android.text.Spannable;
import android.text.TextUtils;
import android.text.TextWatcher;
import android.util.AttributeSet;
import android.view.KeyEvent;
import android.view.MotionEvent;
import android.view.View;
import android.view.inputmethod.EditorInfo;
import android.view.inputmethod.InputConnection;
import android.widget.EditText;
import android.widget.PopupWindow;

import com.aniways.analytics.AnalyticsReporter;
import com.aniways.analytics.AnalyticsReporter.PhrasesEventAction;
import com.aniways.analytics.GoogleAnalyticsReporter;
import com.aniways.contextual.ContextualPopup;
import com.aniways.data.AniwaysAnimatedAssetInfo;
import com.aniways.data.AniwaysConfiguration;
import com.aniways.data.AniwaysConfiguration.Verbosity;
import com.aniways.data.AniwaysPhraseReplacementData;
import com.aniways.data.AniwaysPrivateConfig;
import com.aniways.data.AniwaysStatics;
import com.aniways.data.AniwaysStoreManager;
import com.aniways.data.GiphyIconData;
import com.aniways.data.JsonParser;
import com.aniways.data.Phrase;
import com.aniways.emoticons.button.AniwaysRecentIconsManager;
import com.aniways.quick.action.ContextualItemSelectedByUserEvent;
import com.aniways.quick.action.ContextualPopupCreationContext;
import com.aniways.quick.action.EventAggregator;
import com.aniways.quick.action.IEvent;
import com.aniways.quick.action.IEventListener;
import com.aniways.service.utils.AniwaysAction;
import com.aniways.service.utils.AniwaysServiceUtils;
import com.aniways.volley.toolbox.Volley;

import org.json.JSONObject;

// TODO: deal with substitute == false

public class AniwaysEditText extends EditText implements ISuggestionDisplayer, IAniwaysGestureResponder, IAniwaysTextEditor
{
    private static final String TAG = "AniwaysEditText";
    public static final int REVERT_TO_TEXT = 374;
    private AniwaysEditTextAndTextViewCommonPart mAniwaysEditTextAndTextViewerCommonPart;
    private UUID mMessageId = UUID.randomUUID();
    private IAniwaysWordMarkerSpan mActiveMarkerSpan;
    private HashSet<onContentSelectedListener> mContentSelectedListeners = new HashSet<>();
    private boolean showIndicationForGifSelectionInEditText = true;
    private static boolean ignoredDueToLandscape = false;
    private ContextualPopup popup;

    public interface onContentSelectedListener{
        public void onGiphySelected(GiphySelectedData data);
        public void onGiphyRemoved(GiphySelectedData data);

        class GiphySelectedData {
            public JSONObject jsonEntry;
            public int size;

            public GiphySelectedData(JSONObject jsonEntry, int size){

                this.jsonEntry = jsonEntry;
                this.size = size;
            }
        }
    }

    public AniwaysEditText(Context context)
    {
        this(context, null);
    }

    public AniwaysEditText(Context context, AttributeSet attributeSet)
    {
        super(context, attributeSet);
        if(!this.isInEditMode()){
            mAniwaysEditTextAndTextViewerCommonPart = new AniwaysEditTextAndTextViewCommonPart(this);
            registerToContentSelectedEvents();
            registerGoingToBackgroundDetector();
        }
    }

    public AniwaysEditText(Context context, AttributeSet attrs, int defStyle){
        super(context, attrs, defStyle);
        if(!this.isInEditMode()){
            mAniwaysEditTextAndTextViewerCommonPart = new AniwaysEditTextAndTextViewCommonPart(this);
            registerToContentSelectedEvents();
            registerGoingToBackgroundDetector();
        }
    }

    @SuppressLint("NewApi")
    private void registerGoingToBackgroundDetector() {
        if(!Utils.isAndroidVersionAtLeast(14)){
            return;
        }
        getContext().registerComponentCallbacks(new ComponentCallbacks2() {
            @Override
            public void onConfigurationChanged(Configuration newConfig) {}

            @Override
            public void onLowMemory() {}

            @Override
            public void onTrimMemory(int level) {
                if(level == ComponentCallbacks2.TRIM_MEMORY_UI_HIDDEN) {
                    if(popup != null && popup.isShowing()){
                        Log.i(TAG,"Dismissing popup since going to background");
                        popup.dismiss();
                    }
                }
            }
        });

    }

    private void registerToContentSelectedEvents(){
        EventAggregator.getInstance().register(TAG, new IEventListener() {
            @Override
            public void notify(IEvent event) {

                try {
                    ContextualItemSelectedByUserEvent contextualItemSelectedByUserEvent = event instanceof ContextualItemSelectedByUserEvent ? ((ContextualItemSelectedByUserEvent) event) : null;
                    if (contextualItemSelectedByUserEvent != null) {
                        final IconData selectedIcon = contextualItemSelectedByUserEvent.assetInfo.getIconData();
                        final Phrase phrase = contextualItemSelectedByUserEvent.popupCreationContext.phrase;
                        final AniwaysSuggestionSpan suggestionSpan = contextualItemSelectedByUserEvent.popupCreationContext.suggestionSpan;
                        final IAniwaysImageSpan.IconSelectionOrigin selectionOrigin = contextualItemSelectedByUserEvent.popupCreationContext.selectionOrigin;
                        final int pos = contextualItemSelectedByUserEvent.selectedGridItemInfo.getPosition();
                        final int rowNum = contextualItemSelectedByUserEvent.selectedGridItemInfo.getRow();
                        final int numRows = contextualItemSelectedByUserEvent.selectedGridItemInfo.getNumRows();
                        final int columnNum = contextualItemSelectedByUserEvent.selectedGridItemInfo.getColumn();
                        final int numColumns = contextualItemSelectedByUserEvent.selectedGridItemInfo.getNumColumns();
                        final int totalNumberOfItems = contextualItemSelectedByUserEvent.selectedGridItemInfo.getTotalNumberOfItems();
                        final IAniwaysImageSpan previousImageSpan = contextualItemSelectedByUserEvent.popupCreationContext.previousImageSpan;
                        final IconData previousIcon = previousImageSpan == null ? null : previousImageSpan.getIcon();
                        final AniwaysPrivateConfig cfg = AniwaysPrivateConfig.getInstance();

                        if (selectedIcon != null && selectedIcon.assetType == AssetType.AnimatedGif) {
                            if (selectedIcon.assetProvider == IAniwaysImageSpan.AssetProvider.Giphy) {
                                // Put the gif in the EditText if needed,
                                // If we dont show the indication in the EditText then we report to the app now, since they manage removal themselves,
                                // if we do show the indication then we report all of them in the end when the user sends, in order to avoid messy code telling when
                                // Gifs are added and removed
                                JSONObject json = ((AniwaysAnimatedAssetInfo) contextualItemSelectedByUserEvent.assetInfo).getJson();
                                Object anim = Volley.getImageLoader().getCached(((AniwaysAnimatedAssetInfo) contextualItemSelectedByUserEvent.assetInfo).getLowQualityAnimatedGifUrl(), cfg.getMaxWidthForCache(selectedIcon), cfg.getMaxHeightForCache(selectedIcon), selectedIcon.getFileName());
                                int size = 220000;
                                if (anim != null && anim instanceof byte[]){
                                    size = ((byte[])anim).length;
                                }
                                onContentSelectedListener.GiphySelectedData data = new onContentSelectedListener.GiphySelectedData(json,size);
                                if (showIndicationForGifSelectionInEditText) {
                                    selectedIcon.iconSelectionTag = data;
                                    insetTextCoveredWithIconAfterPhrase(selectedIcon, suggestionSpan, phrase, new IAniwaysImageSpan.ImageSpanMetadata(selectedIcon, selectionOrigin));
                                } else {
                                    for (onContentSelectedListener listener : mContentSelectedListeners) {
                                        listener.onGiphySelected(data);
                                    }
                                }
                                // Fire events
                                AnalyticsReporter.reportPhrasesEvent(
                                        PhrasesEventAction.selected,
                                        mMessageId,
                                        phrase,
                                        new IconData[]{selectedIcon},
                                        AniwaysStoreManager.isIconUnlocked(selectedIcon),
                                        null, //TODO: Need to know if there was a previous animation..
                                        false, null, 0, null, 0, 0, pos + 1, totalNumberOfItems, rowNum, columnNum, numRows, numColumns, 0,
                                        null, null, new IAniwaysImageSpan.ImageSpanMetadata(selectedIcon, selectionOrigin));

                                GoogleAnalyticsReporter.reportEvent(Verbosity.Info, "Picked Animated Gif-"+selectionOrigin, phrase.getPhraseSubPhraseString(), ((GiphyIconData) selectedIcon).idInGiphy, 0);
                                //if (previousImageSpan != null) {
                                //GoogleAnalyticsReporter.reportEvent(Verbosity.Info, "Switched Animated Gif", phrase.getPhraseSubphraseString(), previousImageSpan.getIcon().getFileName() + " - " + selectedIconPath, 0);
                                //}
                            } else {
                                Log.w(true, TAG, "User selected animated Gif with provider which is not Giphy: " + selectedIcon.assetProvider);
                            }
                        } else if (selectedIcon != null && (selectedIcon.assetType == AssetType.Emoticons || selectedIcon.assetType == AssetType.Emoji)) {
                            //ActionItem actionItem = (ActionItem) contextualItemSelectedByUserEvent.assetInfo;
                            final String[] selectedIconPath = {"X_delete - "};

                            if (selectedIcon.id == REVERT_TO_TEXT) {
                                replaceImageWithItsText(previousImageSpan, suggestionSpan, false);
                                assert previousImageSpan != null;
                                selectedIconPath[0] += previousImageSpan.getIcon().getFileName();

                                AnalyticsReporter.reportPhrasesEvent(
                                        PhrasesEventAction.popupDelete,
                                        mMessageId,
                                        phrase,
                                        null,
                                        AniwaysStoreManager.isIconUnlocked(previousImageSpan.getIcon()),
                                        previousIcon,
                                        false, null, 0, null, 0, 0, pos + 1, totalNumberOfItems, rowNum, columnNum, numRows, numColumns, 0,
                                        null, null, new IAniwaysImageSpan.ImageSpanMetadata(previousIcon, selectionOrigin));

                            } else {
                                if (popup != null){
                                    popup.handleLockedIconSelectedEvent(contextualItemSelectedByUserEvent, new AniwaysAction<Boolean>() {
                                        @Override
                                        public void call(Boolean success) {
                                            onSuggestionSelected(selectedIcon, previousImageSpan, suggestionSpan, selectionOrigin);
                                            selectedIconPath[0] = selectedIcon.getFileName();
                                            // Add icon to the recent icons
                                            AniwaysRecentIconsManager.addRecentIcon(selectedIcon.id, selectedIcon.assetType);
                                            AnalyticsReporter.reportPhrasesEvent(
                                                    PhrasesEventAction.selected,
                                                    mMessageId,
                                                    phrase,
                                                    new IconData[]{selectedIcon},
                                                    AniwaysStoreManager.isIconUnlocked(selectedIcon),
                                                    previousIcon,
                                                    false, null, 0, null, 0, 0, pos + 1, totalNumberOfItems, rowNum, columnNum, numRows, numColumns, 0,
                                                    null, null, new IAniwaysImageSpan.ImageSpanMetadata(selectedIcon, selectionOrigin));
                                        }
                                    });
                                }
                            }

                            GoogleAnalyticsReporter.reportEvent(Verbosity.Info, "Picked Icon-" + selectionOrigin, phrase.getPhraseSubPhraseString(), selectedIconPath[0], 0);
                            if (previousImageSpan != null) {
                                GoogleAnalyticsReporter.reportEvent(Verbosity.Info, "Switched Icon", phrase.getPhraseSubPhraseString(), previousImageSpan.getIcon().getFileName() + " - " + selectedIconPath[0], 0);
                            }

                            mActiveMarkerSpan = null;
                        }

                        AnalyticsReporter.ReportTiming(Verbosity.Info, contextualItemSelectedByUserEvent.popupCreationContext.startTimeUserInPopup, "User Action", "Time To Select Icon", phrase.getPhraseSubPhraseString(), TAG, "The phrase");

                        mActiveMarkerSpan = null;
                    }
                } catch (Exception e) {
                    Log.e(true,TAG,"on selected contextual content by user exception", e);
                } finally {
                    // Dismiss the popup
                    ContextualPopup.dismissAllOpenPopups();
                }
            }
        });
    }

    // Supports copy-paste of images
    @Override
    public boolean onTextContextMenuItem(int id){
        try{
            return mAniwaysEditTextAndTextViewerCommonPart.onTextContextMenuItem(id);
        }
        catch(Throwable ex){
            Log.e(true, TAG, "Caught Exception onTextContextMenuItem. id is: " + id, ex);
            return true;
        }
    }

    public void registerContentSelectedListener(onContentSelectedListener listener, boolean showIndicationForGifSelectionInEditText){
        this.showIndicationForGifSelectionInEditText = showIndicationForGifSelectionInEditText;
        this.mContentSelectedListeners.add(listener);
    }

    public void unregisterContentSelectedListener(onContentSelectedListener listener){
        this.mContentSelectedListeners.remove(listener);
    }

    // Recognize the back button press close the suggestion or icon info popup
    @Override
    public boolean dispatchKeyEventPreIme(@NonNull KeyEvent event){
        try{
            return mAniwaysEditTextAndTextViewerCommonPart.dispatchKeyEventPreIme(event);
        }
        catch(Throwable ex){
            Log.e(true, TAG, "Caught Exception in dispatchKeyEventPreIme. Event is: " + event, ex);
            return true;
        }
    }

    @Override
    public boolean onTouchEvent(@NonNull MotionEvent event){
        try{
            return mAniwaysEditTextAndTextViewerCommonPart.onTouchEvent(event);
        }
        catch(Throwable ex){
            // TODO: fix this!!
            Log.e(true, TAG, "Caught excedption in onTouchEvent. Event is: " + event, ex);
            return true;
        }
    }

    // TODO: this should not be public, but it is because it inherits from interface. Find out how to change
    @Override
    public void displaySuggestions(final AniwaysSuggestionSpan suggestionSpan, final JsonParser parser, final IAniwaysImageSpan.IconSelectionOrigin selectionOrigin)
    {
        if (AniwaysStatics.getApplicationContext().getResources().getConfiguration().orientation == Configuration.ORIENTATION_LANDSCAPE)
        {
            ignoredDueToLandscape = true;
            return;
        }

        final long startTime = System.currentTimeMillis();
        Phrase phrase = suggestionSpan.phrase;
        Spannable text = getText();

        int[] startEnd = new int[] {text.getSpanStart(suggestionSpan), text.getSpanEnd(suggestionSpan)};
        Utils.setMaxMinStartEnd(startEnd);
        int start = startEnd[0];
        int end = startEnd[1];

        // Add 'remove icon' action if there is already an icon there
        // TODO: I assume here that the suggestion span is exactly where the image span is and that there
        // is only one image span per suggestion span. Need to verify in the code that the start and end
        // of the image span equals to the start and end of the suggestion span and that there is only one
        // image span there..
        IAniwaysImageSpan[] imageSpans = this.getText().getSpans(start, end, IAniwaysImageSpan.class);
        final IAniwaysImageSpan previousImageSpan = (imageSpans.length > 0 ? imageSpans[0] :null);

        ContextualPopup.dismissAllOpenPopups();
        final ContextualPopupCreationContext context = new ContextualPopupCreationContext(phrase, suggestionSpan, parser, previousImageSpan, selectionOrigin, System.currentTimeMillis(), this.mMessageId, this);
        popup = new ContextualPopup(this.getContext(), context);

        popup.addTouchInterceptor(new OnTouchListener()
        {

            @Override
            public boolean onTouch(View v, MotionEvent event)
            {
                if (event.getAction() == MotionEvent.ACTION_OUTSIDE)
                {
                    suggestionSpan.setDismissEventTime(event.getEventTime());
                }
                return false;
            }

        });

        IAniwaysWordMarkerSpan[] wordMarkerSpans = this.getText().getSpans(start, end, IAniwaysWordMarkerSpan.class);
        IAniwaysWordMarkerSpan targetMarkerSpan = null;
        for (IAniwaysWordMarkerSpan wordMarkerSpan : wordMarkerSpans) {
            final IAniwaysWordMarkerSpan finalWordMarkerSpan = wordMarkerSpan;
            targetMarkerSpan = wordMarkerSpan;
            wordMarkerSpan.setSelected(this, true);
            popup.setOnDismissListener(new PopupWindow.OnDismissListener()
            {
                @Override
                public void onDismiss()
                {
                    finalWordMarkerSpan.setSelected(AniwaysEditText.this, false);
                    mActiveMarkerSpan = null;
                    AnalyticsReporter.ReportTiming(Verbosity.Info, context.startTimeUserInPopup, "User Action", "Time user spent in popup-" + selectionOrigin, "" + 0, TAG, "");
                }
            });
        }

        if (mActiveMarkerSpan != null)
        {
            mActiveMarkerSpan.setSelected(AniwaysEditText.this, false);
        }

        mActiveMarkerSpan = targetMarkerSpan;

        if (ignoredDueToLandscape)
        {
            ignoredDueToLandscape = false;
            final ContextualPopup finalPopup = popup;

            this.postDelayed(new Runnable()
            {
                @Override
                public void run()
                {
                    finalPopup.show(AniwaysEditText.this);
                }
            }, 250);
        }
        else
        {
            popup.show(this);
        }

        if(selectionOrigin == IAniwaysImageSpan.IconSelectionOrigin.ContextualManual) {
            GoogleAnalyticsReporter.reportEvent(AniwaysConfiguration.Verbosity.Info, "Display Suggestions-" + selectionOrigin, phrase.getPhraseSubPhraseString(), "", 0);
            AnalyticsReporter.reportPhrasesEvent(
                    AnalyticsReporter.PhrasesEventAction.tapped,
                    mMessageId,
                    phrase,
                    new IconData[0],
                    false,
                    previousImageSpan == null ? null : previousImageSpan.getIcon(),
                    false, null, 0, null, 0, 0, 0, 0, 0, 0, 0, 0, 0, null, null,
                    null);
        }

        AnalyticsReporter.ReportTiming(Verbosity.Verbose, startTime, "Performance", "Display Suggestions Popup", "", TAG, "");
    }

    // Called only from text changed watcher if it needs to replace automatically
    // once there is a suggestion. When the user picks a suggestion then the replaceWithIcon method is called
    void replaceSuggestionWithIcon(AniwaysSuggestionSpan suggestionSpan, JsonParser parser, IAniwaysImageSpan.IconSelectionOrigin selectionOrigin)
    {
        Phrase phrase = suggestionSpan.phrase;

        IconData[] icons = phrase.emoticonsPhraseAssetBuilder.icons;

        if (icons == null || icons.length == 0){
            Log.w(true, TAG, "No icons for phrase: " + phrase + " . Keywords version is: " + parser.getKeywordsVersion());
            return;
        }

        // TODO: select icon more intelligently, currently - just taking the first one
        IconData icon = icons[0];
        onSuggestionSelected(icon, null, suggestionSpan, selectionOrigin);

    }

    private void replaceWithIcon(IconData icon, IAniwaysImageSpan previousImageSpan, AniwaysSuggestionSpan suggestionSpan, IAniwaysImageSpan.ImageSpanMetadata imageSpanMetadata, boolean removeSuggestionSpan){
        if(suggestionSpan == null){
            Log.e(true, TAG, "SuggestionSpan is null");
            return;
        }
        mAniwaysEditTextAndTextViewerCommonPart.removeTheAniwaysTextWatcher();

        if(previousImageSpan != null){
            this.replaceImageWithItsText(previousImageSpan, suggestionSpan, true);
        }

        int startOfReplacementText = this.getText().getSpanStart(suggestionSpan);
        int endOfReplacementText =  this.getText().getSpanEnd(suggestionSpan);
        if(startOfReplacementText < 0){
            Log.w(true, TAG, "Suggestion span is not in text. Text is: " + this.getText().toString());
            return;
        }

        int selectionStart = getSelectionStart();
        int selectionEnd = getSelectionEnd();

        // Insert the icon to cover the text from startIndex to startIndex + length
        if(removeSuggestionSpan){
            getText().removeSpan(suggestionSpan);
            AniwaysIconConverter.insertAniwaysIconToText(getContext(), getText(), icon, startOfReplacementText, suggestionSpan.phrase,
                    this, null, imageSpanMetadata, false, false, false);
        }
        else {
            AniwaysIconConverter.insertAniwaysIconToText(
                    getContext(),
                    getText(),
                    suggestionSpan,
                    icon,
                    imageSpanMetadata,
                    false);
        }
        int lengthDiff = endOfReplacementText - startOfReplacementText - icon.getUnicodeToReplaceText().length();

        // We do this to remove link between word and autocorrection mechanism
        setText(getText());

        // Setting selection to where it was before, or to after the image if it was inside the image
        if(selectionStart < endOfReplacementText && selectionEnd > startOfReplacementText){
            this.setSelection(endOfReplacementText);
        } else{
            if(selectionEnd >= endOfReplacementText){
                selectionEnd -= lengthDiff;
            }
            if(selectionStart >= endOfReplacementText){
                selectionStart -= lengthDiff;
            }
            setSelection(selectionStart, selectionEnd);
        }

        // If we are at the end of the text then add a space and move the selection after it
        // because otherwise the next letters will not be shown on screen on some phones
        if((this.getText().length() == selectionStart) && (selectionStart == endOfReplacementText - lengthDiff)){
            this.getText().append(" ");
            this.setSelection(selectionStart+1);
        }

        mAniwaysEditTextAndTextViewerCommonPart.addTheAniwaysTextWatcher();
    }

    /**
     * This method is used to force the keyboard not to take the whole screen in
     * landscape mode
     */
    @Override
    public InputConnection onCreateInputConnection(@NonNull EditorInfo outAttrs) {

        if(AniwaysPrivateConfig.getInstance().isImeFlagNoExtractUiForced){
            outAttrs.imeOptions |= EditorInfo.IME_FLAG_NO_EXTRACT_UI;
            this.setImeOptions(outAttrs.imeOptions);
        }

        return super.onCreateInputConnection(outAttrs);
    }

    private void onSuggestionSelected(IconData selectedIcon, IAniwaysImageSpan previousImageSpan, AniwaysSuggestionSpan suggestionSpan, IAniwaysImageSpan.IconSelectionOrigin selectionOrigin)
    {
        replaceWithIcon(selectedIcon, previousImageSpan, suggestionSpan, new IAniwaysImageSpan.ImageSpanMetadata(selectedIcon, selectionOrigin), false);
    }

    // Remove the image span and replace it with its text.
    // Assume that the suggestion span is still there
    private void replaceImageWithItsText(IAniwaysImageSpan imageSpan, AniwaysSuggestionSpan suggestionSpan, boolean keepSuggestionSpan) {
        // TODO: Make sure that the suggestion span is still here and that it has the exact same bounds as the image span..
        Editable text = getText();
        int start = text.getSpanStart(imageSpan);
        int end = text.getSpanEnd(imageSpan);
        text.removeSpan(imageSpan);
        text.removeSpan(suggestionSpan);

        if(imageSpan instanceof IAniwaysDynamicImageSpan){
            this.mAniwaysEditTextAndTextViewerCommonPart.onRemoveDynamicImageSpan((IAniwaysDynamicImageSpan)imageSpan);
        }

        // Marker and suggestion spans will be add by text watcher..
        Phrase phrase = imageSpan.getPhrase();
        if(suggestionSpan.originalText.equalsIgnoreCase(phrase.getPartToReplace())){
            text.replace(start, end, suggestionSpan.originalText);
        } else {
            text.replace(start, end, phrase.getPartToReplace());
        }

        if(keepSuggestionSpan){
            text.setSpan(suggestionSpan, start, start + imageSpan.getPhrase().getLengthOfPartToReplace(),Spannable.SPAN_EXCLUSIVE_EXCLUSIVE);
        }
    }

    @Override
    public void onSingleTap(MotionEvent event) {
        try{
            // If user clicked on an AniwaysSuggestionSpan then call its onClick (this is not
            // called by the EditText because it doesn't receive these events in order for it
            // not to display the paste menu and also to avoid the bug where if the span is
            // at the end of the input and the user clicks somewhere after it then the EditText treats it
            // like a click on the span itself
            int pos = this.mAniwaysEditTextAndTextViewerCommonPart.getPositionOfTouchEventInText(event);
            Log.d(TAG, "pos of touch event in text: " + pos);
            if (pos != -1){
                AniwaysSuggestionSpan[] suggestionSpan = this.getText().getSpans(pos, pos, AniwaysSuggestionSpan.class);
                if (suggestionSpan.length != 0) {
                    long eventTime = -1;
                    if(event.getAction() == MotionEvent.ACTION_DOWN){
                        eventTime = event.getEventTime();
                    }
                    suggestionSpan[0].onClick(eventTime, getText());
                }
            }
        }
        catch(Throwable ex){
            // TODO: fix this!!
            Log.e(true, TAG, "Caught excedption in onSingleTap. Event is: " + event, ex);
        }
    }

    @Override
    public boolean callSuperOnTextContextMenuItem(int id) {
        return super.onTextContextMenuItem(id);
    }

    // Used, among other things, for 'playing back' touch events that were recorded on AniwaysSuggestionSpans before determining
    // if they should be sent to the EditText control (if they are double clicks and not single clicks)
    @Override
    public boolean callSuperOnTouchEvent(MotionEvent event) {
        return super.onTouchEvent(event);
    }

    @Override
    public boolean callSuperDispatchKeyEventPreIme(KeyEvent event) {
        return super.dispatchKeyEventPreIme(event);
    }

    @Override
    public Editable getText() {
        if(!this.isInEditMode()){
            AniwaysStatics.makeSureAniwaysIsInitialized(false);
        }
        return super.getText();
    }

    @Override
    public void setText(CharSequence text, boolean justCallSuper) {
        if(justCallSuper){
            super.setText(text,BufferType.EDITABLE);
        }
        else{
            setText(text, BufferType.EDITABLE);
        }
    }

    public String getOriginalText(){
        Editable text;

        try{
            text = this.getText();
        }
        // This is because that during construction there is a call to this method
        // and there is an Android bug that causes this Exception
        catch (ClassCastException ex)
        {
            text = Editable.Factory.getInstance().newEditable("");
            Log.v(TAG, "set text cast exception caught");
        }

        return Aniways.getOriginalString(this.getContext(), text);
    }

    public void onSendingMessage(){
        Log.d(TAG, "On message sent called");

        Editable oldText;
        try{
            oldText = this.getText();
        }
        // This is because that during construction there is a call to this method
        // and there is an Android bug that causes this Exception
        catch (ClassCastException ex)
        {
            oldText = Editable.Factory.getInstance().newEditable("");
            Log.v(TAG, "cast exception caught in on send message");
        }
        if (TextUtils.isEmpty(oldText)){
            Log.w(true, TAG, "onSendingMessage called when old text is null or empty");
            return;
        }

        // Setting this as the last user activity time (to know that his definitions from the server should
        // be kept fresh)
        AniwaysServiceUtils.setLastUserActivityTime(this.getContext(), System.currentTimeMillis());

        // Report the selected Animated Gifs to the app if it selected to show an indication for them in
        // the EditText and therefore have Aniways manage their removal - sending this here because we want
        // to avoid telling the app every time a Gif is removed since it is hard to really know..
        if(this.showIndicationForGifSelectionInEditText){
            sendIndicationOfSelectedAnimatedGifs();
        }

        // Firing this dimension again because dimensions dont seem to be working all the time, so trying here again..
        GoogleAnalyticsReporter.reportCustomDimention(4, AniwaysPhraseReplacementData.getDataParser().getKeywordsVersion());
        String originalText = this.getOriginalText();
        GoogleAnalyticsReporter.reportEvent(Verbosity.Info, "Statistics", "Message Sent", originalText, originalText.length());
        AnalyticsReporter.reportMessageSentEvent(mMessageId, originalText);

        // Report recognized phrases and whether they have pictures or not..
        // TODO: when we move away from GA this will need to send raw data without processing
        ArrayList<RecognizedPhrase> recognizedPhrases = getRecognizedPhrases(true);
        if(recognizedPhrases != null && recognizedPhrases.size() > 0){
            int serialNumber = 0;
            for(RecognizedPhrase rp : recognizedPhrases){
                serialNumber++;
                Phrase phrase = rp.phrase;
                String imageName = "Not Replaced";
                if(rp.isCoveredByImage()){
                    imageName = rp.icon.getFileName();
                    if(rp.imageSpanMetadata.isFromAniwaysKeyboard()){
                        imageName = "From button: " + imageName;
                    }
                    else if(rp.imageSpanMetadata.isFromUnknownKeyboard()){
                        imageName = "From app: " + imageName;
                    }
                }

                if(phrase == null){
                    Log.e(true, TAG, "phrase is null");
                    continue;
                }

                // TODO: make sure correct. Also GA
                GoogleAnalyticsReporter.reportEvent(Verbosity.Info, "Sent Phrase Statistics", phrase.getPhraseSubPhraseString(), imageName, 0);
                if(!rp.isCoveredByImage() || rp.imageSpanMetadata.isFromContextualReplacement()){
                    AnalyticsReporter.reportPhrasesEvent(
                            PhrasesEventAction.suggested,
                            mMessageId,
                            phrase,
                            null,
                            false,
                            null,
                            false, null, 0, null, 0, 0, 0, 0, 0, 0, 0, 0, serialNumber, null, null, rp.imageSpanMetadata);
                }

                if(rp.isCoveredByImage()) {
                    if (rp.imageSpanMetadata.wasSelectedThroughAniways()) {
                        AnalyticsReporter.reportPhrasesEvent(
                                PhrasesEventAction.sent,
                                mMessageId,
                                phrase,
                                new IconData[]{rp.icon},
                                AniwaysStoreManager.isIconUnlocked(rp.icon),
                                null,
                                false, null, 0, null, 0, 0, 0, 0, 0, 0, 0, 0, serialNumber, null, null, rp.imageSpanMetadata);
                    }
                    // Report icons that were not inserted by aniways
                    else {
                        Log.d(TAG, "Reporting icon from non Aniways keyboard/paste/unknown source");
                        AnalyticsReporter.reportIconNotSelectedThroughAniwaysSent(rp.icon, serialNumber, mMessageId, rp.imageSpanMetadata);
                    }
                }
            }
        }
    }

    private void sendIndicationOfSelectedAnimatedGifs() {
        Spannable spannable = this.getText();
        if(spannable == null){
            return;
        }
        IAniwaysImageSpan spans[] = spannable.getSpans(0, spannable.length(), IAniwaysImageSpan.class);
        if(spans == null || spans.length == 0){
            return;
        }
        for(IAniwaysImageSpan span : spans){
            IconData icon = span.getIcon();
            Object selectionTag = icon.iconSelectionTag;
            if(selectionTag != null){
                if(icon.assetProvider == IAniwaysImageSpan.AssetProvider.Giphy){
                    if(selectionTag instanceof onContentSelectedListener.GiphySelectedData){
                        for (onContentSelectedListener listener : mContentSelectedListeners) {
                            listener.onGiphySelected((onContentSelectedListener.GiphySelectedData)selectionTag);
                        }
                    }
                    else{
                        Log.e(true, TAG, "Encountered non GiphySelectedData selection Tag when provider is Giphy: " + selectionTag.getClass().getName());
                    }
                }
                else{
                    Log.w(true, TAG, "Encountered non Giphy asset provider: " + icon.assetProvider);
                }
            }
        }
    }

    @Override
    public void setText(CharSequence text, BufferType type) {
        Log.d(TAG, "setText Start");

        if(this.isInEditMode()){
            super.setText(text,type);
            return;
        }

        AniwaysStatics.makeSureAniwaysIsInitialized(false);

        try{
            if(text == null){
                text = "";
            }

            Editable oldText;
            // this means that a message is probably being sent, so this EditText is cleared of its content..
            // TODO: think of a better way to know that a message was sent
            try{
                oldText = this.getText();
            }
            // This is because that during construction there is a call to this method
            // and there is an Android bug that causes this Exception
            catch (ClassCastException ex)
            {
                oldText = Editable.Factory.getInstance().newEditable("");
                Log.v(TAG, "set text cast exception caught");
            }
            if (text.length() == 0 && oldText != text){

                if (oldText.length() > 0){
                    if(!AniwaysPrivateConfig.getInstance().appIsCallingOnSendingMessage){
                        this.onSendingMessage();
                    }
                }
            }

            if(oldText != text){
                Log.i(TAG, "Setting a new text");
                this.mMessageId = UUID.randomUUID();
            }
            else{
                Log.i(TAG, "Setting the same text");
            }

            try{
                // In some inits an Edit Text is created in reflection without invoking the ctor and creating the mAniwaysEditTextAndTextViewerCommonPart
                if(mAniwaysEditTextAndTextViewerCommonPart != null){
                    if(text != oldText){
                        text = Editable.Factory.getInstance().newEditable(text);
                        JsonParser parser = AniwaysPhraseReplacementData.getDataParser();
                        if(text.length() > 0){
                            Log.d(TAG, "Start decode");
                            AniwaysIconConverter.decodeMessage(getContext(), (Editable) text, this, null, mAniwaysEditTextAndTextViewerCommonPart, parser, null, false, false, false, false);
                            Log.d(TAG, "End decode");
                        }
                    }
                }
                Log.d(TAG, "superSetText Start");
                super.setText(text, type);
                Log.d(TAG, "superSetText End");
                // In some inits an Edit Text is created in reflection without invoking the ctor and creating the mAniwaysEditTextAndTextViewerCommonPart
                if(mAniwaysEditTextAndTextViewerCommonPart != null){
                    // Do not use getText() because the instances will be different since the set text creates a new editable object
                    mAniwaysEditTextAndTextViewerCommonPart.onSetText(text, oldText);
                }
            }
            catch(IndexOutOfBoundsException ex){
                Log.e(true, TAG, "Caught IndexOutOfBoundsException in setText. Text is: " + text + " . Type is: " + type.toString(), ex);
            }

            Log.d(TAG, "setText End");
        }
        catch(Throwable ex){
            // TODO: fix this!!
            Log.e(true, TAG, "Caught excedption in setText. Text is: " + text, ex);
        }
    }

    public class RecognizedPhrase{
        public Phrase phrase;
        IconData icon;
        IAniwaysImageSpan.ImageSpanMetadata imageSpanMetadata;

        RecognizedPhrase(Phrase p){
            this.phrase = p;
            this.icon = null;
            this.imageSpanMetadata = null;
        }

        boolean isCoveredByImage(){
            return icon != null;
        }
    }

    public ArrayList<RecognizedPhrase> getRecognizedPhrases(boolean includeCoveredByImage){
        Spannable text = this.getText();
        ArrayList<RecognizedPhrase> result = new ArrayList<>();

        if (text == null || text.length() <= 0){
            return result;
        }

        AniwaysSuggestionSpan[] suggestionSpans =  text.getSpans(0, text.length(), AniwaysSuggestionSpan.class);
        if(suggestionSpans != null && suggestionSpans.length > 0){
            for(AniwaysSuggestionSpan ss : suggestionSpans){
                int start = text.getSpanStart(ss);
                int end = text.getSpanEnd(ss);
                RecognizedPhrase recognizedPhrase = new RecognizedPhrase(ss.phrase);
                IAniwaysImageSpan[] imageSpans =  text.getSpans(start, end, IAniwaysImageSpan.class);
                if(imageSpans != null && imageSpans.length > 0){
                    for(IAniwaysImageSpan is : imageSpans){
                        int isStart = text.getSpanStart(is);
                        int isEnd = text.getSpanEnd(is);
                        if(isStart == start && isEnd == end){
                            recognizedPhrase.icon = is.getIcon();
                            recognizedPhrase.imageSpanMetadata = is.getImageSpanMetadata();
                        }
                    }
                }
                if(!recognizedPhrase.isCoveredByImage() || includeCoveredByImage){
                    result.add(recognizedPhrase);
                }
            }
        }

        //Now find images with no suggestion spans..
        IAniwaysImageSpan[] imageSpans =  text.getSpans(0, text.length(), IAniwaysImageSpan.class);
        if(imageSpans != null && imageSpans.length > 0){
            for(IAniwaysImageSpan is : imageSpans){
                int start = text.getSpanStart(is);
                int end = text.getSpanEnd(is);
                RecognizedPhrase recognizedPhrase = new RecognizedPhrase(is.getPhrase());
                suggestionSpans =  text.getSpans(start, end, AniwaysSuggestionSpan.class);
                boolean foundSuggestionSpan = false;
                if(suggestionSpans != null && suggestionSpans.length > 0){
                    for(AniwaysSuggestionSpan ss : suggestionSpans){
                        int ssStart = text.getSpanStart(ss);
                        int ssEnd = text.getSpanEnd(ss);
                        if(ssStart == start && ssEnd == end){
                            foundSuggestionSpan = true;
                            break;
                        }
                    }
                }
                if(!foundSuggestionSpan){
                    recognizedPhrase.icon = is.getIcon();
                    recognizedPhrase.imageSpanMetadata = is.getImageSpanMetadata();

                    if(includeCoveredByImage){
                        result.add(recognizedPhrase);
                    }

                    Log.d(TAG, "Found icon from app");
                    if(!recognizedPhrase.imageSpanMetadata.isFromUnknownKeyboard() && !recognizedPhrase.imageSpanMetadata.isIconSentSeperatelyFromText()){
                        Log.w(false, TAG, "Found icon whitout suggestion span which is not from the app");
                    }
                }
            }
        }

        return result;
    }

    @Override
    public void setSelection(int start, int stop) {
        try{
            mAniwaysEditTextAndTextViewerCommonPart.setSelection(start, stop);
        }
        catch(Throwable ex){
            // TODO: fix this!!
            Log.e(true, TAG, "Caught excedption in setSelection. Start is: " + start + ". stop is: " + stop, ex);
        }
    }

    @Override
    public void setSelection(int index) {
        try{
            // The 'if' is here since this is called from the c'tor
            if(mAniwaysEditTextAndTextViewerCommonPart != null){
                mAniwaysEditTextAndTextViewerCommonPart.setSelection(index);
            }
        }
        catch(Throwable ex){
            // TODO: fix this!!
            Log.e(true, TAG, "Caught excedption in setSelection. index is: " + index, ex);
        }
    }

    @Override
    public void extendSelection(int index) {
        try{
            mAniwaysEditTextAndTextViewerCommonPart.extendSelection(index);
        }
        catch(Throwable ex){
            // TODO: fix this!!
            Log.e(true, TAG, "Caught excedption in extendSelection. index is: " + index, ex);
        }
    }

    @Override
    public void selectAll() {
        try{
            mAniwaysEditTextAndTextViewerCommonPart.selectAll();
        }
        catch(Throwable ex){
            // TODO: fix this!!
            Log.e(true, TAG, "Caught excedption in selectAll", ex);
        }
    }

    @Override
    public void removeTextChangedListener(TextWatcher watcher){
        if(this.isInEditMode() || mAniwaysEditTextAndTextViewerCommonPart == null) {
            Log.w(false, TAG, "not removing text change listeners through aniways since mAniwaysEditTextAndTextViewerCommonPart is null");
            super.removeTextChangedListener(watcher);
            return;
        }
        this.mAniwaysEditTextAndTextViewerCommonPart.removeTextChangedListener(watcher);
    }

    @Override
    public void addTextChangedListener(TextWatcher watcher){
        if(this.isInEditMode() || mAniwaysEditTextAndTextViewerCommonPart == null) {
            Log.w(false, TAG, "not adding text change listeners through aniways since mAniwaysEditTextAndTextViewerCommonPart is null");
            super.addTextChangedListener(watcher);
            return;
        }
        this.mAniwaysEditTextAndTextViewerCommonPart.addTextChangedListener(watcher);
    }

    @Override
    public void callSuperSetSelection(int start, int stop) {
        super.setSelection(start, stop);

    }

    @Override
    public void callSuperSetSelection(int index) {
        super.setSelection(index);
    }

    @Override
    public void callSuperExtendSelection(int index) {
        super.extendSelection(index);
    }

    @Override
    public void callSuperSelectAll() {
        super.selectAll();
    }

    @Override
    public void callSuperRemoveTextChangedListener(TextWatcher watcher){
        super.removeTextChangedListener(watcher);
    }

    @Override
    public void callSuperAddTextChangedListener(TextWatcher watcher){
        super.addTextChangedListener(watcher);
        super.onDetachedFromWindow();
    }

    @Override
    protected void onDetachedFromWindow() {
        this.mAniwaysEditTextAndTextViewerCommonPart.onDetachFromWindow();

        super.onDetachedFromWindow();
    }

    @Override
    protected void onLayout(boolean changed, int left, int top, int right, int bottom) {
        super.onLayout(changed, left, top, right, bottom);

        this.mAniwaysEditTextAndTextViewerCommonPart.onLayoutCalled();
    }

    /**
     * Return the point (in pixels) of the received char position as it is displayed
     * relative to the upper left corner of the widget, or lower left if fromTop == false.
     * It accounts for scroll position and paddings
     * @return !!Be careful, can return null if there was a recent layout change!!
     */
    Point getPointOfPositionInText(int position, boolean fromTop) {
        return this.mAniwaysEditTextAndTextViewerCommonPart.getPointOfPositionInText(position, fromTop);
    }

    public void insetTextCoveredWithIconAfterPhrase(IconData selectedIcon, AniwaysSuggestionSpan suggestionSpan, Phrase phrase, IAniwaysImageSpan.ImageSpanMetadata imageSpanMetadata) {

        Editable editable = this.getText();
        if(editable == null){
            return;
        }

        int oldSelectionStart = getSelectionStart();
        int oldSelectionEnd = getSelectionEnd();

        int spanEnd = editable.getSpanEnd(suggestionSpan);

        if(spanEnd < 0){
            Log.e(true, TAG, "SuggestionSpan not in Text, so not adding icon. Is Null: " + (suggestionSpan == null) + ". Text: " + editable.toString());
            return;
        }

        int marginBetweenPartToReplacAndPhraseEnd = phrase.getMarginBetweenEndOfPartToReplaceToPhraseEnd();
        //TODO: Make sure that indeed the text we are supposed to see in the margin is there..
        int phraseEnd = spanEnd + marginBetweenPartToReplacAndPhraseEnd;

        // Need to remove in order for the addition of the text will not be highlighted and cause the tutorial to appear.
        // The user will not understand what highlighted text the tutorial is talking about since it is covered up with a picture..
        // We add the listener back only after the text is covered in picture since the marker inserter doesn't add the marker span if there
        // is a picture there
        mAniwaysEditTextAndTextViewerCommonPart.removeTheAniwaysTextWatcher();

        // Insert 2 spaces after phrase and put new phrase between them
        // unless there is already a space there, in which case we add only one
        String spacesToAdd = "  ";
        try {
            if (phraseEnd != editable.length() && editable.subSequence(phraseEnd, phraseEnd + 1).toString().equals(" ")) {
                spacesToAdd = " ";
            }
        }
        catch(Throwable ex){
            Log.e(true, TAG, "Could not add spaces to make room for icon.", ex);
        }
        editable.insert(phraseEnd, spacesToAdd);
        // Insert the icon itself
        String phraseName = phrase.getName();
        getText().insert(phraseEnd + 1, phraseName);
        int selectionOffset = phraseName.length() + spacesToAdd.length();
        if(oldSelectionStart >= phraseEnd){
            setSelection(oldSelectionStart + selectionOffset, oldSelectionEnd + selectionOffset);
        }
        else if(oldSelectionEnd >= phraseEnd){
            setSelection(oldSelectionStart, oldSelectionEnd + selectionOffset);
        }
        int indexOfPartToReplace = phraseEnd + 1 + phrase.getIndexOfPartToReplaceInPhrase();

        // Add a suggestion span so the icon could later be replaced
        AniwaysSuggestionSpan ass = new AniwaysSuggestionSpan(phrase, this, phrase.getPartToReplace());
        ass.spawninSpan = suggestionSpan;
        getText().setSpan(ass, indexOfPartToReplace, indexOfPartToReplace + phrase.getPartToReplace().length(), Spannable.SPAN_EXCLUSIVE_EXCLUSIVE);

        // TODO: This is a hack - we if we werent supposed to add the suggestion span, then we tell next function to remove it..
        replaceWithIcon(selectedIcon, null, ass, imageSpanMetadata, true);

        Log.d(TAG, "in insetTextCoveredWithIcon - adding back text changed listener..");
        mAniwaysEditTextAndTextViewerCommonPart.addTheAniwaysTextWatcher();
    }

    public void insetTextCoveredWithIcon(IconData selectedIcon, int selectionStart, int selectionEnd, Phrase phrase, IAniwaysImageSpan.ImageSpanMetadata imageSpanMetadata, boolean addSuggestionSpan) {
        Log.d(TAG, "in insetTextCoveredWithIcon");

        // Need to remove in order for the addition of the text will not be highlighted and cause the tutorial to appear.
        // The user will not understand what highlighted text the tutorial is talking about since it is covered up with a picture..
        // We add the listener back only after the text is covered in picture since the marker inserter doesn't add the marker span if there
        // is a picture there
        mAniwaysEditTextAndTextViewerCommonPart.removeTheAniwaysTextWatcher();

        // TODO: What happens if there are Aniways icons and spans here, need to remove!!
        String phraseName = phrase.getName();
        getText().replace(selectionStart, selectionEnd, phraseName, 0, phraseName.length());
        setSelection(selectionStart + phraseName.length());
        int indexOfPartToReplace = selectionStart + phrase.getIndexOfPartToReplaceInPhrase();

        // Add a suggestion span so the icon could later be replaced
        AniwaysSuggestionSpan ass = new AniwaysSuggestionSpan(phrase, this, phrase.getPartToReplace());
        getText().setSpan(ass, indexOfPartToReplace, indexOfPartToReplace + phrase.getPartToReplace().length(), Spannable.SPAN_EXCLUSIVE_EXCLUSIVE);

        // TODO: This is a hack - we if we werent supposed to add the suggestion span, then we tell next function to remove it..
        replaceWithIcon(selectedIcon, null, ass, imageSpanMetadata, !addSuggestionSpan);

        Log.d(TAG, "in insetTextCoveredWithIcon - adding back text changed listener..");
        mAniwaysEditTextAndTextViewerCommonPart.addTheAniwaysTextWatcher();
    }

    @Override
    public IAniwaysTextWatcher addTheAniwaysTextWatcher() {
        return this.mAniwaysEditTextAndTextViewerCommonPart.addTheAniwaysTextWatcher();
    }

    @Override
    public IAniwaysTextWatcher removeTheAniwaysTextWatcher() {
        return this.mAniwaysEditTextAndTextViewerCommonPart.removeTheAniwaysTextWatcher();
    }

    public UUID getMessageId() {
        return this.mMessageId;
    }

    public IAniwaysTextContainer getTextContainer() {
        return this.mAniwaysEditTextAndTextViewerCommonPart;
    }
}