/**
 *
 */
package com.aniways;

import java.security.InvalidParameterException;

import android.annotation.SuppressLint;
import android.app.ActivityManager;
import android.app.ActivityManager.RunningAppProcessInfo;
import android.content.Context;
import android.content.SharedPreferences;
import android.content.SharedPreferences.Editor;
import android.os.Handler;
import android.os.Looper;
import android.text.Editable;
import android.text.Spannable;
import android.text.TextUtils;
import android.view.View;
import android.view.ViewGroup;
import android.widget.LinearLayout;

import com.aniways.analytics.AnalyticsReporter;
import com.aniways.analytics.NonThrowingRunnable;
import com.aniways.billing.IAniwaysPaymentCallback;
import com.aniways.data.AniwaysBackendSyncChecker;
import com.aniways.data.AniwaysPhraseReplacementData;
import com.aniways.data.AniwaysPrivateConfig;
import com.aniways.data.AniwaysStatics;
import com.aniways.data.AppData;
import com.aniways.data.SettingsKeys;
import com.aniways.emoticons.button.AniwaysEmoticonsButtonMaker;
import com.aniways.quick.action.QuickAction;
import com.aniways.service.utils.AniwaysServiceUtils;
import com.aniways.ui.AniwaysUiUtil;
import com.aniways.ui.views.AniwaysPopupWindow;

/**
 * @author Shai
 *
 */
public class Aniways {

	private static final String TAG = "Aniways";

	private static boolean isInit = false;

	private static Context sApplicationContext;

	private static Configuration sConfiguration;

	private static Context sContext;

	public static IAniwaysPaymentCallback mPaymentCallback;

	/**
	 * Initializes Aniways and starts the Aniways service which updates the phrases and emoticons.
	 * Needs to be called from the onCreate() of all entry point activities to the application, like this:
	 * if(!Aniways.isInit()){
	 *		Aniways.init(this);
	 *	}
	 * @param context the app context.
	 */
	public static synchronized void init(Context context){
		if(isInit()){
			Log.i(TAG, "Not initializing Aniways because it is already initialized");
			return;
		}

		// If this process is the Aniways service process, then do not init
		// (this can happen is the app inits Aniways in the onCreate of the Application class
		// which is called whenever a process in the app starts)
		String currentProcName = "";
		try{
			int pid = android.os.Process.myPid();
			ActivityManager manager = (ActivityManager) context.getSystemService(Context.ACTIVITY_SERVICE);
			for (RunningAppProcessInfo processInfo : manager.getRunningAppProcesses())
			{
				if (processInfo.pid == pid)
				{
					currentProcName = processInfo.processName;
				}
			}
		}
		catch(Throwable ex){
			android.util.Log.e(TAG, "Caught an Exception in closeAllOpenedPopups", ex);
			ex.printStackTrace();
		}
		if(currentProcName != null && currentProcName.endsWith(":AniwaysService")){
			Log.i(TAG, "Not initializing Aniways because it is called from the Aniways Service");
			return;
		}


		android.util.Log.i(TAG, "Initializing Aniways version " + AniwaysServiceUtils.SDK_VERSION);

		if(context == null){
			throw new IllegalArgumentException("Context cannot be null");
		}

		sContext = context;

		sApplicationContext = context.getApplicationContext();

		AniwaysStatics.init(sApplicationContext, sContext, false);

		isInit = true;
	}

	/**
	 * @return whether Aniways is already initialized (could have been initialized from the onCreate() method of another activity).
	 */
	public static boolean isInit(){
		return isInit;
	}

	/**
	 * Closes all Aniways popups if any are showing (including the contextual suggestions, icon info and emoticons button popups).
	 * Returns true if it actually closed something and false if otherwise.
	 */
	public static boolean closeAllOpenedPopups(){
		AniwaysStatics.makeSureAniwaysIsInitialized(false);
		try{
            // QuickAction.dismissAllOpenQuickActions()
			return AniwaysPopupWindow.dismissAllOpenPopups() | AniwaysEmoticonsButtonMaker.dismissAllOpenedButtonPopups();
		}
		catch(Throwable ex){
			Log.e(true, TAG, "Caught an Exception in closeAllOpenedPopups", ex);
			return false;
		}
	}

	/**
	 * Gets the original message string without the Aniways Suffixes.
	 * @param context - the activity context
	 * @param source - the source string which contains the Aniways suffix
	 * @return the original message string without the Aniways Suffixes.
	 */
	public static String getOriginalString(Context context, Spannable source) {
		AniwaysStatics.makeSureAniwaysIsInitialized(false);

		if(source == null){
			source = Editable.Factory.getInstance().newEditable("");
		}

		try{
			return AniwaysIconConverter.getOriginalString(source, true, true);
		}
		catch(Throwable ex){
			Log.e(true, TAG, "Caught an Exception in getOriginalString", ex);
			return source.toString();
		}
	}

	public static String removeAniwaysUnicodes(String text){
		AniwaysStatics.makeSureAniwaysIsInitialized(false);

		try{
			Log.i(TAG, "removeAniwaysUnicodes called on text: " + text + ". Length: " + (TextUtils.isEmpty(text) ? 0 : text.length()));

			if(TextUtils.isEmpty(text)){
				return "";
			}

			String result = AniwaysPrivateConfig.getInstance().decoderCodepointsPattern.matcher(text).replaceAll("");
			Log.i(TAG, "removeAniwaysUnicodes returns result: " + result + ". Length: " + (TextUtils.isEmpty(result) ? 0 : result.length()));
			return result;
		}
		catch(Throwable ex){
			Log.e(true, TAG, "Caught an Exception in removeAniwaysUnicodes, returning original text", ex);
			return text;
		}
	}

	/**
	 * Receives a text with Aniways Icons in it and returns a string in which the icons are encoded as text.
	 * This string can then be sent to the other side.
	 * @param textWithIcons - the text which contains the Aniways icons.
	 * @return a string in which the icons are encoded as text. This string can be then sent to the other side.
	 */
	public static String encodeMessage(Spannable textWithIcons){
		if(textWithIcons == null){
			Log.w(false, TAG, "Trying to encode a null message. Returning null");
			return null;
		}

		AniwaysStatics.makeSureAniwaysIsInitialized(false);
		try{
			return AniwaysIconConverter.encodeMessage(textWithIcons, sApplicationContext, true, null, false);
		}
		catch(Throwable ex){
			Log.e(true, TAG, "Caught exception while encoding message. Returning original message", ex);
			return textWithIcons.toString();
		}
	}

    // Used for display in notification menu
    public static CharSequence getDecodedMessageWithEmojiFallbacksForIcons(CharSequence text){
        AniwaysStatics.makeSureAniwaysIsInitialized(false);

        if(text == null){
            text = "";
        }

        Editable editable = Editable.Factory.getInstance().newEditable(text);

        try{

            AniwaysIconConverter.decodeMessage(sApplicationContext, editable, null, null, null, AniwaysPhraseReplacementData.getDataParser(), null, true, false, true, false);
            return AniwaysIconConverter.getOriginalString(editable, true, false);
        }
        catch(Throwable ex){
            Log.e(true, TAG, "Caught an Exception in decodeMessage", ex);
        }
        return editable;
    }

	public static Editable decodeMessage(CharSequence text, IIconInfoDisplayer infoDisplayer, IAniwaysTextContainer textContainer, boolean useSmallIcons){
		AniwaysStatics.makeSureAniwaysIsInitialized(false);

		if(text == null){
			text = "";
		}

		Editable editable = Editable.Factory.getInstance().newEditable(text.toString());

		try{
			AniwaysIconConverter.decodeMessage(sApplicationContext, editable, null, infoDisplayer, textContainer, AniwaysPhraseReplacementData.getDataParser(), null, useSmallIcons, false, false, false);
		}
		catch(Throwable ex){
			Log.e(true, TAG, "Caught an Exception in decodeMessage", ex);
		}
		return editable;
	}

    public static String getAniwaysStickerUrl(CharSequence text){
        AniwaysStatics.makeSureAniwaysIsInitialized(false);

        try{
            return AniwaysIconConverter.getAniwaysStickerUrl(text);
        }
        catch(Throwable ex){
            Log.e(true, TAG, "Caught an Exception in getAniwaysStickerUrl", ex);
            return null;
        }
    }

    public static boolean isMessageAniwaysSticker(CharSequence text){
        AniwaysStatics.makeSureAniwaysIsInitialized(false);

        try{
            return AniwaysIconConverter.getAniwaysStickerUrl(text) != null;
        }
        catch(Throwable ex){
            Log.e(true, TAG, "Caught an Exception in isMessageAniwaysSticker", ex);
            return false;
        }
    }

	public static boolean makeButtonAniwaysEmoticonsButton(View button, LinearLayout aniwaysEmoticonsButtonPlaceholder, final ViewGroup parentLayout, AniwaysEditText editText, OnAniwaysEmoticonsKeyboardListener listener, boolean usePadding){
		AniwaysStatics.makeSureAniwaysIsInitialized(false);

		try{
			return new AniwaysEmoticonsButtonMaker().makeButtonAniwaysEmoticonsButton(button, aniwaysEmoticonsButtonPlaceholder, parentLayout, editText, listener, usePadding);
		}
		catch(Throwable ex){
			Log.e(true, TAG, "Caught an Exception in makeButtonAniwaysEmoticonsButton", ex);
			return false;
		}
	}


	public static boolean makeButtonAniwaysEmoticonsButton(View button, final ViewGroup parentLayout, AniwaysEditText editText, OnAniwaysEmoticonsKeyboardListener listener, boolean usePadding){
		AniwaysStatics.makeSureAniwaysIsInitialized(false);

		return makeButtonAniwaysEmoticonsButton(button, null, parentLayout, editText, listener, usePadding);
	}


	public static void registerPaymentCallback(IAniwaysPaymentCallback callback){
		mPaymentCallback = callback;
	}

	public static IAniwaysPaymentCallback getPaymentCallback(){
		return mPaymentCallback;
	}

	public static Configuration getConfiguration(Context ctx){
		if(ctx == null){
			throw new InvalidParameterException("ctx must not be null");
		}

		if(sConfiguration == null){
			sConfiguration = new Configuration(ctx);
		}
		return sConfiguration;
	}

    public static String serializeMessage(Editable text) {
        if(text == null){
            Log.w(false, TAG, "Trying to serialize a null message. Returning null");
            return null;
        }

        AniwaysStatics.makeSureAniwaysIsInitialized(false);
        try{
            // Encoding in a way that will reserve the text as is - no autoreplace text to icons icons and no encoding icons with emoji replacement
            // as emoji without the text below, so the message can be reconstructed.
            // TODO: Make this a real serialize and add data that we normally do not have in encode in order to be able to really preserve all the data
            // in the span objects
            return AniwaysIconConverter.encodeMessage(text, sApplicationContext, false, null, true);
        }
        catch(Throwable ex){
            Log.e(true, TAG, "Caught exception while serializing message. Returning original message", ex);
            return text.toString();
        }
    }

    public interface OnAniwaysEmoticonsKeyboardListener{
		public void onShow();
		public void onHide();
	}

	public static class Configuration{


		private static final String KEY_CONTEXTUAL_ANIMATIONS_SUGGESTIONS = "com.aniways.CONTEXTUAL_ANIMATIONS_SUGGESTIONS";
		private static final String KEY_CONTEXTUAL_MUSIC_SUGGESTIONS = "com.aniways.CONTEXTUAL_MUSIC_SUGGESTIONS";
		private static final String KEY_CONTEXTUAL_LOCATION_BASED_SUGGESTIONS = "com.aniways.CONTEXTUAL_LOCATION_BASED_SUGGESTIONS";
		private static final String KEY_CONTEXTUAL_CONTENT_SUGGESTIONS = "com.aniways.CONTEXTUAL_CONTENT_SUGGESTIONS";
		private static final String KEY_CONTEXTUAL_SUGGESTIONS = "com.aniways.CONTEXTUAL_SUGGESTIONS";

		private Context mContext;
		private boolean mPerformUpdate = true;
		private boolean mValueUpdated = false;

		public Configuration(Context ctx) {
			mContext = ctx.getApplicationContext();
			AniwaysStatics.init(mContext, ctx, true);
		}

        //TODO: Take all keys here from SettingsKeys
		public void enableContextualIconSuggestions(boolean enable){
			Log.i(TAG, "enableContextualIconSuggestions called with param: " + enable);
			try{
				setConfigProperty(SettingsKeys.KEY_CONTEXTUAL_ICON_SUGGESTIONS, enable, null);
			}
			catch(Throwable tr){
				Log.e(true, TAG, "Caught exception in enableContextualIconSuggestions. Value: " + enable, tr);
			}
		}
		public void enableContextualAnimationsSuggestions(boolean enable){
			Log.i(TAG, "enableContextualAnimationsSuggestions called with param: " + enable);
			setConfigProperty(KEY_CONTEXTUAL_ANIMATIONS_SUGGESTIONS, enable, null);
		}
		public void enableContextualMusicSuggestions(boolean enable){
			Log.i(TAG, "enableContextualMusicSuggestions called with param: " + enable);
			setConfigProperty(KEY_CONTEXTUAL_MUSIC_SUGGESTIONS, enable, null);
		}
		public void enableContextualLocationBasedSuggestions(boolean enable){
			Log.i(TAG, "enableContextualLocationBasedSuggestions called with param: " + enable);
			setConfigProperty(KEY_CONTEXTUAL_LOCATION_BASED_SUGGESTIONS, enable, null);
		}
		// Does not include the interactive experience..
		public void enableContextualContentSuggestions(final boolean enable){
			Log.i(TAG, "enableContextualContentSuggestions called with param: " + enable);
			setConfigProperty(KEY_CONTEXTUAL_CONTENT_SUGGESTIONS, enable, new Runnable(){
				@Override
				public void run(){
					enableContextualIconSuggestions(enable);
					enableContextualAnimationsSuggestions(enable);
					enableContextualMusicSuggestions(enable);
				}
			});
		}
		// Includes both content and interactive experience
		public void enableContextualSuggestions(final boolean enable){
			Log.i(TAG, "enableContextualSuggestions called with param: " + enable);
			setConfigProperty(KEY_CONTEXTUAL_SUGGESTIONS, enable, new Runnable(){
				@Override
				public void run(){
					enableContextualContentSuggestions(enable);
					enableContextualLocationBasedSuggestions(enable);
				}
			});
		}

        public void setIconsInTextViewSize(int iconSizeInDp) {
            Log.i(TAG, "setIconsInTextViewSize called with param: " + iconSizeInDp);
            setConfigProperty(SettingsKeys.KEY_ICON_IN_TEXTVIEW_SIZE, iconSizeInDp, null);
        }

        public void enableAutoPopup(boolean enable) {
            Log.i(TAG, "enableAutoPopup called with param: " + enable);
            setConfigProperty(SettingsKeys.KEY_AUTOMATIC_POPUP, enable, null);
        }

		public boolean isContextualIconSuggestionsEnabled(){
			try{
				boolean result = AniwaysPrivateConfig.getInstance().contextualIconSuggestionsEnabled;
				Log.i(TAG, "isContextualIconSuggestionsEnabled called. Returning:  " + result);
				return result;
			}
			catch(Throwable tr){
				Log.e(true, TAG, "Caught exception in isContextualIconSuggestionsEnabled", tr);
				return true;
			}
		}
		public boolean isContextualAnimationSuggestionsEnabled(){
			try{
				boolean result = AniwaysPrivateConfig.getInstance().contextualAnimationsSuggestionsEnabled;
				Log.i(TAG, "isContextualAnimationSuggestionsEnabled called. Returning:  " + result);
				return result;
			}
			catch(Throwable tr){
				Log.e(true, TAG, "Caught exception in isContextualAnimationSuggestionsEnabled", tr);
				return true;
			}
		}
		public boolean isContextualMusicSuggestionsEnabled(){
			try{
				boolean result = AniwaysPrivateConfig.getInstance().contextualMusicSuggestionsEnabled;
				Log.i(TAG, "isContextualMusicSuggestionsEnabled called. Returning:  " + result);
				return result;
			}
			catch(Throwable tr){
				Log.e(true, TAG, "Caught exception in isContextualMusicSuggestionsEnabled", tr);
				return true;
			}
		}
		public boolean isContextualLocationBasedSuggestionsEnabled(){
			try{
				boolean result = AniwaysPrivateConfig.getInstance().contextualLocationBasedSuggestionsEnabled;
				Log.i(TAG, "isContextualLocationBasedSuggestionsEnabled called. Returning:  " + result);
				return result;
			}
			catch(Throwable tr){
				Log.e(true, TAG, "Caught exception in isContextualLocationBasedSuggestionsEnabled", tr);
				return true;
			}
		}
		// Does not include the interactive experience..
		public boolean isContextualContentSuggestionsEnabled(){
			try{
				boolean result = AniwaysPrivateConfig.getInstance().contextualContentSuggestionsEnabled;
				Log.i(TAG, "isContextualContentSuggestionsEnabled called. Returning:  " + result);
				return result;
			}
			catch(Throwable tr){
				Log.e(true, TAG, "Caught exception in isContextualContentSuggestionsEnabled", tr);
				return true;
			}
		}
		// Includes both content and interactive experience
		public boolean isContextualSuggestionsEnabled(){
			try{
				boolean result = AniwaysPrivateConfig.getInstance().contextualSuggestionsEnabled;
				Log.i(TAG, "isContextualSuggestionsEnabled called. Returning:  " + result);
				return result;
			}
			catch(Throwable tr){
				Log.e(true, TAG, "Caught exception in isContextualSuggestionsEnabled", tr);
				return true;
			}
		}

        public float getIconsInTextViewSize() {
            try{
                //Since the height and width are the same, we take the width
                float result = AniwaysPrivateConfig.getInstance().iconInTextViewWidth;
                result = AniwaysUiUtil.convertPixelsToDips(result);
                Log.i(TAG, "getIconsInTextViewSize called. Returning:  " + result);
                return result;
            }
            catch(Throwable tr){
                Log.e(true, TAG, "Caught exception in getIconsInTextViewSize", tr);
                return 0;
            }
        }

        public boolean isAutoPopupEnabled() {
            try{
                AniwaysPrivateConfig.SuggestionMode mode = AniwaysPrivateConfig.getInstance().suggestionMode;
                boolean result = (mode == AniwaysPrivateConfig.SuggestionMode.AutoDisplaySuggestions);
                Log.i(TAG, "isAutoPopupEnabled called. Returning:  " + result);
                return result;
            }
            catch(Throwable tr){
                Log.e(true, TAG, "Caught exception in isContextualContentSuggestionsEnabled", tr);
                return true;
            }
        }

		private void updatePrivateConfig() {
			Log.i(TAG, "Updating private config");
			Handler handler = new Handler(Looper.getMainLooper());
			// Parse the config file on a different thread
			handler.post(new NonThrowingRunnable(TAG, "Aniways config parsing", "") {
				@SuppressLint("NewApi")
				@Override
				public void innerRun(){
					AniwaysBackendSyncChecker.parseConfingAndKeywordsIfNecessary(sContext, true, true);
				}
			});

		}

		private synchronized void setConfigProperty(String key, boolean value, Runnable postSet) {
			try{
				Log.i(TAG, "Setting config property with key: " + key + ". To: " + value + ". Runnable: " + postSet + ". Should update: " + mPerformUpdate);
				boolean shouldUpdate = mPerformUpdate;
				mPerformUpdate = false;

				// get old value, update only if changed
				Boolean oldValue = getConfigProperty(key);
				Log.i(TAG, "Old value is: " + oldValue);
				if(oldValue == null || value != oldValue.booleanValue()){
					Log.i(TAG, "Old value != new value");
					// open the shared preferences
					SharedPreferences prefs =
							mContext.getSharedPreferences(AniwaysServiceUtils.SHARED_PREFERENCES, Utils.getSharedPreferencesFlags());
					Editor edit = prefs.edit();
					// write the new value
					edit.putBoolean(key, value);
					edit.commit();

					mValueUpdated = true;

					AnalyticsReporter.reportUserConfigChanged(key, oldValue, value);

					// Run all the other needed 'set' ops
					if(postSet != null){
						Log.i(TAG, "Calling post set");
						postSet.run();
					}
				}

				Log.i(TAG, "Should update: " + shouldUpdate + ". Value updated: " + mValueUpdated);
				if(shouldUpdate)
				{
					this.mPerformUpdate = true;
					if(mValueUpdated){
						// Update private config
						mValueUpdated = false;
						updatePrivateConfig();
					}
				}
			}
			catch(Throwable tr){
				Log.e(true, TAG, "Caught exception in setConfigProperty. Key: " + key + ". Value: " + value, tr);
				// So it will not get stuck..
				mPerformUpdate = true;
			}
		}

        //TODO: might have code in common with the setConfigProperty method which takes boolean
        private synchronized void setConfigProperty(String key, int value, Runnable postSet) {
            try{
                Log.i(TAG, "Setting config property with key: " + key + ". To: " + value + ". Runnable: " + postSet + ". Should update: " + mPerformUpdate);
                boolean shouldUpdate = mPerformUpdate;
                mPerformUpdate = false;

                // get old value, update only if changed
                Integer oldValue = getIntConfigProperty(key);
                Log.i(TAG, "Old value is: " + oldValue);
                if(oldValue == null || value != oldValue.intValue()){
                    Log.i(TAG, "Old value != new value");
                    // open the shared preferences
                    SharedPreferences prefs =
                            mContext.getSharedPreferences(AniwaysServiceUtils.SHARED_PREFERENCES, Utils.getSharedPreferencesFlags());
                    Editor edit = prefs.edit();
                    // write the new value
                    edit.putInt(key, value);
                    edit.commit();

                    mValueUpdated = true;

                    AnalyticsReporter.reportUserConfigChanged(key, oldValue, value);

                    // Run all the other needed 'set' ops
                    if(postSet != null){
                        Log.i(TAG, "Calling post set");
                        postSet.run();
                    }
                }

                Log.i(TAG, "Should update: " + shouldUpdate + ". Value updated: " + mValueUpdated);
                if(shouldUpdate)
                {
                    this.mPerformUpdate = true;
                    if(mValueUpdated){
                        // Update private config
                        mValueUpdated = false;
                        updatePrivateConfig();
                    }
                }
            }
            catch(Throwable tr){
                Log.e(true, TAG, "Caught exception in setConfigProperty. Key: " + key + ". Value: " + value, tr);
                // So it will not get stuck..
                mPerformUpdate = true;
            }
        }

		private synchronized Boolean getConfigProperty(String key){
			// open the shared preferences
			SharedPreferences prefs =
					mContext.getSharedPreferences(AniwaysServiceUtils.SHARED_PREFERENCES, Utils.getSharedPreferencesFlags());

			if(!prefs.contains(key)){
				return null;
			}

			// return the values that stored there, in case that no value
			// is stored return the default
			return prefs.getBoolean(key, true);
		}

        private synchronized Integer getIntConfigProperty(String key){
            // open the shared preferences
            SharedPreferences prefs =
                    mContext.getSharedPreferences(AniwaysServiceUtils.SHARED_PREFERENCES, Utils.getSharedPreferencesFlags());

            if(!prefs.contains(key)){
                return null;
            }

            // return the values that stored there, in case that no value
            // is stored return the default
            int result = prefs.getInt(key, -1);
            if(result == -1){
                return null;
            }
            return result;
        }

		private synchronized static boolean getConfigProperty(Context ctx, String key, boolean defaultValue){
			// open the shared preferences
			return AppData.getBoolean(key,defaultValue);
		}

		public static void setPrivateConfigProperties(Context ctx, AniwaysPrivateConfig config) {
			try{
                boolean isAutoPopupEnabled = getConfigProperty(ctx, SettingsKeys.KEY_AUTOMATIC_POPUP, config.suggestionMode == AniwaysPrivateConfig.SuggestionMode.AutoDisplaySuggestions);
                config.suggestionMode = isAutoPopupEnabled ? AniwaysPrivateConfig.SuggestionMode.AutoDisplaySuggestions : AniwaysPrivateConfig.SuggestionMode.Manual;
				config.contextualIconSuggestionsEnabled = getConfigProperty(ctx, SettingsKeys.KEY_CONTEXTUAL_ICON_SUGGESTIONS, config.contextualIconSuggestionsEnabled);
				config.contextualAnimationsSuggestionsEnabled = getConfigProperty(ctx, KEY_CONTEXTUAL_ANIMATIONS_SUGGESTIONS, config.contextualAnimationsSuggestionsEnabled);
				config.contextualMusicSuggestionsEnabled = getConfigProperty(ctx, KEY_CONTEXTUAL_MUSIC_SUGGESTIONS, config.contextualMusicSuggestionsEnabled);
				config.contextualLocationBasedSuggestionsEnabled = getConfigProperty(ctx, KEY_CONTEXTUAL_LOCATION_BASED_SUGGESTIONS, config.contextualLocationBasedSuggestionsEnabled);
				config.contextualContentSuggestionsEnabled = getConfigProperty(ctx, KEY_CONTEXTUAL_CONTENT_SUGGESTIONS, config.contextualContentSuggestionsEnabled);
				config.contextualSuggestionsEnabled = getConfigProperty(ctx, KEY_CONTEXTUAL_SUGGESTIONS, config.contextualSuggestionsEnabled);

                //Currently user can only a square icon size in the settings screen. hence the heigh=width.
                config.iconInTextViewHeight = AppData.getInt(SettingsKeys.KEY_ICON_IN_TEXTVIEW_SIZE, config.iconInTextViewHeight);
                config.iconInTextViewWidth = AppData.getInt(SettingsKeys.KEY_ICON_IN_TEXTVIEW_SIZE, config.iconInTextViewWidth);
			}
			catch(Throwable tr){
				Log.e(true, TAG, "Caught exception while parsing user config", tr);
			}
		}

		/**
		 * Use this in scenarios where you want Aniways to suspend N/W usage (for example, a VOIP call on a low bandwidth connection).
		 * Make sure to call resumeNetworkUse() later, or the Aniways will essentially be disabled
		 */
		public static void suspendNetworkUse(){
			//TODO: suspend the N/W usage
		}

		/**
		 * Resumes Aniways network use, after a call to suspendNetworkUse() suspended it.
		 */
		public static void resumeNetworkUse(){
			//TODO: resume the N/W usage
		}

		public static boolean shouldDisplayDecodedMessageAsSticker(Spannable message){
			if(message == null){
				Log.e(true, TAG, "Message is null");
				return false;
			}

			IAniwaysImageSpan[] spans = message.getSpans(0, message.length(), IAniwaysImageSpan.class);
			if(spans == null){
				return false;
			}
			if(spans.length != 1){
				return false;
			}
			IAniwaysImageSpan span = spans[0];
			if (message.getSpanStart(span) != 0){
				return false;
			}
			if (message.getSpanEnd(span) != message.length()){
				return false;
			}
			return true;
		}
    }
}
