package com.aniways.ui.views;


import android.content.Context;
import android.content.res.Resources;
import android.graphics.Color;
import android.util.AttributeSet;
import android.view.View;
import android.view.ViewGroup;
import android.view.animation.OvershootInterpolator;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;

import com.aniways.AssetType;
import com.aniways.R;
import com.aniways.data.AniwaysPrivateConfig;
import com.aniways.nineoldandroids.animation.Animator;
import com.aniways.nineoldandroids.animation.AnimatorSet;
import com.aniways.nineoldandroids.animation.ObjectAnimator;
import com.aniways.nineoldandroids.view.ViewHelper;
import com.aniways.quick.action.IAniwaysPopupView;
import com.aniways.quick.action.PhraseContextualContentViewProvider;
import com.aniways.ui.AniwaysUiUtil;

/**
 * Encapsulates the visuals and behaviors of a single asset type's viewing card.
 */
public class ContextualAssetView extends LinearLayout
{
    public static final float IconMaxAlpha = 0.3f;

    public int index;
    private AniwaysSlidingLayer slidingLayer;
    private View cardHolder;
    private View cardBackgroundView;
    private ImageView cardIcon;
    private RelativeLayout innerContentHolder;
    private int cardBaseColor;
    private boolean isContentOverSized;
    private int cardMinHeight;
    private int hintHeight;
    private int winkWidth;
    private View cardIconContainer;
    private int cardIconHolderWidth;
    private int cardIconHolderOverlapWidth;
    private IAniwaysPopupView content;

    public ContextualAssetView(Context context)
    {
        super(context);
    }

    public ContextualAssetView(Context context, AttributeSet attrs)
    {
        super(context, attrs);
    }

    @Override
    public void onFinishInflate()
    {
        super.onFinishInflate();

        Resources res = getResources();
        this.cardBaseColor = res.getColor(R.color.aniways_contextual_asset_card_background_color);

        this.slidingLayer = (AniwaysSlidingLayer)findViewById(R.id.aniways_contextual_asset_view);

        this.cardBackgroundView = findViewById(R.id.aniways_contextual_card_background);
        this.cardIcon = (ImageView)findViewById(R.id.aniways_contextual_card_icon);
        this.cardIconContainer = findViewById(R.id.aniways_contextual_card_icon_container);

        this.innerContentHolder = (RelativeLayout)findViewById(R.id.aniways_contextual_card_content_holder);
        this.cardHolder = findViewById(R.id.aniways_contextual_card_holder);

        this.cardMinHeight = (int)getResources().getDimension(R.dimen.aniways_contextual_card_min_height);
        this.hintHeight = (int)res.getDimension(R.dimen.aniways_contextual_card_hint_height);
        this.winkWidth = (int)res.getDimension(R.dimen.aniways_contextual_card_wink_width);
        this.cardIconHolderWidth = (int)res.getDimension(R.dimen.aniways_contextual_card_icon_holder_width);
        this.cardIconHolderOverlapWidth = this.cardIconHolderWidth - (int)res.getDimension(R.dimen.aniways_contextual_card_reveal_width);
    }

    public void initContent(PhraseContextualContentViewProvider contentProvider, int pagerHeight, int index, int pageSpacerWidth, AniwaysSlidingLayer.OnInteractListener onInteractListener)
    {
        content = contentProvider.getView();
        View innerContentView = (View) content;

        setInnerContentLayoutParams(innerContentView);

        this.innerContentHolder.addView(innerContentView);
        this.slidingLayer.setOnInteractListener(onInteractListener);
        this.index = index;

        measureView();
        setCardVisibilityByIndex(index);

        int cardWidth = innerContentHolder.getMeasuredWidth() - pageSpacerWidth;
        content.initialize(cardWidth);

        LinearLayout.LayoutParams lp = (LinearLayout.LayoutParams)this.slidingLayer.getLayoutParams();
        cardHolder.measure(0, 0);
        int cardHeight = cardHolder.getMeasuredHeight();

        lp.topMargin = pagerHeight - cardHeight;
        this.slidingLayer.setLayoutParams(lp);

        //TODO: This is a hack since on some devices prior to android L the card height is 2 pixels higher than the defined 70dp - so we need to invest time to figure out why.
        this.isContentOverSized = cardHeight > this.cardMinHeight + AniwaysPrivateConfig.getInstance().contextualCardHeightThreshold;

        AssetType assetType = contentProvider.getAssetType();

        switch (assetType)
        {
            case Emoticons:
                this.cardIcon.setBackgroundResource(R.drawable.aniways_contextual_asset_emoticons);
                break;
            case AnimatedGif:
                this.cardIcon.setBackgroundResource(R.drawable.aniways_contextual_asset_gifs);
                break;
            case Place:
                this.cardIcon.setBackgroundResource(R.drawable.aniways_contextual_asset_locations);
                break;
        }
    }

    private void setInnerContentLayoutParams(View innerContentView)
    {
        RelativeLayout.LayoutParams overviewParams =
                new RelativeLayout.LayoutParams(RelativeLayout.LayoutParams.WRAP_CONTENT,
                                                RelativeLayout.LayoutParams.WRAP_CONTENT
                );

        overviewParams.addRule(RelativeLayout.CENTER_IN_PARENT);

        innerContentView.setLayoutParams(overviewParams);
    }

    private void setCardVisibilityByIndex(int index)
    {
        if (index == 0)
        {
            // first card not showing icon so removing the space
            cardIconContainer.setVisibility(GONE);
        }
        else
        {
            cardIconContainer.setVisibility(VISIBLE);
        }
    }

    private void measureView()
    {
        int screenWidth = AniwaysUiUtil.getScreenDisplayMatrix().widthPixels;

        this.measure(MeasureSpec.makeMeasureSpec(screenWidth, MeasureSpec.EXACTLY), 0);
    }

    /**
     * Updates the visuals of this instance to match its horizontal scroll position.
     *
     * @param position The view's horizontal position along a scrolling view.
     *                 (-Infinity, -1] means the view is not visible.
     *                 (-1, 0) means the view's peeking in from the left, partially visible.
     *                 [0, 0] means the view is completely visible.
     *                 (0, Infinity) means the view's leftmost content is partially covered.
     */
    public void updateByPosition(float position)
    {
        updateCardBackground(position);
        updateCardIcon(position);
    }

    public void wink()
    {
        ObjectAnimator winkAnimation = ObjectAnimator.ofFloat(cardHolder, "translationX", 0, -winkWidth);
        ObjectAnimator holdAnimation = ObjectAnimator.ofFloat(cardHolder, "translationX", -winkWidth, -winkWidth);
        ObjectAnimator returnAnimation = ObjectAnimator.ofFloat(cardHolder, "translationX", -winkWidth, 0);

        winkAnimation.setInterpolator(new OvershootInterpolator(3));
        holdAnimation.setDuration(650);
        returnAnimation.setInterpolator(new OvershootInterpolator(3));

        AnimatorSet story = new AnimatorSet();
        story.play(winkAnimation).before(holdAnimation);
        story.play(holdAnimation).before(returnAnimation);
        story.start();
    }

    public void showHint()
    {
        if (this.isContentOverSized)
        {
            postDelayed(new Runnable()
            {
                @Override
                public void run()
                {
                    ContextualAssetView.this.slidingLayer.smoothScrollBy(0, ContextualAssetView.this.hintHeight);
                }
            }, 0);
        }
    }

    public void hideHint()
    {
        if (this.isContentOverSized)
        {
            this.slidingLayer.closeLayer(true, true);
        }
    }

    private void updateCardBackground(float position)
    {
        int cardColor = getShadedColorByPosition(position, cardBaseColor);
        this.cardBackgroundView.setBackgroundColor(cardColor);
        this.cardIconContainer.setBackgroundColor(cardColor);
    }

    private void updateCardIcon(float position)
    {
        // Determines the position at which the icon becomes invisible, gradually.
        float iconVisibilityCutOff = 1.0f - (float)this.cardIconHolderOverlapWidth / getWidth();

        // Determines the position at which the icon holder becomes invisible, gradually.
        float iconHolderVisibilityCutOff = iconVisibilityCutOff - (float)this.cardIconHolderWidth / getWidth();

        float iconHolderMotionRange = iconVisibilityCutOff - iconHolderVisibilityCutOff;

        float iconOpacity = IconMaxAlpha;
        int iconHolderWidth = this.cardIconHolderWidth;

        if (position < iconHolderVisibilityCutOff)
        {
            iconOpacity = 0.0f;
            iconHolderWidth = 0;
        }
        else if (position < iconVisibilityCutOff)
        {
            iconOpacity = 0.0f;
            iconHolderWidth = (int)(this.cardIconHolderWidth * (1.0f - (iconVisibilityCutOff - position) / iconHolderMotionRange));
        }
        else if (position <= 1.0f)
        {
            // Gradually disappear until reaching the cut-off position.
            iconOpacity = IconMaxAlpha * (1.0f - (1.0f - position) / (1.0f - iconVisibilityCutOff));
        }

        ViewGroup.LayoutParams lp = this.cardIconContainer.getLayoutParams();
        lp.width = iconHolderWidth;
        this.cardIconContainer.setLayoutParams(lp);

        ViewHelper.setAlpha(this.cardIcon, iconOpacity);
    }

    private static int getShadedColorByPosition(float position, int baseColor)
    {
        final float LightCutOff = 0.7f;
        final float BrightnessStepSize = 0.12f;

        int retColor;
        if (position <= 0)
        {
            retColor = baseColor;
        }
        else
        {
            int stepsDown = (int)position;
            float remainder = position % 1;
            if (remainder > LightCutOff) { remainder = LightCutOff; }

            float ratio = 1.0f - (stepsDown * BrightnessStepSize) - (remainder / LightCutOff) * BrightnessStepSize;

            retColor = changeColorBrightness(baseColor, ratio);
        }

        return retColor;
    }

    private static int changeColorBrightness(int color, float brightnessRatio)
    {
        float[] hsv = new float[3];
        Color.colorToHSV(color, hsv);
        hsv[2] *= brightnessRatio;

        return Color.HSVToColor(hsv);
    }

    public void notifyNotSelected() {
        if(content != null){
            content.notifyNotSelected();
        }
    }

    public void notifySelected() {
        if(content != null){
            content.notifySelected();
        }
    }

    public void notifyRevealMore() {
        if(content != null){
            content.notifyRevealMore();
        }
    }

    public void notifyRevealLess() {
        if(content != null){
            content.notifyRevealLess();
        }
    }
}