package com.aniways.quick.action;

import com.aniways.Log;

import android.app.Activity;
import android.graphics.Color;
import android.graphics.drawable.ColorDrawable;
import android.view.MotionEvent;
import android.view.View;
import android.view.WindowManager;
import android.view.View.OnTouchListener;
import android.widget.PopupWindow;
import android.content.Context;

import java.util.Timer;
import java.util.TimerTask;

/**
 * Custom popup window.
 * 
 * @author Lorensius W. L. T <lorenz@londatiga.net>
 *
 */
public class PopupWindows {
	protected Context mContext;
	protected PopupWindow mWindow;
	protected View mRootView;
	protected WindowManager mWindowManager;
    protected boolean mDismissed;
    private boolean mDelayAutoDismiss;
    private Timer mDismissalDelayTimer;
	private static final String TAG = "AniwaysPopupWindow";

	/**
	 * Constructor.
	 * 
	 * @param context Context
	 */
	public PopupWindows(Context context, boolean delayAutoDismiss) {
		mContext	= context;
		mWindow 	= new PopupWindow(context);
        mDelayAutoDismiss = delayAutoDismiss;
		mWindowManager = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
	}

	/**
	 * On pre show
	 */
	protected void preShow(boolean fillParentWidth) {
		if (mRootView == null) 
			throw new IllegalStateException("setContentView was not called with a view to display.");

		mWindow.setBackgroundDrawable(new ColorDrawable(Color.TRANSPARENT));

		mWindow.setWidth(fillParentWidth ? WindowManager.LayoutParams.MATCH_PARENT: WindowManager.LayoutParams.WRAP_CONTENT);
		mWindow.setHeight(WindowManager.LayoutParams.WRAP_CONTENT);
		mWindow.setTouchable(true);
		// This makes sure that the popup will disappear when continue to edit
		mWindow.setFocusable(false);
		mWindow.setInputMethodMode(PopupWindow.INPUT_METHOD_NOT_NEEDED);
		mWindow.setOutsideTouchable(true);

        mWindow.setTouchInterceptor(new OnTouchListener()
        {
            @Override
            public boolean onTouch(View v, MotionEvent event)
            {
                return processPopupTouchEvent(v, event);
            }
        });

		mWindow.setContentView(mRootView);
	}

    public void cancelDismissal()
    {
        Timer activeTimer = mDismissalDelayTimer;

        if (activeTimer != null)
        {
            activeTimer.cancel();
        }
    }

	/**
	 * Set content view.
	 * 
	 * @param root Root view
	 */
	public void setContentView(View root) {
		mRootView = root;

		mWindow.setContentView(root);
	}

	/**
	 * Set listener on window dismissed.
	 * 
	 * @param listener The OnDismiss callback listener.
	 */
	public void setOnDismissListener(PopupWindow.OnDismissListener listener) {
		mWindow.setOnDismissListener(listener);  
	}

	/**
	 * Dismiss the popup window.
	 */
	public void dismiss() {
		// Because view detached exception might be thrown
		try{

            mDismissalDelayTimer = null;
			mWindow.dismiss();
            mDismissed = true;
		}
		catch (Exception e){
			Log.w(true, TAG, "caught Exception while dismissing popup", e);
		}

	}

	public Boolean isShowing()
	{
		return mWindow != null && mWindow.isShowing();
	}

    private class DismissalDelayTask extends TimerTask
    {
        private final PopupWindows mPopupToDismiss;

        public DismissalDelayTask(PopupWindows popupToDismiss)
        {
            mPopupToDismiss = popupToDismiss;
        }

        @Override
        public void run()
        {
            ((Activity)mPopupToDismiss.mContext).runOnUiThread(new Runnable()
            {
                @Override
                public void run()
                {
                    mPopupToDismiss.dismiss();
                }
            });
        }
    }

    protected boolean processPopupTouchEvent(View v, MotionEvent event)
    {
        try
        {
            if (event.getAction() == MotionEvent.ACTION_OUTSIDE)
            {
                if (mDelayAutoDismiss)
                {
                    mDismissalDelayTimer = new Timer();
                    mDismissalDelayTimer.schedule(new DismissalDelayTask(PopupWindows.this), 400);
                }
                else
                {
                    dismiss();
                }

                return true;
            }
        }
        catch (Throwable ex)
        {
            Log.e(true, TAG, "Caught Exception in onTouch", ex);
        }

        return false;
    }
}