package com.aniways.emoticons.button;

import android.content.Context;
import android.content.SharedPreferences;
import android.content.SharedPreferences.Editor;

import com.aniways.AssetType;
import com.aniways.Log;
import com.aniways.Utils;
import com.aniways.data.AniwaysPrivateConfig;
import com.aniways.service.utils.AniwaysServiceUtils;

import java.util.Arrays;
import java.util.Dictionary;
import java.util.Hashtable;
import java.util.LinkedHashSet;
import java.util.Stack;

public class AniwaysRecentIconsManager
{
    // V1 was when there were no numerical ids
    private static final String KEY_RECENT_ICONS_OLD = "com.aniways.RECENT_ICONS_V2";
    private static final String KEY_RECENT_EMOJI = "com.aniways.RECENT_EMOJI";
    private static final String KEY_RECENT_EMOTICONS = "com.aniways.RECENT_EMOTICONS";
    private static final String IDS_DIVIDER = ";;";
    private static final String TAG = "AniwaysRecentIconsManager";

    // The recently used icons in the app
    private static Dictionary<AssetType, Stack<String>> sRecentIconsByAssetType;

    private static Context sContext;

    public static void forceInit(Context context)
    {
        if (sContext != null)
        {
            Log.w(true, TAG, "Calling init of AniwaysRecentIconsManager more than once");
        }

        sContext = context;
        populateRecentIds();
    }

    public static void addRecentIcon(long iconId, AssetType assetType)
    {
        String idString = Long.toString(iconId);

        Stack<String> recentIcons = sRecentIconsByAssetType.get(assetType);
        // Remove the icon if it is already in the stack and push it to the top
        recentIcons.remove(idString);
        recentIcons.push(idString);

        // Remove excess icon to be in the max size.
        // Do not assume that need to remove maximum of one because a change in config might force us to remove more
        while (recentIcons.size() > AniwaysPrivateConfig.getInstance().maxIconsInRecentsTab)
        {
            recentIcons.remove(0);
        }

        LinkedHashSet<String> set = new LinkedHashSet<>();
        for (int i = 0; i < recentIcons.size(); i++)
        {
            set.add(recentIcons.get(i));
        }

        // Update the shared preferences
        SharedPreferences prefs = sContext.getSharedPreferences(AniwaysServiceUtils.SHARED_PREFERENCES, Utils.getSharedPreferencesFlags());

        String recentIds = convertSetToIdsString(set);

        Editor edit = prefs.edit();

        if (assetType == AssetType.Emoji)
        {
            edit.putString(KEY_RECENT_EMOJI, recentIds);
        }
        else if (assetType == AssetType.Emoticons)
        {
            edit.putString(KEY_RECENT_EMOTICONS, recentIds);
        }

        edit.commit();
    }

    static boolean isEmpty(AssetType assetType)
    {
        Stack<String> idsStack = sRecentIconsByAssetType.get(assetType);
        return idsStack == null || idsStack.isEmpty();
    }

    /**
     * Most recent is first.
     * Returns an array with length == 0 if empty, never returns null!!
     *
     */
    static String[] getRecentIcons(AssetType assetType)
    {
        Stack<String> idsStack = sRecentIconsByAssetType.get(assetType);

        if (idsStack != null && idsStack.size() > 0)
        {
            String[] result = new String[idsStack.size()];

            // Because the sRecentIcons is a stack, we take from the end to the start
            int lastIndex = result.length - 1;
            for (int i = lastIndex; i > -1; i--)
            {
                result[lastIndex - i] = idsStack.get(i);
            }

            return result;
        }

        return new String[0];
    }

    private static void populateRecentIds()
    {
        // open the shared preferences
        SharedPreferences prefs = sContext.getSharedPreferences(AniwaysServiceUtils.SHARED_PREFERENCES, Utils.getSharedPreferencesFlags());

        sRecentIconsByAssetType = new Hashtable<>();
        Stack<String> emojiIds = new Stack<>();
        Stack<String> emoticonIds = new Stack<>();
        sRecentIconsByAssetType.put(AssetType.Emoji, emojiIds);
        sRecentIconsByAssetType.put(AssetType.Emoticons, emoticonIds);

        // TODO yoavsion: Temp code - for this version (and this version only), we convert the old recent ids string to the new format.
        if (prefs.contains(KEY_RECENT_ICONS_OLD))
        {
            Log.i(TAG, "Shared preferences contains the old recent icons key. Converting to new format.");

            // !! Not using the getStringSet() function because it does not maintain order
            String recentIdsString = prefs.getString(KEY_RECENT_ICONS_OLD, "");

            if (recentIdsString.length() > 0)
            {
                LinkedHashSet<String> recentIds = convertIdsStringToSet(recentIdsString);

                for (String recentIdStr : recentIds)
                {
                    try
                    {
                        int recentId = Integer.parseInt(recentIdStr);

                        // Hack. Only valid for this version and will be removed in the release following this version.
                        if (recentId >= 1000 && recentId <= 1845)
                        {
                            emojiIds.push(recentIdStr);
                        }
                        else
                        {
                            emoticonIds.push(recentIdStr);
                        }
                    }
                    catch (NumberFormatException e)
                    {
                        // Not a number - definitely not an Emoji, so add to the Emoticons stack.
                        emoticonIds.push(recentIdStr);
                    }
                }
            }

            Editor editor = prefs.edit();

            LinkedHashSet<String> set = new LinkedHashSet<>();
            set.addAll(emojiIds);

            String recentIds = convertSetToIdsString(set);
            editor.putString(KEY_RECENT_EMOJI, recentIds);

            set.clear();
            set.addAll(emoticonIds);
            recentIds = convertSetToIdsString(set);
            editor.putString(KEY_RECENT_EMOTICONS, recentIds);

            editor.remove(KEY_RECENT_ICONS_OLD);

            editor.commit();
        }
        else
        {
            String recentEmojiIdsString = prefs.getString(KEY_RECENT_EMOJI, "");

            if (recentEmojiIdsString.length() > 0)
            {
                LinkedHashSet<String> recentEmojiIds = convertIdsStringToSet(recentEmojiIdsString);
                emojiIds.addAll(recentEmojiIds);
            }
            else
            {
                Log.i(TAG, "Shared preferences do not contain a value for recent emojis, the recent emoji this tab will be empty");
            }

            String recentEmoticonsIdsString = prefs.getString(KEY_RECENT_EMOTICONS, "");

            if (recentEmoticonsIdsString.length() > 0)
            {
                LinkedHashSet<String> recentEmoticonIds = convertIdsStringToSet(recentEmoticonsIdsString);
                emoticonIds.addAll(recentEmoticonIds);
            }
            else
            {
                Log.i(TAG, "Shared preferences do not contain a value for recent emoticons, the recent emoticons this tab will be empty");
            }
        }
    }

    private static LinkedHashSet<String> convertIdsStringToSet(String idsString)
    {
        String[] ids = idsString.split(IDS_DIVIDER);
        LinkedHashSet<String> result = new LinkedHashSet<>();

        if (ids.length > 0)
        {
            result.addAll(Arrays.asList(ids));
        }
        else
        {
            Log.w(false, TAG, "No ids to extract from ids string:" + idsString);
        }

        return result;
    }

    private static String convertSetToIdsString(LinkedHashSet<String> ids)
    {
        StringBuilder sb = new StringBuilder("");
        if (ids != null && !ids.isEmpty())
        {
            for (String id : ids)
            {
                sb.append(id);
                sb.append(IDS_DIVIDER);
            }

            // Remove last divider
            sb.delete(sb.length() - IDS_DIVIDER.length(), sb.length());
        }
        else
        {
            Log.w(false, TAG, "No ids to convert to string:" + ids);
        }

        return sb.toString();
    }
}