package com.aniways.contextual;

import android.content.Context;
import android.content.SharedPreferences;
import android.content.res.Resources;
import android.support.v4.view.PagerAdapter;
import android.support.v4.view.ViewPager;
import android.support.v7.widget.CardView;
import android.view.Gravity;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;
import android.widget.Toast;

import com.aniways.AssetType;
import com.aniways.R;
import com.aniways.Utils;
import com.aniways.analytics.AnalyticsReporter;
import com.aniways.analytics.GoogleAnalyticsReporter;
import com.aniways.data.AniwaysConfiguration;
import com.aniways.data.AniwaysStatics;
import com.aniways.data.SettingsKeys;
import com.aniways.quick.action.ContextualPopupCreationContext;
import com.aniways.quick.action.PhraseContextualContentViewProvider;
import com.aniways.quick.action.SuggestionPopupModelBuilder;
import com.aniways.service.utils.AniwaysServiceUtils;
import com.aniways.ui.views.AniwaysSlidingLayer;
import com.aniways.ui.views.ContextualAssetView;
import com.aniways.ui.views.ContextualSettingsView;

import java.util.List;

/**
 * Initializes and updates the contextual pages for the contextual content view pager.
 */
public class ContextualPopupAdapter extends PagerAdapter implements ViewPager.OnPageChangeListener
{
    // The inflater service.
    private final LayoutInflater inflater;

    // Holds the instantiated asset views.
    private final ContextualAssetView[] assetViews;

    // Holds the view pager that this adapter "serves".
    private final ViewPager pager;

    private final ContextualPopupCreationContext creationContext;

    // Holds the elevation, in pixels, that should be applied between each of the asset cards.
    private final float cardElevationInPixels;

    // Holds the width, in pixels, that should be set to the right of each asset view, in order
    // to reveal the card under it.
    private final float cardRevealWidthInPixels;

    // Holds the margin to be applied to the right of each card, to allow the shadow drop.
    private final float cardRightMarginInPixels;

    private final List<PhraseContextualContentViewProvider> contextualTabs;

    // Holds the corner radius defined for the contextual cards.
    private final float cardCornerRadius;

    // Holds the toast height.
    private final int toastHeight;

    // Holds the paging toast view.
    private final Toast pagingToast;

    // Holds the paging toast's icon ImageView.
    private final ImageView pagingToastImageView;

    // Holds the paging toast's TextView.
    private final TextView pagingToastTextView;

    // Holds the index of the page currently being shown.
    private int currentPageIndex = 0;

    AniwaysSlidingLayer.OnInteractListener slidingLayerInteractListener;

    /**
     * Initializes a new instance of the ContextualPopupAdapter.
     *
     * @param context  The visual context.
     * @param inflater The inflater service to be used when creating asset views.
     * @param pager    The pager that this adapter serves.
     */
    public ContextualPopupAdapter(Context context, LayoutInflater inflater, ViewPager pager, final ContextualPopupCreationContext creationContext)
    {
        this.inflater = inflater;
        this.pager = pager;
        this.creationContext = creationContext;
        this.contextualTabs = SuggestionPopupModelBuilder.getInstance().build(creationContext, context).getTabs();

        Resources res = context.getResources();
        this.cardElevationInPixels = res.getDimension(R.dimen.aniways_contextual_card_elevation);
        this.cardRevealWidthInPixels = res.getDimension(R.dimen.aniways_contextual_card_reveal_width);
        this.cardRightMarginInPixels = res.getDimension(R.dimen.aniways_contextual_card_right_margin_width);
        this.cardCornerRadius = res.getDimension(R.dimen.cardview_default_radius);
        this.toastHeight = (int)res.getDimension(R.dimen.aniways_asset_toast_height);

        View toastLayout = inflater.inflate(R.layout.aniways_asset_toast, null);
        this.pagingToast = new Toast(context);
        this.pagingToast.setView(toastLayout);
        this.pagingToastImageView = (ImageView)toastLayout.findViewById(R.id.aniways_asset_toast_icon);
        this.pagingToastTextView = (TextView)toastLayout.findViewById(R.id.aniways_asset_toast_text);

        this.assetViews = new ContextualAssetView[this.contextualTabs.size()];

        initSlidingLayerInteractionHandler(creationContext);
    }

    private void initSlidingLayerInteractionHandler(final ContextualPopupCreationContext creationContext) {
        this.slidingLayerInteractListener = new AniwaysSlidingLayer.OnInteractListener()
        {
            @Override
            public void onOpen()
            {
                pagingToast.cancel();
                int position = currentPageIndex;
                AssetType assetType;

                if (position == contextualTabs.size())
                {
                    assetType = AssetType.Settings;
                }
                else
                {
                    assetType = contextualTabs.get(position).getAssetType();
                    assetViews[position].notifyRevealMore();
                }


                GoogleAnalyticsReporter.reportEvent(AniwaysConfiguration.Verbosity.Info, "Popup Reveal More-" + creationContext.selectionOrigin, assetType.toString() + "-" + position, creationContext.phrase.getPhraseSubphraseString(), 0);
                AnalyticsReporter.reportRevealMoreInPopup(assetType, position, creationContext);
            }

            @Override
            public void onClose()
            {
                int position = currentPageIndex;

                if (position != contextualTabs.size())
                {
                    assetViews[position].notifyRevealLess();
                }
            }

            @Override
            public void onOpened()
            {

            }

            @Override
            public void onClosed()
            {

            }
        };
    }

    @Override
    public Object instantiateItem(ViewGroup container, int position)
    {
        if (position < assetViews.length)
        {
            int distFromBottom = getCount() - (position + 1);

            ContextualAssetView assetView = (ContextualAssetView)this.inflater.inflate(R.layout.aniways_contextual_card, null);
            container.addView(assetView);

            assetViews[position] = assetView;

            CardView cardView = (CardView)assetView.findViewById(R.id.aniways_contextual_card_view);
            cardView.setCardElevation(distFromBottom * this.cardElevationInPixels);

            int cardSidePadding = 0;

            if (!Utils.isAndroidVersionAtLeast(21))
            {
                // Compensate for the padding pre-L versions impose on the card, for shadow drawing.
                float maxCardElevation = cardView.getMaxCardElevation();
                cardSidePadding = (int)(maxCardElevation + (1 - Math.cos(45)) * this.cardCornerRadius);
                int cardTopBottomPadding = (int)(maxCardElevation * 1.5 + (1 - Math.cos(45)) * this.cardCornerRadius);

                LinearLayout.LayoutParams cardLayoutParams = (LinearLayout.LayoutParams)cardView.getLayoutParams();
                cardLayoutParams.setMargins(-cardSidePadding, -cardTopBottomPadding, 0, -cardTopBottomPadding);
            }

            View rightSpacer = assetView.findViewById(R.id.aniways_contextual_card_right_spacer);
            ViewGroup.LayoutParams spacerLayoutParams = rightSpacer.getLayoutParams();
            int spacerWidth = getSpacerWidth(distFromBottom);
            spacerLayoutParams.width = spacerWidth + cardSidePadding;
            rightSpacer.setLayoutParams(spacerLayoutParams);

            assetView.initContent(this.contextualTabs.get(position), this.pager.getMeasuredHeight(), position, spacerWidth, slidingLayerInteractListener);

            if (position == 0)
            {
                showFirstPageHintOrWink();
                setFirstPageAsSelected();
            }

            return assetView;
        }
        else
        {
            ContextualSettingsView settingsView = (ContextualSettingsView)this.inflater.inflate(R.layout.aniways_contextual_settings_card, null);
            container.addView(settingsView);
            settingsView.initContent(this.pager.getMeasuredHeight());

            if (!Utils.isAndroidVersionAtLeast(21))
            {
                CardView cardView = (CardView)settingsView.findViewById(R.id.aniways_contextual_settings_card_view);

                // Compensate for the padding pre-L versions impose on the card, for shadow drawing.
                float maxCardElevation = cardView.getMaxCardElevation();
                int cardSidePadding = (int)(maxCardElevation + (1 - Math.cos(45)) * this.cardCornerRadius);
                int cardTopBottomPadding = (int)(maxCardElevation * 1.5 + (1 - Math.cos(45)) * this.cardCornerRadius);

                LinearLayout.LayoutParams cardLayoutParams = (LinearLayout.LayoutParams)cardView.getLayoutParams();
                cardLayoutParams.setMargins(-cardSidePadding, -cardTopBottomPadding, 0, -cardTopBottomPadding);
            }

            return settingsView;
        }
    }

    public void setToastOffset(int yOffset)
    {
        this.pagingToast.setGravity(Gravity.TOP, 0, yOffset - this.toastHeight);
    }

    private int getSpacerWidth(int distFromBottom)
    {
        return (int)(this.cardRevealWidthInPixels * distFromBottom - this.cardRightMarginInPixels);
    }

    @Override
    public void destroyItem(ViewGroup container, int position, Object object)
    {
        container.removeView((View)object);

        if (position < this.assetViews.length)
        {
            assetViews[position] = null;
        }
    }

    @Override
    public int getCount()
    {
        return this.contextualTabs.size() + 1;
    }

    @Override
    public boolean isViewFromObject(View view, Object object)
    {
        return view.equals(object);
    }

    /**
     * This method will be invoked when the current page is scrolled, either as part
     * of a programatically initiated smooth scroll or a user initiated touch scroll.
     *
     * @param position             Position index of the first page currently being displayed.
     *                             Page position+1 will be visible if positionOffset is nonzero.
     * @param positionOffset       Value from [0, 1) indicating the offset from the page at position.
     * @param positionOffsetPixels Value in pixels indicating the offset from position.
     */
    @Override
    public void onPageScrolled(int position, float positionOffset, int positionOffsetPixels)
    {

    }

    /**
     * This method will be invoked when a new page becomes selected. Animation is not
     * necessarily complete.
     *
     * @param position Position index of the new selected page.
     */
    @Override
    public void onPageSelected(int position)
    {
        int oldPos = this.currentPageIndex;
        this.currentPageIndex = position;

        AssetType newAssetType;
        if (position == this.contextualTabs.size())
        {
            newAssetType = AssetType.Settings;
        }
        else
        {
            newAssetType = this.contextualTabs.get(position).getAssetType();
            assetViews[position].notifySelected();
        }

        AssetType oldAssetType;
        if (oldPos == this.contextualTabs.size())
        {
            oldAssetType = AssetType.Settings;
        }
        else
        {
            oldAssetType = this.contextualTabs.get(oldPos).getAssetType();
            assetViews[oldPos].notifyNotSelected();
        }

        GoogleAnalyticsReporter.reportEvent(AniwaysConfiguration.Verbosity.Info, "Popup Scroll-" + creationContext.selectionOrigin, "To:" + newAssetType.toString() + "-" + position + ". From: " + oldAssetType.toString() + "-" + oldPos, this.creationContext.phrase.getPhraseSubphraseString(), 0);
        AnalyticsReporter.reportPopupScroll(newAssetType, position, oldAssetType, oldPos, this.creationContext);

        switch (newAssetType)
        {
            case Emoticons:
                this.pagingToastImageView.setImageResource(R.drawable.aniways_contextual_asset_emoticons_white);
                this.pagingToastTextView.setText(R.string.aniways_asset_type_emoticons);
                this.pagingToast.show();
                break;
            case AnimatedGif:
                this.pagingToastImageView.setImageResource(R.drawable.aniways_contextual_asset_gifs_white);
                this.pagingToastTextView.setText(R.string.aniways_asset_type_gifs);
                this.pagingToast.show();
                break;
            case Settings:
                this.pagingToast.cancel();
        }

        SharedPreferences prefs = AniwaysStatics.getApplicationContext().getSharedPreferences(AniwaysServiceUtils.SHARED_PREFERENCES, Utils.getSharedPreferencesFlags());

        if (!prefs.contains(SettingsKeys.KEY_CONTEXTUAL_POPUP_CARD_SWIPED))
        {
            prefs.edit().putString(SettingsKeys.KEY_CONTEXTUAL_POPUP_CARD_SWIPED, "1").commit();
        }
    }

    /**
     * Called when the scroll state changes. Useful for discovering when the user
     * begins dragging, when the pager is automatically settling to the current page,
     * or when it is fully stopped/idle.
     *
     * @param state The new scroll state.
     * @see android.support.v4.view.ViewPager#SCROLL_STATE_IDLE
     * @see android.support.v4.view.ViewPager#SCROLL_STATE_DRAGGING
     * @see android.support.v4.view.ViewPager#SCROLL_STATE_SETTLING
     */
    @Override
    public void onPageScrollStateChanged(int state)
    {
        // Settings page.
        if (this.currentPageIndex == this.assetViews.length) { return; }

        switch (state)
        {
            case ViewPager.SCROLL_STATE_DRAGGING:
                this.assetViews[this.currentPageIndex].hideHint();
                break;
            case ViewPager.SCROLL_STATE_IDLE:
                this.assetViews[this.currentPageIndex].showHint();
                break;
        }
    }

    private void setFirstPageAsSelected(){
        this.assetViews[0].notifySelected();
    }

    private void showFirstPageHintOrWink()
    {
        SharedPreferences prefs = AniwaysStatics.getApplicationContext().getSharedPreferences(AniwaysServiceUtils.SHARED_PREFERENCES, Utils.getSharedPreferencesFlags());

        if (!prefs.contains(SettingsKeys.KEY_CONTEXTUAL_POPUP_CARD_SWIPED))
        {
            pager.postDelayed(new Runnable()
            {
                @Override
                public void run()
                {
                    if (ContextualPopupAdapter.this.assetViews[0] != null)
                    {
                        ContextualPopupAdapter.this.assetViews[0].wink();
                    }

                }
            }, 800);
        }
        else
        {
            pager.postDelayed(new Runnable()
            {
                @Override
                public void run()
                {
                    if (ContextualPopupAdapter.this.assetViews[0] != null)
                    {
                        ContextualPopupAdapter.this.assetViews[0].showHint();
                    }

                }
            }, 800);
        }
    }
}